<?php
/**
 * משיכת שערי מט"ח מ-API רשמי של בנק ישראל
 * מחזיר שער יציג + מרווח 0.86% (כמו מזרחי-טפחות)
 */

header('Content-Type: application/json; charset=utf-8');

// מרווח ברירת מחדל - מחושב לפי שער המחאות מכירה של מזרחי-טפחות
define('DEFAULT_MARGIN_PERCENT', 0.86);

function fetchBankOfIsraelRates() {
    // API רשמי של בנק ישראל - מחזיר JSON
    $url = 'https://boi.org.il/PublicApi/GetExchangeRates';
    
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_HTTPHEADER => [
            'Accept: application/json'
        ]
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($httpCode !== 200 || !$response) {
        return [
            'success' => false,
            'error' => 'שגיאה בחיבור ל-API של בנק ישראל: ' . ($error ?: "HTTP $httpCode")
        ];
    }
    
    // פענוח JSON
    $data = json_decode($response, true);
    
    if (!$data || !isset($data['exchangeRates'])) {
        return [
            'success' => false,
            'error' => 'תשובה לא תקינה מ-API',
            'raw' => substr($response, 0, 500)
        ];
    }
    
    // חיפוש USD ו-EUR
    $rates = [];
    foreach ($data['exchangeRates'] as $rate) {
        if ($rate['key'] === 'USD') {
            $rates['USD'] = (float)$rate['currentExchangeRate'];
        }
        if ($rate['key'] === 'EUR') {
            $rates['EUR'] = (float)$rate['currentExchangeRate'];
        }
    }
    
    if (empty($rates['USD']) || empty($rates['EUR'])) {
        return [
            'success' => false,
            'error' => 'לא נמצאו שערי USD/EUR בתשובה'
        ];
    }
    
    return [
        'success' => true,
        'rates' => $rates,
        'lastUpdate' => $data['exchangeRates'][0]['lastUpdate'] ?? null
    ];
}

// קבלת אחוז מרווח מפרמטר או ברירת מחדל
$marginPercent = isset($_GET['margin']) ? (float)$_GET['margin'] : DEFAULT_MARGIN_PERCENT;

// משיכת שערים
$result = fetchBankOfIsraelRates();

if ($result['success']) {
    // חישוב שער מכירה (יציג + מרווח)
    $usdRepresentative = $result['rates']['USD'];
    $eurRepresentative = $result['rates']['EUR'];
    
    $usdSelling = round($usdRepresentative * (1 + $marginPercent / 100), 4);
    $eurSelling = round($eurRepresentative * (1 + $marginPercent / 100), 4);
    
    $result = [
        'success' => true,
        'source' => 'בנק ישראל (API רשמי)',
        'type' => 'שער יציג + מרווח ' . $marginPercent . '%',
        'fetched_at' => date('Y-m-d H:i:s'),
        'margin_percent' => $marginPercent,
        'rates' => [
            'USD' => $usdSelling,
            'EUR' => $eurSelling
        ],
        'representative_rates' => [
            'USD' => $usdRepresentative,
            'EUR' => $eurRepresentative
        ]
    ];
}

// אם נשלח פרמטר save=1, שומר ל-DB
if (isset($_GET['save']) && $_GET['save'] == '1' && $result['success']) {
    require_once __DIR__ . '/../includes/auth.php';
    
    try {
        $pdo = getDB();
        
        // וידוא עמודת source
        try {
            $stmt = $pdo->query("SHOW COLUMNS FROM exchange_rates LIKE 'source'");
            if ($stmt->rowCount() == 0) {
                $pdo->exec("ALTER TABLE exchange_rates ADD COLUMN source VARCHAR(100) DEFAULT 'ידני'");
            }
        } catch (Exception $e) {}
        
        $agentId = $_SESSION['agent_id'] ?? 0;
        $source = 'בנק ישראל + ' . $marginPercent . '% (אוטומטי)';
        
        try {
            $stmt = $pdo->prepare("INSERT INTO exchange_rates (currency, rate, updated_at, updated_by, source) 
                                   VALUES (?, ?, NOW(), ?, ?) 
                                   ON DUPLICATE KEY UPDATE rate = VALUES(rate), updated_at = NOW(), updated_by = VALUES(updated_by), source = VALUES(source)");
            $stmt->execute(['USD', $result['rates']['USD'], $agentId, $source]);
            $stmt->execute(['EUR', $result['rates']['EUR'], $agentId, $source]);
        } catch (Exception $e) {
            // ללא source
            $stmt = $pdo->prepare("INSERT INTO exchange_rates (currency, rate, updated_at, updated_by) 
                                   VALUES (?, ?, NOW(), ?) 
                                   ON DUPLICATE KEY UPDATE rate = VALUES(rate), updated_at = NOW(), updated_by = VALUES(updated_by)");
            $stmt->execute(['USD', $result['rates']['USD'], $agentId]);
            $stmt->execute(['EUR', $result['rates']['EUR'], $agentId]);
        }
        
        $result['saved'] = true;
        $result['message'] = 'השערים נשמרו בהצלחה!';
    } catch (Exception $e) {
        $result['saved'] = false;
        $result['save_error'] = $e->getMessage();
    }
}

echo json_encode($result, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
