<?php
/**
 * API לניהול החזרים - order_refunds
 * GET: שליפת החזרים להזמנה
 * POST: הוספת החזר
 * DELETE: מחיקת החזר
 */

require_once '../includes/auth.php';
requireLogin();

header('Content-Type: application/json; charset=utf-8');

$pdo = getDB();
$agent = getCurrentAgent();

try {
    $method = $_SERVER['REQUEST_METHOD'];
    
    // GET - שליפת החזרים להזמנה
    if ($method === 'GET') {
        $orderId = $_GET['order_id'] ?? null;
        
        if (!$orderId) {
            throw new Exception('חסר מזהה הזמנה');
        }
        
        $stmt = $pdo->prepare("
            SELECT r.*, a.full_name as created_by_name
            FROM order_refunds r
            LEFT JOIN agents a ON r.created_by = a.id
            WHERE r.order_id = ?
            ORDER BY r.refund_date DESC, r.created_at DESC
        ");
        $stmt->execute([$orderId]);
        $refunds = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // חישוב סיכומים
        $totalReceived = 0;
        $totalReturned = 0;
        foreach ($refunds as $r) {
            $totalReceived += floatval($r['received_from_supplier']);
            $totalReturned += floatval($r['returned_to_customer']);
        }
        
        echo json_encode([
            'success' => true,
            'refunds' => $refunds,
            'summary' => [
                'total_received' => $totalReceived,
                'total_returned' => $totalReturned,
                'total_profit' => $totalReceived - $totalReturned
            ]
        ]);
    }
    
    // POST - הוספת החזר
    elseif ($method === 'POST') {
        $input = json_decode(file_get_contents('php://input'), true);
        
        $orderId = $input['order_id'] ?? null;
        $description = trim($input['description'] ?? '');
        $supplierName = trim($input['supplier_name'] ?? '');
        $airline = trim($input['airline'] ?? '');
        $receivedFromSupplier = floatval($input['received_from_supplier'] ?? 0);
        $returnedToCustomer = floatval($input['returned_to_customer'] ?? 0);
        $refundDate = $input['refund_date'] ?? date('Y-m-d');
        $notes = trim($input['notes'] ?? '');
        
        // וידוא
        if (!$orderId) {
            throw new Exception('חסר מזהה הזמנה');
        }
        if (empty($description)) {
            throw new Exception('חסר תיאור ההחזר');
        }
        if ($receivedFromSupplier <= 0 && $returnedToCustomer <= 0) {
            throw new Exception('יש להזין סכום');
        }
        
        // הוספה
        $stmt = $pdo->prepare("
            INSERT INTO order_refunds 
            (order_id, description, supplier_name, airline, received_from_supplier, returned_to_customer, refund_date, notes, created_by)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $orderId,
            $description,
            $supplierName ?: null,
            $airline ?: null,
            $receivedFromSupplier,
            $returnedToCustomer,
            $refundDate,
            $notes ?: null,
            $agent['id']
        ]);
        
        $refundId = $pdo->lastInsertId();
        
        echo json_encode([
            'success' => true,
            'message' => 'ההחזר נוסף בהצלחה',
            'refund_id' => $refundId
        ]);
    }
    
    // DELETE - מחיקת החזר
    elseif ($method === 'DELETE') {
        $input = json_decode(file_get_contents('php://input'), true);
        $refundId = $input['refund_id'] ?? $_GET['refund_id'] ?? null;
        
        if (!$refundId) {
            throw new Exception('חסר מזהה החזר');
        }
        
        // רק אדמין יכול למחוק
        if ($agent['role'] !== 'admin') {
            throw new Exception('אין הרשאה למחוק החזרים');
        }
        
        $stmt = $pdo->prepare("DELETE FROM order_refunds WHERE id = ?");
        $stmt->execute([$refundId]);
        
        if ($stmt->rowCount() > 0) {
            echo json_encode([
                'success' => true,
                'message' => 'ההחזר נמחק בהצלחה'
            ]);
        } else {
            throw new Exception('ההחזר לא נמצא');
        }
    }
    
    else {
        throw new Exception('שיטה לא נתמכת');
    }
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
?>
