<?php
// API - שמירת הזמנה לייט (עם קבצים)
header('Content-Type: application/json');

require_once '../includes/auth.php';

// בדיקת התחברות
if (!isLoggedIn()) {
    echo json_encode(['success' => false, 'error' => 'לא מחובר']);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'error' => 'שיטה לא נתמכת']);
    exit;
}

try {
    $pdo = getDB();
    
    // יצירת טוקן ייחודי
    $token = generateToken();
    
    // תיקייה לשמירת קבצים
    $uploadDirRelative = 'uploads/vouchers/' . date('Y/m/');  // נתיב יחסי ל-root (בלי ../)
    $uploadDirFull = __DIR__ . '/../' . $uploadDirRelative;   // נתיב מלא בשרת
    
    if (!file_exists($uploadDirFull)) {
        mkdir($uploadDirFull, 0755, true);
    }
    
    // שמירת קבצים מ-base64
    $voucherFiles = [];
    
    // פונקציה לשמירת קובץ מ-base64
    function saveBase64File($base64Data, $uploadDirFull, $uploadDirRelative, $prefix) {
        if (empty($base64Data) || strpos($base64Data, 'data:') !== 0) {
            return null;
        }
        
        // חילוץ הסוג והנתונים
        preg_match('/data:([^;]+);base64,(.+)/', $base64Data, $matches);
        if (count($matches) !== 3) {
            return null;
        }
        
        $mimeType = $matches[1];
        $data = base64_decode($matches[2]);
        
        // קביעת סיומת הקובץ
        $extensions = [
            'image/jpeg' => 'jpg',
            'image/png' => 'png',
            'image/gif' => 'gif',
            'image/webp' => 'webp',
            'application/pdf' => 'pdf'
        ];
        $ext = $extensions[$mimeType] ?? 'bin';
        
        // שם קובץ ייחודי
        $filename = $prefix . '_' . uniqid() . '.' . $ext;
        $filepathFull = $uploadDirFull . $filename;
        $filepathRelative = $uploadDirRelative . $filename;  // בלי ../
        
        // שמירה
        if (file_put_contents($filepathFull, $data)) {
            return $filepathRelative;
        }
        
        return null;
    }
    
    // בדיקה אם יש הפרדה בין טיסות
    $hasReturnFlight = !empty($_POST['voucher_return_data']);
    $orderType = $_POST['order_type'] ?? 'separate';
    $destination = sanitize($_POST['destination'] ?? '');
    $destinationHebrew = sanitize($_POST['destination_hebrew'] ?? $destination);
    
    // שמירת אישור טיסת הלוך (או "אישור טיסה" אם אין הפרדה)
    if (!empty($_POST['voucher_outbound_data'])) {
        $path = saveBase64File($_POST['voucher_outbound_data'], $uploadDirFull, $uploadDirRelative, 'flight_out');
        if ($path) {
            // שם הקובץ לפי סוג ההזמנה
            if ($orderType === 'package') {
                $label = 'אישור חבילת נופש ל' . $destinationHebrew;
            } elseif ($orderType === 'tour') {
                $label = 'אישור טיול מאורגן ל' . $destinationHebrew;
            } else {
                $label = $hasReturnFlight ? 'אישור טיסת הלוך' : 'אישור טיסה';
            }
            $voucherFiles[] = ['type' => 'flight_outbound', 'path' => $path, 'label' => $label];
        }
    }
    
    // שמירת אישור טיסת חזור (רק אם יש הפרדה)
    if ($hasReturnFlight) {
        $path = saveBase64File($_POST['voucher_return_data'], $uploadDirFull, $uploadDirRelative, 'flight_ret');
        if ($path) {
            $voucherFiles[] = ['type' => 'flight_return', 'path' => $path, 'label' => 'אישור טיסת חזור'];
        }
    }
    
    // שמירת אישורי שונות (extras)
    $extraIndex = 1;
    foreach ($_POST as $key => $value) {
        if (strpos($key, 'voucher_extra_') === 0 && strpos($key, '_data') !== false) {
            $path = saveBase64File($value, $uploadDirFull, $uploadDirRelative, 'extra_' . $extraIndex);
            if ($path) {
                $description = $_POST['extra_description'][$extraIndex - 1] ?? 'פריט נוסף ' . $extraIndex;
                $voucherFiles[] = ['type' => 'extra', 'path' => $path, 'label' => $description];
                $extraIndex++;
            }
        }
    }
    
    // שמירת קבצים נוספים (מהקטע המתקפל)
    if (!empty($_FILES['extra_file']['name'][0])) {
        $extraFileNames = $_POST['extra_file_name'] ?? [];
        
        foreach ($_FILES['extra_file']['name'] as $idx => $fileName) {
            if (empty($fileName) || $_FILES['extra_file']['error'][$idx] !== UPLOAD_ERR_OK) {
                continue;
            }
            
            $tmpPath = $_FILES['extra_file']['tmp_name'][$idx];
            $ext = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
            
            // בדיקת סוג קובץ מותר
            $allowedExts = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'pdf'];
            if (!in_array($ext, $allowedExts)) {
                continue;
            }
            
            $newFilename = 'extra_file_' . uniqid() . '.' . $ext;
            $destPath = $uploadDirFull . $newFilename;
            
            if (move_uploaded_file($tmpPath, $destPath)) {
                $label = !empty($extraFileNames[$idx]) ? $extraFileNames[$idx] : 'קובץ נוסף ' . ($idx + 1);
                $voucherFiles[] = [
                    'type' => 'extra_file', 
                    'path' => $uploadDirRelative . $newFilename, 
                    'label' => $label
                ];
            }
        }
    }
    
    // הכנת הנתונים
    $orderType = $_POST['order_type'] ?? 'separate';
    error_log("ORDER_TYPE RECEIVED: '$orderType' from POST: " . print_r($_POST['order_type'] ?? 'NOT SET', true));
    // וידוא שהערך תקין
    if (!in_array($orderType, ['separate', 'package', 'tour'])) {
        error_log("ORDER_TYPE INVALID, setting to separate");
        $orderType = 'separate';
    }
    
    // נתוני הלוך
    $airlineOutbound = sanitize($_POST['airline_outbound'] ?? '');
    $bookingCodeOutbound = sanitize($_POST['booking_code_outbound'] ?? '');
    $flightSupplier = sanitize($_POST['flight_supplier'] ?? '');
    
    // נתוני חזור
    $airlineReturn = sanitize($_POST['airline_return'] ?? '');
    $bookingCodeReturn = sanitize($_POST['booking_code_return'] ?? '');
    $returnFlightSupplier = sanitize($_POST['return_flight_supplier'] ?? '');
    
    // DEBUG - לוג לראות מה מגיע
    error_log("RETURN DATA: airline='$airlineReturn', code='$bookingCodeReturn', supplier='$returnFlightSupplier'");
    error_log("OUTBOUND DATA: airline='$airlineOutbound', code='$bookingCodeOutbound', supplier='$flightSupplier'");
    
    // העתקה אוטומטית: אם חברת תעופה וקוד הזמנה בחזור ריקים - המשתמש לא מילא פרטי חזור
    // במקרה כזה, להעתיק הכל מההלוך
    if (empty($airlineReturn) && empty($bookingCodeReturn)) {
        error_log("COPYING FROM OUTBOUND TO RETURN");
        $airlineReturn = $airlineOutbound;
        $bookingCodeReturn = $bookingCodeOutbound;
        $returnFlightSupplier = $flightSupplier;
    } else {
        error_log("NOT COPYING - airlineReturn or bookingCodeReturn not empty");
    }
    
    // פענוח מלונות - לקחת את הראשון לתאימות לאחור
    $hotelsJson = json_decode($_POST['hotels'] ?? '[]', true);
    $firstHotel = !empty($hotelsJson) && is_array($hotelsJson) ? $hotelsJson[0] : [];
    $hotelSupplier = $firstHotel['supplier'] ?? 'DIRECT';
    $hotelBookingCode = $firstHotel['booking_code'] ?? '';
    $hotelCost = (float)($firstHotel['cost'] ?? 0);
    $hotelCurrency = $firstHotel['currency'] ?? 'USD';
    $hotelPaid = $firstHotel['paid'] ?? 'unpaid';
    
    $data = [
        'draft_token' => $token,
        'agent_id' => $_SESSION['agent_id'],
        'deal_builder' => sanitize($_POST['deal_builder'] ?? ''),
        'order_type' => $orderType,
        'order_date' => $_POST['order_date'] ?? date('Y-m-d'),
        'customer_phone' => sanitize($_POST['customer_phone'] ?? ''),
        'customer_email' => (function() {
            $emails = [sanitize($_POST['customer_email'] ?? '')];
            if (!empty($_POST['additional_emails'])) {
                foreach ($_POST['additional_emails'] as $email) {
                    $email = sanitize(trim($email));
                    if ($email && filter_var($email, FILTER_VALIDATE_EMAIL)) {
                        $emails[] = $email;
                    }
                }
            }
            return implode(', ', array_filter($emails));
        })(),
        'passengers' => $_POST['passengers'] ?? '[]',
        'destination' => sanitize($_POST['destination'] ?? ''),
        'departure_date' => $_POST['departure_date'] ?: null,
        'return_date' => $_POST['return_date'] ?: null,
        'airline_outbound' => $airlineOutbound,
        'booking_code_outbound' => $bookingCodeOutbound,
        'airline_return' => $airlineReturn,
        'booking_code_return' => $bookingCodeReturn,
        'return_flight_supplier' => $returnFlightSupplier,
        'return_flight_supplier_cost' => (float)($_POST['return_flight_supplier_cost'] ?? 0),
        'return_flight_supplier_currency' => $_POST['return_flight_supplier_currency'] ?? 'ILS',
        'return_flight_supplier_paid' => $_POST['return_flight_supplier_paid'] ?? 'unpaid',
        'flight_supplier' => $flightSupplier,
        'flight_supplier_cost' => (float)($_POST['flight_supplier_cost'] ?? 0),
        'flight_supplier_currency' => $_POST['flight_supplier_currency'] ?? 'ILS',
        'flight_supplier_paid' => $_POST['flight_supplier_paid'] ?? 'unpaid',
        // שדות מלון - לוקחים מהמלון הראשון לתאימות לאחור
        'hotel_supplier' => $hotelSupplier,
        'hotel_supplier_booking' => $hotelBookingCode,
        'hotel_supplier_cost' => $hotelCost,
        'hotel_supplier_currency' => $hotelCurrency,
        'hotel_supplier_paid' => $hotelPaid,
        'luggage_details' => sanitize($_POST['luggage_details'] ?? ''),
        'total_price' => (float)($_POST['total_price'] ?? 0),
        'profit' => (float)($_POST['profit'] ?? 0),
        'usd_rate' => (float)($_POST['usd_rate'] ?? 3.65),
        'eur_rate' => (float)($_POST['eur_rate'] ?? 3.95),
        'payments' => $_POST['payments'] ?? '[]',
        'extras' => $_POST['extras'] ?? '[]',
        'email_text' => $_POST['email_text'] ?? '',
        'notes' => sanitize($_POST['notes'] ?? ''),
        'status' => 'confirmed'
    ];
    
    // שמירה בדאטאבייס
    $sql = "INSERT INTO orders (
        draft_token, agent_id, deal_builder, order_type, order_date, customer_phone, customer_email, 
        passengers, destination, departure_date, return_date,
        airline_outbound, booking_code_outbound, 
        airline_return, booking_code_return,
        return_flight_supplier, return_flight_supplier_cost, return_flight_supplier_currency, return_flight_supplier_paid,
        flight_supplier, flight_supplier_cost, flight_supplier_currency, flight_supplier_paid,
        hotel_supplier, hotel_supplier_booking, hotel_supplier_cost, hotel_supplier_currency, hotel_supplier_paid,
        luggage_details, total_price, profit, usd_rate, eur_rate, payments, extras, email_text, notes, status
    ) VALUES (
        :draft_token, :agent_id, :deal_builder, :order_type, :order_date, :customer_phone, :customer_email,
        :passengers, :destination, :departure_date, :return_date,
        :airline_outbound, :booking_code_outbound,
        :airline_return, :booking_code_return,
        :return_flight_supplier, :return_flight_supplier_cost, :return_flight_supplier_currency, :return_flight_supplier_paid,
        :flight_supplier, :flight_supplier_cost, :flight_supplier_currency, :flight_supplier_paid,
        :hotel_supplier, :hotel_supplier_booking, :hotel_supplier_cost, :hotel_supplier_currency, :hotel_supplier_paid,
        :luggage_details, :total_price, :profit, :usd_rate, :eur_rate, :payments, :extras, :email_text, :notes, :status
    )";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([
        'draft_token' => $data['draft_token'],
        'agent_id' => $data['agent_id'],
        'deal_builder' => $data['deal_builder'],
        'order_type' => $data['order_type'],
        'order_date' => $data['order_date'],
        'customer_phone' => $data['customer_phone'],
        'customer_email' => $data['customer_email'],
        'passengers' => $data['passengers'],
        'destination' => $data['destination'],
        'departure_date' => $data['departure_date'],
        'return_date' => $data['return_date'],
        'airline_outbound' => $data['airline_outbound'],
        'booking_code_outbound' => $data['booking_code_outbound'],
        'airline_return' => $data['airline_return'],
        'booking_code_return' => $data['booking_code_return'],
        'return_flight_supplier' => $data['return_flight_supplier'],
        'return_flight_supplier_cost' => $data['return_flight_supplier_cost'],
        'return_flight_supplier_currency' => $data['return_flight_supplier_currency'],
        'return_flight_supplier_paid' => $data['return_flight_supplier_paid'],
        'flight_supplier' => $data['flight_supplier'],
        'flight_supplier_cost' => $data['flight_supplier_cost'],
        'flight_supplier_currency' => $data['flight_supplier_currency'],
        'flight_supplier_paid' => $data['flight_supplier_paid'],
        'hotel_supplier' => $data['hotel_supplier'],
        'hotel_supplier_booking' => $data['hotel_supplier_booking'],
        'hotel_supplier_cost' => $data['hotel_supplier_cost'],
        'hotel_supplier_currency' => $data['hotel_supplier_currency'],
        'hotel_supplier_paid' => $data['hotel_supplier_paid'],
        'luggage_details' => $data['luggage_details'],
        'total_price' => $data['total_price'],
        'profit' => $data['profit'],
        'usd_rate' => $data['usd_rate'],
        'eur_rate' => $data['eur_rate'],
        'payments' => $data['payments'],
        'extras' => $data['extras'],
        'email_text' => $data['email_text'],
        'notes' => $data['notes'],
        'status' => $data['status']
    ]);
    
    $orderId = $pdo->lastInsertId();
    
    // שמירת מלונות בטבלה החדשה
    $hotels = json_decode($_POST['hotels'] ?? '[]', true);
    if (!empty($hotels) && is_array($hotels)) {
        $hotelStmt = $pdo->prepare("
            INSERT INTO order_hotels 
            (order_id, supplier, confirmation_code, supplier_cost, supplier_currency, supplier_paid, 
             voucher_path, sort_order)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        ");
        
        foreach ($hotels as $index => $hotel) {
            // שמירת וואוצ'ר מלון
            $voucherPath = null;
            if (!empty($_POST['voucher_hotel_' . $index . '_data'])) {
                $voucherPath = saveBase64File(
                    $_POST['voucher_hotel_' . $index . '_data'], 
                    $uploadDirFull, 
                    $uploadDirRelative, 
                    'hotel_' . $index
                );
                if ($voucherPath) {
                    $voucherFiles[] = ['type' => 'hotel', 'path' => $voucherPath, 'label' => 'אישור מלון #' . ($index + 1)];
                }
            }
            
            $hotelStmt->execute([
                $orderId,
                $hotel['supplier'] ?? 'DIRECT',
                $hotel['booking_code'] ?? '',
                (float)($hotel['cost'] ?? 0),
                $hotel['currency'] ?? 'USD',
                $hotel['paid'] ?? 'unpaid',
                $voucherPath,
                $index + 1
            ]);
        }
    }
    
    // עדכון הטיוטה עם מספר ההזמנה (אם יש)
    $draftId = $_POST['draft_id'] ?? null;
    if ($draftId) {
        $pdo->prepare("UPDATE drafts SET order_id = ? WHERE id = ?")->execute([$orderId, $draftId]);
        
        // עדכון גם בטבלת bank_transfers אם יש העברות בנקאיות לטיוטה הזו
        try {
            $pdo->prepare("UPDATE bank_transfers SET order_id = ? WHERE draft_id = ?")->execute([$orderId, $draftId]);
        } catch (Exception $e) {
            // אם אין טבלה או שגיאה - ממשיכים
        }
    }
    
    // טיפול בהעברות בנקאיות - יצירת רשומות לכל העברה (גם בלי קובץ)
    $payments = json_decode($_POST['payments'] ?? '[]', true);
    $bankProofIndexes = $_POST['bank_proof_indexes'] ?? [];
    
    // מיפוי קבצים לאינדקסים
    $uploadedProofs = [];
    if (!empty($_FILES['bank_proofs']['tmp_name'])) {
        $bankUploadDir = __DIR__ . '/../uploads/bank_proofs/' . date('Y/m/');
        if (!file_exists($bankUploadDir)) {
            mkdir($bankUploadDir, 0755, true);
        }
        
        foreach ($_FILES['bank_proofs']['tmp_name'] as $key => $tmpName) {
            if (empty($tmpName)) continue;
            
            $paymentIndex = isset($bankProofIndexes[$key]) ? (int)$bankProofIndexes[$key] : $key;
            
            // שמירת הקובץ
            $originalName = $_FILES['bank_proofs']['name'][$key];
            $ext = pathinfo($originalName, PATHINFO_EXTENSION);
            $filename = 'bank_' . $orderId . '_' . $paymentIndex . '_' . uniqid() . '.' . $ext;
            $filepath = $bankUploadDir . $filename;
            $relPath = 'uploads/bank_proofs/' . date('Y/m/') . $filename;
            
            if (move_uploaded_file($tmpName, $filepath)) {
                $uploadedProofs[$paymentIndex] = $relPath;
            }
        }
    }
    
    // יצירת רשומות העברה בנקאית לתשלומים חדשים
    // בודק כל תשלום - אם זו העברה בנקאית בלי bank_transfer_id, יוצר רשומה
    $destination = sanitize($_POST['destination'] ?? '');
    $bankTransfersCreated = 0;
    $paymentsUpdated = false;
    
    // לוג לדיבאג
    $debugLog = __DIR__ . '/../logs/bank_debug.log';
    @file_put_contents($debugLog, date('Y-m-d H:i:s') . " - Order $orderId - Checking payments for bank transfers\n", FILE_APPEND);
    
    foreach ($payments as $index => &$payment) {
        $method = $payment['method'] ?? '';
        $existingBankTransferId = $payment['bank_transfer_id'] ?? null;
        
        // בדיקה אם זו העברה בנקאית חדשה (בלי bank_transfer_id קיים)
        if ($method === 'bank_transfer' && !empty($payment['amount']) && empty($existingBankTransferId)) {
            try {
                $proofFile = $uploadedProofs[$index] ?? null;
                $payerName = $payment['name'] ?? 'לקוח';
                $amount = $payment['amount'] ?? 0;
            
                // יצירת רשומת העברה בנקאית
                $stmt = $pdo->prepare("
                    INSERT INTO bank_transfers (
                        order_id, transferrer_name, amount, screenshot_path, status, created_at
                    ) VALUES (?, ?, ?, ?, 'pending', NOW())
                ");
                $stmt->execute([
                    $orderId,
                    $payerName,
                    $amount,
                    $proofFile
                ]);
                
                // שמירת ה-ID של ההעברה בתשלום - קשר ישיר 1:1
                $bankTransferId = $pdo->lastInsertId();
                $payment['bank_transfer_id'] = $bankTransferId;
                $paymentsUpdated = true;
                
                $bankTransfersCreated++;
                @file_put_contents($debugLog, "SUCCESS: Bank transfer #$bankTransferId created for order $orderId, payment index $index\n", FILE_APPEND);
                
            } catch (Exception $e) {
                @file_put_contents($debugLog, "ERROR: " . $e->getMessage() . "\n", FILE_APPEND);
                error_log("Error saving bank transfer: " . $e->getMessage());
            }
        } elseif ($method === 'bank_transfer' && !empty($existingBankTransferId)) {
            @file_put_contents($debugLog, "SKIP: Payment $index already has bank_transfer_id #$existingBankTransferId\n", FILE_APPEND);
        }
    }
    unset($payment); // חשוב אחרי foreach עם reference
    
    // עדכון ה-payments בהזמנה עם ה-bank_transfer_id
    if ($paymentsUpdated) {
        try {
            $stmt = $pdo->prepare("UPDATE orders SET payments = ? WHERE id = ?");
            $stmt->execute([json_encode($payments, JSON_UNESCAPED_UNICODE), $orderId]);
            @file_put_contents($debugLog, "Updated payments JSON with bank_transfer_ids\n", FILE_APPEND);
        } catch (Exception $e) {
            @file_put_contents($debugLog, "ERROR updating payments: " . $e->getMessage() . "\n", FILE_APPEND);
        }
    }
    
    @file_put_contents($debugLog, "Total bank transfers created: $bankTransfersCreated\n\n", FILE_APPEND);
    
    // שמירת קבצי הוואצ'ר בקובץ JSON
    if (!empty($voucherFiles)) {
        $voucherDataFile = $uploadDirFull . 'order_' . $orderId . '_files.json';
        file_put_contents($voucherDataFile, json_encode([
            'order_id' => $orderId,
            'token' => $token,
            'files' => $voucherFiles,
            'created_at' => date('Y-m-d H:i:s')
        ]));
    }
    
    // שליחת קבלות אוטומטית - לכל ההזמנות
    $receiptsSent = 0;
    $receiptsErrors = [];
    $updatedPayments = [];
    
    require_once '../includes/invoice4u.php';
    
    $destination = sanitize($_POST['destination'] ?? '');
    $departureDate = $_POST['departure_date'] ?? '';
    $returnDate = $_POST['return_date'] ?? '';
    $orderDates = $departureDate . ' - ' . $returnDate;
    
    if (!empty($payments)) {
        $invoice = new Invoice4U();
        
        foreach ($payments as $index => $payment) {
            $email = $payment['email'] ?? '';
            $name = $payment['name'] ?? 'לקוח';
            $amount = floatval($payment['amount'] ?? 0);
            $method = $payment['method'] ?? 'credit_card';
            $installments = intval($payment['installments'] ?? 1);
            $last4 = $payment['last4'] ?? '';
            
            // העתק את התשלום
            $updatedPayment = $payment;
                
                // דלג על קבלה אם אין מייל או סכום
                if (!$email || !filter_var($email, FILTER_VALIDATE_EMAIL) || $amount <= 0) {
                    $updatedPayments[] = $updatedPayment;
                    continue;
                }
                
                // המרת שיטת תשלום
                $paymentType = 'credit_card';
                if ($method === 'bank_transfer') {
                    $paymentType = 'transfer';
                } elseif ($method === 'cash') {
                    $paymentType = 'cash';
                } elseif ($method === 'bit') {
                    $paymentType = 'bit';
                }
                
                // תיאור לקבלה
                $description = 'תשלום עבור חבילת נופש';
                if ($destination) {
                    $description .= ' - ' . $destination;
                }
                
                // יצירת קבלה
                $receiptData = [
                    'customer_name' => $name,
                    'customer_email' => $email,
                    'amount' => $amount,
                    'description' => $description,
                    'payment_type' => $paymentType,
                    'last_4_digits' => $last4,
                    'installments' => $installments
                ];
                
                $result = $invoice->createReceipt($receiptData);
                
                if ($result['success']) {
                    $receiptsSent++;
                    // שמירת מספר קבלה ולינק PDF
                    $updatedPayment['receipt_number'] = $result['document_number'] ?? '';
                    $updatedPayment['receipt_pdf'] = $result['pdf_link'] ?? '';
                    error_log("Order $orderId: Receipt #{$result['document_number']} sent to $email - Amount: $amount ILS");
                } else {
                    $receiptsErrors[] = "נכשל עבור $name: " . ($result['error'] ?? 'שגיאה');
                    error_log("Order $orderId: Receipt failed for $email - " . ($result['error'] ?? 'unknown'));
                }
                
                $updatedPayments[] = $updatedPayment;
            }
            
            // עדכון ההזמנה עם מספרי הקבלות
            if (!empty($updatedPayments)) {
                try {
                    $stmt = $pdo->prepare("UPDATE orders SET payments = ? WHERE id = ?");
                    $stmt->execute([json_encode($updatedPayments, JSON_UNESCAPED_UNICODE), $orderId]);
                } catch (Exception $e) {
                    error_log("Error updating payments with receipt numbers: " . $e->getMessage());
                }
            }
        }
    
    echo json_encode([
        'success' => true,
        'order_id' => $orderId,
        'token' => $token,
        'files_count' => count($voucherFiles),
        'receipts_sent' => $receiptsSent,
        'receipts_errors' => $receiptsErrors,
        'from_draft' => !empty($draftId)
    ]);
    
} catch (Exception $e) {
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
}
