<?php
// שליחת אישור הזמנה עם קבצים מצורפים
require_once '../includes/auth.php';
requireLogin();

header('Content-Type: application/json; charset=utf-8');

try {
    $pdo = getDB();
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    $orderId = $input['order_id'] ?? null;
    $email = $input['email'] ?? '';
    $subject = $input['subject'] ?? 'אישור הזמנה - חופשה במבצע';
    $body = $input['body'] ?? '';
    $existingFiles = $input['existing_files'] ?? [];
    $newFiles = $input['new_files'] ?? [];
    $attachTerms = $input['attach_terms'] ?? false;
    
    // לוג לדיבאג
    $debugLog = __DIR__ . '/../logs/email_debug.log';
    $debugDir = dirname($debugLog);
    if (!is_dir($debugDir)) mkdir($debugDir, 0755, true);
    file_put_contents($debugLog, date('Y-m-d H:i:s') . " - Files received: " . json_encode($existingFiles) . "\n", FILE_APPEND);
    
    if (!$orderId) {
        throw new Exception('חסר מזהה הזמנה');
    }
    
    if (!$email || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        throw new Exception('כתובת מייל לא תקינה');
    }
    
    if (empty($body)) {
        throw new Exception('תוכן המייל ריק');
    }
    
    // שליפת פרטי ה-agent שיצר את ההזמנה
    $agentEmail = null;
    $agentSendFrom = null;
    $agentName = null;
    try {
        $stmtAgent = $pdo->prepare("
            SELECT a.email, a.send_from_email, a.full_name 
            FROM orders o 
            JOIN agents a ON o.agent_id = a.id 
            WHERE o.id = ?
        ");
        $stmtAgent->execute([$orderId]);
        $agentData = $stmtAgent->fetch(PDO::FETCH_ASSOC);
        if ($agentData) {
            $agentEmail = $agentData['email'] ?? null;
            $agentSendFrom = $agentData['send_from_email'] ?? null;
            $agentName = $agentData['full_name'] ?? null;
        }
    } catch (Exception $e) {
        // fallback אם העמודה לא קיימת
        $stmtAgent = $pdo->prepare("
            SELECT a.email, a.full_name 
            FROM orders o 
            JOIN agents a ON o.agent_id = a.id 
            WHERE o.id = ?
        ");
        $stmtAgent->execute([$orderId]);
        $agentData = $stmtAgent->fetch(PDO::FETCH_ASSOC);
        if ($agentData) {
            $agentEmail = $agentData['email'] ?? null;
            $agentName = $agentData['full_name'] ?? null;
        }
    }
    
    // שימוש באימייל הנציג אם קיים
    $fromEmail = (!empty($agentSendFrom) && filter_var($agentSendFrom, FILTER_VALIDATE_EMAIL)) 
        ? $agentSendFrom 
        : 'orders@holidayinsale.com';
    $fromName = !empty($agentName) ? $agentName . ' - חופשה במבצע' : 'חופשה במבצע';
    
    // בניית המייל עם קבצים מצורפים (multipart)
    $boundary = md5(time());
    
    $headers = [];
    $headers[] = 'MIME-Version: 1.0';
    $headers[] = 'Content-Type: multipart/mixed; boundary="' . $boundary . '"';
    $headers[] = 'From: =?UTF-8?B?' . base64_encode($fromName) . '?= <' . $fromEmail . '>';
    $headers[] = 'Reply-To: ' . $fromEmail;
    
    // הוספת CC ל-agent
    if (!empty($agentEmail) && filter_var($agentEmail, FILTER_VALIDATE_EMAIL)) {
        $headers[] = 'Cc: ' . $agentEmail;
    }
    
    // תוכן המייל כ-HTML - יישור ימין, חתימה מתחת
    $htmlBody = '<!DOCTYPE html><html dir="rtl"><head><meta charset="UTF-8"></head>';
    $htmlBody .= '<body style="font-family:Arial,sans-serif;direction:rtl;text-align:right;line-height:1.8;margin:0;padding:20px;">';
    $htmlBody .= '<div style="text-align:right;">';
    $htmlBody .= nl2br(htmlspecialchars($body));
    $htmlBody .= '<br><br><br>';
    $htmlBody .= '<img src="https://hufshagroup.com/newprojectmillion/logoformail.png" alt="חופשה במבצע" style="max-width:150px;">';
    $htmlBody .= '</div></body></html>';
    
    // בניית גוף המייל
    $message = '';
    
    // חלק HTML
    $message .= "--{$boundary}\r\n";
    $message .= "Content-Type: text/html; charset=UTF-8\r\n";
    $message .= "Content-Transfer-Encoding: base64\r\n\r\n";
    $message .= chunk_split(base64_encode($htmlBody)) . "\r\n";
    
    // קבצים קיימים - תיקון נתיב
    $basePath = __DIR__ . '/../';
    $filesAttached = 0;
    
    foreach ($existingFiles as $filePath) {
        // נתיב מלא
        $fullPath = $basePath . $filePath;
        
        file_put_contents($debugLog, "  Trying: {$fullPath} - exists: " . (file_exists($fullPath) ? 'YES' : 'NO') . "\n", FILE_APPEND);
        
        if (file_exists($fullPath)) {
            $filename = basename($fullPath);
            $content = file_get_contents($fullPath);
            $mimeType = mime_content_type($fullPath) ?: 'application/octet-stream';
            
            $message .= "--{$boundary}\r\n";
            $message .= "Content-Type: {$mimeType}; name=\"{$filename}\"\r\n";
            $message .= "Content-Transfer-Encoding: base64\r\n";
            $message .= "Content-Disposition: attachment; filename=\"{$filename}\"\r\n\r\n";
            $message .= chunk_split(base64_encode($content)) . "\r\n";
            $filesAttached++;
        }
    }
    
    // קבצים חדשים (base64)
    foreach ($newFiles as $file) {
        if (!empty($file['data']) && !empty($file['name'])) {
            // הסרת prefix של data URL
            $data = $file['data'];
            if (preg_match('/^data:([^;]+);base64,(.+)$/', $data, $matches)) {
                $mimeType = $matches[1];
                $content = base64_decode($matches[2]);
            } else {
                continue;
            }
            
            $filename = $file['name'];
            
            $message .= "--{$boundary}\r\n";
            $message .= "Content-Type: {$mimeType}; name=\"{$filename}\"\r\n";
            $message .= "Content-Transfer-Encoding: base64\r\n";
            $message .= "Content-Disposition: attachment; filename=\"{$filename}\"\r\n\r\n";
            $message .= chunk_split(base64_encode($content)) . "\r\n";
            $filesAttached++;
        }
    }
    
    // תקנון
    if ($attachTerms) {
        $termsPath = __DIR__ . '/../terms.pdf';
        if (file_exists($termsPath)) {
            $content = file_get_contents($termsPath);
            $message .= "--{$boundary}\r\n";
            $message .= "Content-Type: application/pdf; name=\"terms.pdf\"\r\n";
            $message .= "Content-Transfer-Encoding: base64\r\n";
            $message .= "Content-Disposition: attachment; filename=\"terms.pdf\"\r\n\r\n";
            $message .= chunk_split(base64_encode($content)) . "\r\n";
            $filesAttached++;
        }
    }
    
    $message .= "--{$boundary}--\r\n";
    
    file_put_contents($debugLog, "  Total files attached: {$filesAttached}\n", FILE_APPEND);
    
    // שליחה
    $encodedSubject = '=?UTF-8?B?' . base64_encode($subject) . '?=';
    $sent = mail($email, $encodedSubject, $message, implode("\r\n", $headers));
    
    if ($sent) {
        // לוג
        $logFile = __DIR__ . '/../logs/email_' . date('Y-m-d') . '.log';
        
        $logEntry = date('Y-m-d H:i:s') . " - Order #{$orderId} confirmation sent to {$email}\n";
        $logEntry .= "  Files attached: {$filesAttached}\n";
        
        file_put_contents($logFile, $logEntry, FILE_APPEND);
        
        echo json_encode([
            'success' => true,
            'message' => 'המייל נשלח בהצלחה',
            'files_attached' => $filesAttached
        ]);
    } else {
        throw new Exception('שגיאה בשליחת המייל');
    }
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
