<?php
// API - שליחת מייל אישור הזמנה ללקוח
header('Content-Type: application/json');

require_once '../includes/auth.php';

// בדיקת התחברות
if (!isLoggedIn()) {
    echo json_encode(['success' => false, 'error' => 'לא מחובר']);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'error' => 'שיטה לא נתמכת']);
    exit;
}

try {
    $pdo = getDB();
    
    $customerEmailInput = $_POST['customer_email'] ?? '';
    $emailSubject = $_POST['email_subject'] ?? 'אישור הזמנה - חופשה במבצע';
    $emailBody = $_POST['email_body'] ?? '';
    $orderId = $_POST['order_id'] ?? '';
    
    // פיצול מיילים (אם יש כמה, מופרדים בפסיק)
    $emails = array_map('trim', explode(',', $customerEmailInput));
    $emails = array_filter($emails, function($e) {
        return filter_var($e, FILTER_VALIDATE_EMAIL);
    });
    
    if (empty($emails)) {
        echo json_encode(['success' => false, 'error' => 'כתובת מייל לא תקינה']);
        exit;
    }
    
    $customerEmail = $emails[0]; // המייל הראשי
    $ccEmails = array_slice($emails, 1); // מיילים נוספים
    
    // שליפת פרטי ה-agent שיצר את ההזמנה
    $agentEmail = null;
    $agentSendFrom = null;
    $agentName = null;
    if (!empty($orderId)) {
        try {
            // ניסיון לשלוף עם send_from_email
            $stmt = $pdo->prepare("
                SELECT a.email, a.send_from_email, a.full_name 
                FROM orders o 
                JOIN agents a ON o.agent_id = a.id 
                WHERE o.id = ?
            ");
            $stmt->execute([$orderId]);
            $agentData = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($agentData) {
                $agentEmail = $agentData['email'] ?? null;
                $agentSendFrom = $agentData['send_from_email'] ?? null;
                $agentName = $agentData['full_name'] ?? null;
            }
        } catch (Exception $e) {
            // fallback - אם העמודה לא קיימת
            $stmt = $pdo->prepare("
                SELECT a.email, a.full_name 
                FROM orders o 
                JOIN agents a ON o.agent_id = a.id 
                WHERE o.id = ?
            ");
            $stmt->execute([$orderId]);
            $agentData = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($agentData) {
                $agentEmail = $agentData['email'] ?? null;
                $agentName = $agentData['full_name'] ?? null;
            }
        }
    }
    
    // בדיקת תקינות
    if (empty($customerEmail) || !filter_var($customerEmail, FILTER_VALIDATE_EMAIL)) {
        echo json_encode(['success' => false, 'error' => 'כתובת מייל לא תקינה']);
        exit;
    }
    
    if (empty($emailBody)) {
        echo json_encode(['success' => false, 'error' => 'תוכן המייל ריק']);
        exit;
    }
    
    // איסוף קבצים מצורפים
    $attachments = [];
    
    // הוספת הלוגו כקובץ מוטמע
    $logoPath = '../logoformail.png';
    $hasLogo = file_exists($logoPath);
    
    // הוספת תקנון PDF
    $takanonPath = '../תקנון ותנאי רכישה חופשה במבצע.pdf';
    $hasTakanon = file_exists($takanonPath);
    
    if (!empty($orderId)) {
        $uploadDirs = [
            '../uploads/vouchers/' . date('Y/m/'),
            '../uploads/vouchers/'
        ];
        
        foreach ($uploadDirs as $dir) {
            $jsonFile = $dir . 'order_' . $orderId . '_files.json';
            if (file_exists($jsonFile)) {
                $data = json_decode(file_get_contents($jsonFile), true);
                if ($data && !empty($data['files'])) {
                    foreach ($data['files'] as $index => $file) {
                        $filePath = $file['path'];
                        if (strpos($filePath, '../') === 0) {
                            $filePath = substr($filePath, 3);
                        }
                        $fullPath = '../' . $filePath;
                        
                        if (file_exists($fullPath)) {
                            $ext = pathinfo($fullPath, PATHINFO_EXTENSION);
                            $attachments[] = [
                                'path' => $fullPath,
                                'name' => $file['label'] . '.' . $ext,
                                'label' => $file['label'],
                                'cid' => 'image' . $index . '@voucher'
                            ];
                        }
                    }
                }
                break;
            }
        }
    }
    
    // יצירת מייל
    $boundary = md5(time());
    $boundaryRelated = md5(time() . 'related');
    
    // שימוש באימייל הנציג אם קיים, אחרת ברירת מחדל
    $fromEmail = (!empty($agentSendFrom) && filter_var($agentSendFrom, FILTER_VALIDATE_EMAIL)) 
        ? $agentSendFrom 
        : 'orders@holidayinsale.com';
    $fromName = !empty($agentName) ? $agentName . ' - חופשה במבצע' : 'חופשה במבצע';
    
    // כותרות
    $headers = [];
    $headers[] = "From: =?UTF-8?B?" . base64_encode($fromName) . "?= <{$fromEmail}>";
    $headers[] = "Reply-To: {$fromEmail}";
    
    // בניית רשימת CC - כולל agent ומיילים נוספים
    $ccList = [];
    if (!empty($agentEmail) && filter_var($agentEmail, FILTER_VALIDATE_EMAIL)) {
        $ccList[] = $agentEmail;
    }
    if (!empty($ccEmails)) {
        $ccList = array_merge($ccList, $ccEmails);
    }
    if (!empty($ccList)) {
        $headers[] = "Cc: " . implode(', ', $ccList);
    }
    
    $headers[] = "MIME-Version: 1.0";
    $headers[] = "Content-Type: multipart/mixed; boundary=\"{$boundary}\"";
    
    // בניית HTML עם תמונות מוטמעות בגודל קבוע
    $htmlBody = '<!DOCTYPE html><html dir="rtl"><head><meta charset="UTF-8"></head><body style="font-family: Arial, sans-serif; font-size: 14px; line-height: 1.8; direction: rtl; text-align: right;">';
    $htmlBody .= '<div style="padding: 20px; max-width: 600px;">';
    $htmlBody .= nl2br(htmlspecialchars($emailBody));
    
    // הוספת לוגו בסוף הטקסט
    if ($hasLogo) {
        $htmlBody .= '<div style="margin-top: 30px;"><img src="cid:logo@hufshagroup" style="max-width: 200px; height: auto;" alt="חופשה במבצע"></div>';
    }
    
    $htmlBody .= '</div>';
    
    // הוספת תמונות בגוף המייל בגודל קבוע
    if (!empty($attachments)) {
        $htmlBody .= '<hr style="border: 1px solid #ddd; margin: 30px 0;">';
        $htmlBody .= '<div style="padding: 20px; direction: rtl; text-align: right;">';
        $htmlBody .= '<h2 style="color: #667eea; font-size: 18px;">📎 אישורי הזמנה מצורפים:</h2>';
        
        foreach ($attachments as $attachment) {
            $htmlBody .= '<div style="margin: 20px 0; padding: 15px; background: #f5f5f5; border-radius: 10px;">';
            $htmlBody .= '<h3 style="color: #333; margin-bottom: 10px; font-size: 16px;">📄 ' . htmlspecialchars($attachment['label']) . '</h3>';
            $htmlBody .= '<img src="cid:' . $attachment['cid'] . '" style="width: 500px; max-width: 100%; height: auto; border: 1px solid #ddd; border-radius: 5px; display: block;" alt="' . htmlspecialchars($attachment['label']) . '">';
            $htmlBody .= '</div>';
        }
        
        $htmlBody .= '</div>';
    }
    
    $htmlBody .= '</body></html>';
    
    // בניית גוף המייל - multipart/mixed (מכיל related + attachments)
    $mailBody = "--{$boundary}\r\n";
    $mailBody .= "Content-Type: multipart/related; boundary=\"{$boundaryRelated}\"\r\n\r\n";
    
    // HTML part
    $mailBody .= "--{$boundaryRelated}\r\n";
    $mailBody .= "Content-Type: text/html; charset=UTF-8\r\n";
    $mailBody .= "Content-Transfer-Encoding: base64\r\n\r\n";
    $mailBody .= chunk_split(base64_encode($htmlBody)) . "\r\n";
    
    // תמונות inline (לתצוגה בגוף המייל)
    foreach ($attachments as $attachment) {
        $fileContent = file_get_contents($attachment['path']);
        $mimeType = mime_content_type($attachment['path']);
        $fileName = $attachment['name'];
        
        $mailBody .= "--{$boundaryRelated}\r\n";
        $mailBody .= "Content-Type: {$mimeType}; name=\"{$fileName}\"\r\n";
        $mailBody .= "Content-Transfer-Encoding: base64\r\n";
        $mailBody .= "Content-ID: <{$attachment['cid']}>\r\n";
        $mailBody .= "Content-Disposition: inline; filename=\"{$fileName}\"\r\n\r\n";
        $mailBody .= chunk_split(base64_encode($fileContent)) . "\r\n";
    }
    
    // הוספת הלוגו כתמונה מוטמעת
    if ($hasLogo) {
        $logoContent = file_get_contents($logoPath);
        $logoMimeType = mime_content_type($logoPath);
        
        $mailBody .= "--{$boundaryRelated}\r\n";
        $mailBody .= "Content-Type: {$logoMimeType}; name=\"logo.png\"\r\n";
        $mailBody .= "Content-Transfer-Encoding: base64\r\n";
        $mailBody .= "Content-ID: <logo@hufshagroup>\r\n";
        $mailBody .= "Content-Disposition: inline; filename=\"logo.png\"\r\n\r\n";
        $mailBody .= chunk_split(base64_encode($logoContent)) . "\r\n";
    }
    
    $mailBody .= "--{$boundaryRelated}--\r\n\r\n";
    
    // קבצים מצורפים (attachments להורדה)
    foreach ($attachments as $attachment) {
        $fileContent = file_get_contents($attachment['path']);
        $mimeType = mime_content_type($attachment['path']);
        $fileName = $attachment['name'];
        
        $mailBody .= "--{$boundary}\r\n";
        $mailBody .= "Content-Type: {$mimeType}; name=\"{$fileName}\"\r\n";
        $mailBody .= "Content-Transfer-Encoding: base64\r\n";
        $mailBody .= "Content-Disposition: attachment; filename=\"{$fileName}\"\r\n\r\n";
        $mailBody .= chunk_split(base64_encode($fileContent)) . "\r\n";
    }
    
    // צירוף תקנון PDF
    if ($hasTakanon) {
        $takanonContent = file_get_contents($takanonPath);
        $takanonFileName = 'תקנון ותנאי רכישה - חופשה במבצע.pdf';
        
        $mailBody .= "--{$boundary}\r\n";
        $mailBody .= "Content-Type: application/pdf; name=\"{$takanonFileName}\"\r\n";
        $mailBody .= "Content-Transfer-Encoding: base64\r\n";
        $mailBody .= "Content-Disposition: attachment; filename=\"{$takanonFileName}\"\r\n\r\n";
        $mailBody .= chunk_split(base64_encode($takanonContent)) . "\r\n";
    }
    
    $mailBody .= "--{$boundary}--";
    
    // קידוד נושא בעברית
    $encodedSubject = "=?UTF-8?B?" . base64_encode($emailSubject) . "?=";
    
    // שליחה לכל הכתובות
    $allRecipients = array_merge([$customerEmail], $ccEmails);
    $toAddresses = implode(', ', $allRecipients);
    
    $sent = mail($toAddresses, $encodedSubject, $mailBody, implode("\r\n", $headers));
    
    if ($sent) {
        $attachCount = count($attachments);
        $message = 'המייל נשלח בהצלחה ל-' . $toAddresses;
        if ($attachCount > 0) {
            $message .= ' (עם ' . $attachCount . ' קבצים)';
        }
        echo json_encode(['success' => true, 'message' => $message]);
    } else {
        echo json_encode(['success' => false, 'error' => 'שגיאה בשליחת המייל']);
    }
    
} catch (Exception $e) {
    echo json_encode(['success' => false, 'error' => 'שגיאה: ' . $e->getMessage()]);
}
