<?php
// API - שליחת קבלות למשלמים דרך Invoice4U
header('Content-Type: application/json; charset=utf-8');

require_once '../includes/auth.php';
require_once '../includes/config.php';
require_once '../includes/invoice4u.php';

if (!isLoggedIn()) {
    echo json_encode(['success' => false, 'error' => 'לא מחובר']);
    exit;
}

$agent = getCurrentAgent();

try {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (empty($input['payments'])) {
        throw new Exception('לא נמצאו תשלומים לשליחה');
    }
    
    $payments = $input['payments'];
    $orderDestination = $input['order_destination'] ?? '';
    $orderDates = $input['order_dates'] ?? '';
    
    // יצירת אובייקט Invoice4U
    $invoice = new Invoice4U();
    
    $sent = 0;
    $errors = [];
    
    foreach ($payments as $payment) {
        $email = $payment['email'] ?? '';
        $name = $payment['name'] ?? 'לקוח';
        $amount = floatval($payment['amount'] ?? 0);
        $method = $payment['method'] ?? 'אשראי';
        $installments = intval($payment['installments'] ?? 1);
        $last4 = $payment['last4'] ?? '';
        
        if (!$email || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $errors[] = "מייל לא תקין: $email";
            continue;
        }
        
        if ($amount <= 0) {
            $errors[] = "סכום לא תקין עבור: $name";
            continue;
        }
        
        // המרת שיטת תשלום לפורמט Invoice4U
        $paymentType = 'credit_card';
        if ($method === 'העברה בנקאית') {
            $paymentType = 'transfer';
        } elseif ($method === 'מזומן') {
            $paymentType = 'cash';
        } elseif ($method === 'ביט') {
            $paymentType = 'bit';
        }
        
        // תיאור לקבלה
        $description = 'תשלום עבור חבילת נופש';
        if ($orderDestination) {
            $description .= ' - ' . $orderDestination;
        }
        if ($orderDates && $orderDates !== ' - ') {
            $description .= ' (' . $orderDates . ')';
        }
        
        // יצירת קבלה ב-Invoice4U
        $receiptData = [
            'customer_name' => $name,
            'customer_email' => $email,
            'amount' => $amount,
            'description' => $description,
            'payment_type' => $paymentType,
            'last_4_digits' => $last4,
            'installments' => $installments
        ];
        
        $result = $invoice->createReceipt($receiptData);
        
        if ($result['success']) {
            $sent++;
            error_log("Receipt created via Invoice4U for $email - Amount: $amount ILS - Doc: " . ($result['document_number'] ?? 'N/A'));
        } else {
            $errorMsg = $result['error'] ?? 'שגיאה לא ידועה';
            $errors[] = "נכשל ליצור קבלה עבור $name: $errorMsg";
            error_log("Invoice4U receipt failed for $email: $errorMsg");
        }
    }
    
    echo json_encode([
        'success' => $sent > 0,
        'sent' => $sent,
        'total' => count($payments),
        'errors' => $errors
    ]);
    
} catch (Exception $e) {
    error_log("send_receipts.php error: " . $e->getMessage());
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
}
