<?php
// קבלת אישור העברה בנקאית מלקוח
// הסכום מתעדכן מיד - אח"כ הנציגים מאמתים מול הבנק
require_once '../includes/config.php';

header('Content-Type: application/json; charset=utf-8');

$pdo = getDB();

try {
    $token = $_POST['token'] ?? '';
    $draftId = (int)($_POST['draft_id'] ?? 0);
    $transferrerName = trim($_POST['transferrer_name'] ?? $_POST['payer_name'] ?? '');
    $transferrerPhone = trim($_POST['transferrer_phone'] ?? '');
    $transferrerEmail = trim($_POST['payer_email'] ?? $_POST['transferrer_email'] ?? '');
    $amount = floatval($_POST['amount'] ?? 0);
    $transferDate = $_POST['transfer_date'] ?? date('Y-m-d');
    $referenceNumber = trim($_POST['reference_number'] ?? '');
    $notes = trim($_POST['notes'] ?? '');
    
    // וידוא נתונים
    if (!$token || !$draftId || !$transferrerName || $amount <= 0) {
        throw new Exception('חסרים פרטים חובה');
    }
    
    // וידוא הטיוטה קיימת
    $stmt = $pdo->prepare("SELECT * FROM drafts WHERE id = ? AND token = ?");
    $stmt->execute([$draftId, $token]);
    $draft = $stmt->fetch();
    
    if (!$draft) {
        throw new Exception('טיוטה לא נמצאה');
    }
    
    // וידוא שיש קובץ
    if (!isset($_FILES['screenshot']) || $_FILES['screenshot']['error'] !== UPLOAD_ERR_OK) {
        throw new Exception('יש להעלות צילום מסך של ההעברה');
    }
    
    // העלאת הקובץ
    $uploadDir = '../uploads/bank_transfers/';
    if (!is_dir($uploadDir)) {
        mkdir($uploadDir, 0755, true);
    }
    
    $file = $_FILES['screenshot'];
    $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    $allowedExts = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'pdf'];
    
    if (!in_array($ext, $allowedExts)) {
        throw new Exception('סוג קובץ לא נתמך');
    }
    
    $newFilename = 'transfer_' . $draftId . '_' . time() . '.' . $ext;
    $targetPath = $uploadDir . $newFilename;
    
    if (!move_uploaded_file($file['tmp_name'], $targetPath)) {
        throw new Exception('שגיאה בהעלאת הקובץ');
    }
    
    // שמירה בדאטאבייס - סטטוס "pending" (ממתין לאימות מול הבנק)
    $stmt = $pdo->prepare("INSERT INTO bank_transfers 
        (draft_id, order_id, transferrer_name, transferrer_phone, amount, transfer_date, reference_number, screenshot_path, notes, status) 
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending')");
    
    $stmt->execute([
        $draftId,
        $draft['order_id'] ?? $draft['parent_order_id'] ?? null,
        $transferrerName,
        $transferrerPhone,
        $amount,
        $transferDate,
        $referenceNumber,
        'uploads/bank_transfers/' . $newFilename,
        $notes
    ]);
    
    $transferId = $pdo->lastInsertId();
    
    // עדכון היתרה בטיוטה מיד!
    $currentPaid = floatval($draft['amount_paid'] ?? 0);
    $newPaid = $currentPaid + $amount;
    $totalPrice = floatval($draft['total_price']);
    $newStatus = ($newPaid >= $totalPrice) ? 'paid' : 'partial';
    
    $stmt = $pdo->prepare("UPDATE drafts SET amount_paid = ?, status = ?, payment_method = 'bank_transfer', payment_date = NOW() WHERE id = ?");
    $stmt->execute([$newPaid, $newStatus, $draftId]);
    
    // אם זו תוספת - עדכון order_additions
    if (!empty($draft['parent_order_id'])) {
        try {
            // עדכון לפי draft_id
            $pdo->prepare("UPDATE order_additions SET customer_paid = 1, customer_paid_date = NOW() WHERE draft_id = ?")
                ->execute([$draftId]);
            
            $logDir = __DIR__ . '/../logs';
            if (!is_dir($logDir)) mkdir($logDir, 0755, true);
            file_put_contents($logDir . '/bank_transfers_' . date('Y-m-d') . '.log', 
                date('Y-m-d H:i:s') . " - Updated order_additions customer_paid for draft #{$draftId}\n", FILE_APPEND);
        } catch (Exception $e) {
            // לוג שגיאה
            $logDir = __DIR__ . '/../logs';
            if (!is_dir($logDir)) mkdir($logDir, 0755, true);
            file_put_contents($logDir . '/bank_transfers_' . date('Y-m-d') . '.log', 
                date('Y-m-d H:i:s') . " - order_additions update error: " . $e->getMessage() . "\n", FILE_APPEND);
        }
    }
    
    // הוספה לטבלת draft_payments עם bank_transfer_id
    $insertSuccess = false;
    $insertError = '';
    
    // נסיון 1: עם bank_transfer_id
    try {
        $pdo->prepare("INSERT INTO draft_payments (draft_id, payer_name, payer_email, amount, payment_method, transaction_id, bank_transfer_id, payment_date) VALUES (?, ?, ?, ?, 'bank_transfer', ?, ?, NOW())")
            ->execute([$draftId, $transferrerName, $transferrerEmail, $amount, 'BANK_' . $transferId, $transferId]);
        $insertSuccess = true;
    } catch (Exception $e) {
        $insertError = $e->getMessage();
        // נסיון 2: בלי bank_transfer_id (אם העמודה לא קיימת)
        try {
            $pdo->prepare("INSERT INTO draft_payments (draft_id, payer_name, payer_email, amount, payment_method, transaction_id, payment_date) VALUES (?, ?, ?, ?, 'bank_transfer', ?, NOW())")
                ->execute([$draftId, $transferrerName, $transferrerEmail, $amount, 'BANK_' . $transferId]);
            $insertSuccess = true;
        } catch (Exception $e2) {
            $insertError = $e2->getMessage();
        }
    }
    
    // נסיון 2: בלי payment_method
    if (!$insertSuccess) {
        try {
            $pdo->prepare("INSERT INTO draft_payments (draft_id, payer_name, payer_email, amount, transaction_id, payment_date) VALUES (?, ?, ?, ?, ?, NOW())")
                ->execute([$draftId, $transferrerName, $transferrerEmail, $amount, 'BANK_' . $transferId]);
            $insertSuccess = true;
        } catch (Exception $e) {
            $insertError = $e->getMessage();
        }
    }
    
    // נסיון 3: בלי payer_email
    if (!$insertSuccess) {
        try {
            $pdo->prepare("INSERT INTO draft_payments (draft_id, payer_name, amount, transaction_id, payment_date) VALUES (?, ?, ?, ?, NOW())")
                ->execute([$draftId, $transferrerName, $amount, 'BANK_' . $transferId]);
            $insertSuccess = true;
        } catch (Exception $e) {
            $insertError = $e->getMessage();
        }
    }
    
    // לוג
    $logDir = __DIR__ . '/../logs';
    if (!is_dir($logDir)) mkdir($logDir, 0755, true);
    if ($insertSuccess) {
        file_put_contents($logDir . '/bank_transfers_' . date('Y-m-d') . '.log', 
            date('Y-m-d H:i:s') . " - draft_payments INSERT success for draft #{$draftId}\n", FILE_APPEND);
    } else {
        file_put_contents($logDir . '/bank_transfers_' . date('Y-m-d') . '.log', 
            date('Y-m-d H:i:s') . " - draft_payments INSERT FAILED: {$insertError}\n", FILE_APPEND);
    }
    
    // הפקת קבלה מ-Invoice4U
    $receiptUrl = null;
    $receiptNumber = null;
    try {
        require_once '../includes/invoice4u.php';
        $invoice4u = new Invoice4U();
        
        // תיאור לקבלה
        if (!empty($draft['parent_order_id']) && !empty($draft['addon_description'])) {
            $receiptDescription = $draft['addon_description'];
        } else {
            $receiptDescription = 'Holiday Package - ' . $draft['destination'];
        }
        
        $receiptResult = $invoice4u->createReceipt([
            'customer_name' => $transferrerName,
            'customer_email' => $transferrerEmail,
            'customer_phone' => $transferrerPhone,
            'amount' => $amount,
            'description' => $receiptDescription,
            'payment_type' => 'transfer',
            'reference_number' => $referenceNumber ?: ('BANK_' . $transferId)
        ]);
        
        if ($receiptResult['success']) {
            $receiptUrl = $receiptResult['pdf_link'] ?? null;
            $receiptNumber = $receiptResult['document_number'] ?? null;
            
            // עדכון הטיוטה עם פרטי הקבלה
            $pdo->prepare("UPDATE drafts SET receipt_number = ?, receipt_url = ? WHERE id = ?")
                ->execute([$receiptNumber, $receiptUrl, $draftId]);
            
            // עדכון draft_payments עם פרטי הקבלה
            try {
                $pdo->prepare("UPDATE draft_payments SET receipt_number = ?, receipt_pdf = ? WHERE draft_id = ? AND amount = ? ORDER BY payment_date DESC LIMIT 1")
                    ->execute([$receiptNumber, $receiptUrl, $draftId, $amount]);
            } catch (Exception $e) {
                // אם העמודות לא קיימות, ננסה להוסיף אותן
                try {
                    $pdo->exec("ALTER TABLE draft_payments ADD COLUMN receipt_number VARCHAR(50) DEFAULT NULL");
                    $pdo->exec("ALTER TABLE draft_payments ADD COLUMN receipt_pdf VARCHAR(500) DEFAULT NULL");
                    $pdo->prepare("UPDATE draft_payments SET receipt_number = ?, receipt_pdf = ? WHERE draft_id = ? AND amount = ? ORDER BY payment_date DESC LIMIT 1")
                        ->execute([$receiptNumber, $receiptUrl, $draftId, $amount]);
                } catch (Exception $e2) {
                    // ignore
                }
            }
        }
    } catch (Exception $e) {
        // לוג שגיאה אבל לא נכשל
        file_put_contents($logDir . '/bank_transfers_' . date('Y-m-d') . '.log', 
            date('Y-m-d H:i:s') . " - Invoice4U error: " . $e->getMessage() . "\n", FILE_APPEND);
    }
    
    // לוג
    $logDir = __DIR__ . '/../logs';
    if (!is_dir($logDir)) mkdir($logDir, 0755, true);
    $logEntry = date('Y-m-d H:i:s') . " - Bank transfer #{$transferId} submitted & credited: {$transferrerName}, ₪{$amount}, draft #{$draftId}\n";
    file_put_contents($logDir . '/bank_transfers_' . date('Y-m-d') . '.log', $logEntry, FILE_APPEND);
    
    $remaining = $totalPrice - $newPaid;
    
    echo json_encode([
        'success' => true,
        'transfer_id' => $transferId,
        'new_paid' => $newPaid,
        'remaining' => $remaining,
        'message' => $remaining <= 0 
            ? 'תודה! ההעברה התקבלה והתשלום הושלם!' 
            : 'תודה! ההעברה התקבלה. נותר לתשלום: ₪' . number_format($remaining)
    ]);
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
