<?php
/**
 * סקריפט גיבוי מלא לפרויקט חופשה במבצע
 * מייצר ZIP עם כל הקבצים + מבנה הטבלאות
 * 
 * שימוש: פשוט פתח את הקובץ בדפדפן והורד
 */

require_once 'includes/config.php';

// הגדרות
$projectDir = __DIR__;
$zipFilename = 'project_backup_' . date('Y-m-d_H-i-s') . '.zip';
$zipPath = sys_get_temp_dir() . '/' . $zipFilename;

// יצירת ZIP
$zip = new ZipArchive();
if ($zip->open($zipPath, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== TRUE) {
    die("לא ניתן ליצור קובץ ZIP");
}

// ========== חלק 1: מבנה הטבלאות ==========
$pdo = getDB();
$dbStructure = "-- מבנה טבלאות - " . date('Y-m-d H:i:s') . "\n";
$dbStructure .= "-- ============================================\n\n";

// רשימת הטבלאות החשובות
$tables = ['orders', 'drafts', 'draft_payments', 'bank_transfers', 'order_additions', 'users', 'settings'];

foreach ($tables as $table) {
    try {
        // מבנה הטבלה
        $stmt = $pdo->query("SHOW CREATE TABLE `$table`");
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($row) {
            $dbStructure .= "-- טבלה: $table\n";
            $dbStructure .= $row['Create Table'] . ";\n\n";
        }
    } catch (Exception $e) {
        $dbStructure .= "-- טבלה $table לא קיימת\n\n";
    }
}

// הוספת מבנה הטבלאות ל-ZIP
$zip->addFromString('_DATABASE_STRUCTURE.sql', $dbStructure);

// ========== חלק 2: קבצי הפרויקט ==========

// תיקיות לכלול
$includeDirs = ['', 'api', 'includes', 'css', 'assets', 'migrations'];

// קבצים/תיקיות לדלג עליהם
$excludePatterns = [
    'logs',
    'uploads', 
    '*.log',
    '*.zip',
    'backup.php',
    'project_backup_*.zip',
    '.git',
    'node_modules'
];

function shouldExclude($path, $excludePatterns) {
    $basename = basename($path);
    foreach ($excludePatterns as $pattern) {
        if (fnmatch($pattern, $basename)) {
            return true;
        }
    }
    return false;
}

function addDirectoryToZip($zip, $dir, $baseDir, $excludePatterns) {
    $files = scandir($dir);
    foreach ($files as $file) {
        if ($file === '.' || $file === '..') continue;
        
        $fullPath = $dir . '/' . $file;
        $relativePath = str_replace($baseDir . '/', '', $fullPath);
        
        if (shouldExclude($fullPath, $excludePatterns)) continue;
        
        if (is_dir($fullPath)) {
            // רק תיקיות ספציפיות
            if (in_array($file, ['api', 'includes', 'css', 'assets', 'migrations'])) {
                addDirectoryToZip($zip, $fullPath, $baseDir, $excludePatterns);
            }
        } else {
            // רק קבצי PHP, CSS, SQL, MD
            $ext = pathinfo($file, PATHINFO_EXTENSION);
            if (in_array($ext, ['php', 'css', 'sql', 'md', 'js', 'html', 'png', 'jpg', 'gif'])) {
                $zip->addFile($fullPath, $relativePath);
            }
        }
    }
}

// הוספת קבצים מהתיקייה הראשית
$files = glob($projectDir . '/*.php');
foreach ($files as $file) {
    $basename = basename($file);
    if (!shouldExclude($file, $excludePatterns)) {
        $zip->addFile($file, $basename);
    }
}

// הוספת קבצים נוספים
if (file_exists($projectDir . '/README.md')) {
    $zip->addFile($projectDir . '/README.md', 'README.md');
}
if (file_exists($projectDir . '/database.sql')) {
    $zip->addFile($projectDir . '/database.sql', 'database.sql');
}

// הוספת תיקיות משנה
foreach (['api', 'includes', 'css', 'assets', 'migrations'] as $subdir) {
    $subdirPath = $projectDir . '/' . $subdir;
    if (is_dir($subdirPath)) {
        addDirectoryToZip($zip, $subdirPath, $projectDir, $excludePatterns);
    }
}

// ========== חלק 3: קובץ סיכום ==========
$summary = "# 📦 גיבוי פרויקט חופשה במבצע
תאריך: " . date('Y-m-d H:i:s') . "

## 📁 תוכן הגיבוי:
- כל קבצי ה-PHP
- תיקיות: api, includes, css, assets, migrations
- מבנה הטבלאות: _DATABASE_STRUCTURE.sql

## 🔧 שימוש בצ'אט חדש עם Claude:
1. העלה את קובץ ה-ZIP הזה
2. כתוב: 'הנה גיבוי הפרויקט המלא, תחלץ אותו ונמשיך לעבוד'
3. Claude יחלץ את הקבצים ויהיה לו ראייה מלאה

## ⚠️ חשוב:
- הקובץ _DATABASE_STRUCTURE.sql מכיל את מבנה הטבלאות
- אין נתונים אמיתיים - רק מבנה
";

$zip->addFromString('_README.md', $summary);

// סגירת ה-ZIP
$zip->close();

// הורדת הקובץ
header('Content-Type: application/zip');
header('Content-Disposition: attachment; filename="' . $zipFilename . '"');
header('Content-Length: ' . filesize($zipPath));
header('Cache-Control: no-cache, must-revalidate');
header('Pragma: no-cache');

readfile($zipPath);

// מחיקת הקובץ הזמני
unlink($zipPath);
exit;
?>
