<?php
// דוח העברות בנקאיות לאימות - מאוחד
require_once 'includes/auth.php';
requireLogin();
blockAccountant();

$pdo = getDB();
$agent = getCurrentAgent();

// רק מנהל יכול לגשת להעברות בנקאיות
if ($agent['role'] !== 'admin') {
    header('Location: orders_panel.php');
    exit;
}

// פילטר נוכחי
$currentFilter = $_GET['filter'] ?? 'pending';

// טיפול בפעולות
$message = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    $action = $_POST['action'];
    $transferId = (int)($_POST['transfer_id'] ?? 0);
    
    if ($transferId) {
        if ($action === 'verify') {
            $stmt = $pdo->prepare("UPDATE bank_transfers SET status = 'approved', reviewed_by = ?, reviewed_at = NOW() WHERE id = ?");
            $stmt->execute([$agent['id'], $transferId]);
            header("Location: bank_transfers.php?filter=verified&date_from=" . ($_GET['date_from'] ?? date('Y-m-01')) . "&date_to=" . ($_GET['date_to'] ?? date('Y-m-d')) . "&msg=verified");
            exit;
            
        } elseif ($action === 'unverify') {
            $stmt = $pdo->prepare("UPDATE bank_transfers SET status = 'pending', reviewed_by = NULL, reviewed_at = NULL WHERE id = ?");
            $stmt->execute([$transferId]);
            header("Location: bank_transfers.php?filter=pending&date_from=" . ($_GET['date_from'] ?? date('Y-m-01')) . "&date_to=" . ($_GET['date_to'] ?? date('Y-m-d')) . "&msg=unverified");
            exit;
            
        } elseif ($action === 'not_received') {
            $reason = $_POST['rejection_reason'] ?? '';
            $stmt = $pdo->prepare("UPDATE bank_transfers SET status = 'rejected', reviewed_by = ?, reviewed_at = NOW(), rejection_reason = ? WHERE id = ?");
            $stmt->execute([$agent['id'], $reason, $transferId]);
            header("Location: bank_transfers.php?filter=not_received&date_from=" . ($_GET['date_from'] ?? date('Y-m-01')) . "&date_to=" . ($_GET['date_to'] ?? date('Y-m-d')) . "&msg=not_received");
            exit;
        }
    }
}

// הודעות מ-redirect
if (isset($_GET['msg'])) {
    switch ($_GET['msg']) {
        case 'verified':
            $message = '<div class="alert success">✅ אומת! ההעברה סומנה כ"התקבל בבנק"</div>';
            break;
        case 'unverified':
            $message = '<div class="alert warning">↩️ האימות בוטל</div>';
            break;
        case 'not_received':
            $message = '<div class="alert warning">⚠️ סומן כ"לא התקבל"</div>';
            break;
    }
}

// פילטר תאריכים
$dateFrom = $_GET['date_from'] ?? date('Y-m-01');
$dateTo = $_GET['date_to'] ?? date('Y-m-d');

// שליפת העברות מטבלת bank_transfers
$sql = "SELECT bt.*, 
               d.customer_name as draft_customer, 
               d.destination, 
               d.total_price, 
               d.amount_paid,
               d.token as draft_token,
               d.order_id as draft_order_id,
               d.addon_description,
               bt.order_id,
               a.full_name as reviewer_name,
               ag.full_name as agent_name
        FROM bank_transfers bt
        LEFT JOIN drafts d ON bt.draft_id = d.id
        LEFT JOIN agents a ON bt.reviewed_by = a.id
        LEFT JOIN agents ag ON d.agent_id = ag.id
        WHERE DATE(bt.created_at) BETWEEN ? AND ?
        ORDER BY bt.created_at DESC";

$stmt = $pdo->prepare($sql);
$stmt->execute([$dateFrom, $dateTo]);
$transfers = $stmt->fetchAll(PDO::FETCH_ASSOC);

// ספירה לפי סטטוס
$counts = [
    'pending' => 0,
    'approved' => 0,
    'rejected' => 0
];
foreach ($transfers as $t) {
    $status = $t['status'];
    if (empty($status) || $status === 'pending') {
        $counts['pending']++;
    } elseif ($status === 'approved') {
        $counts['approved']++;
    } elseif ($status === 'rejected') {
        $counts['rejected']++;
    } else {
        $counts['pending']++;
    }
}

$totalTransfers = count($transfers);
$pendingCount = $counts['pending'];
$verifiedCount = $counts['approved'];
$notReceivedCount = $counts['rejected'];

// סכומים
$totalPending = 0;
$totalVerified = 0;
foreach ($transfers as $t) {
    $status = $t['status'];
    $isPending = (empty($status) || $status === 'pending');
    if ($isPending) {
        $totalPending += (float)$t['amount'];
    } elseif ($status === 'approved') {
        $totalVerified += (float)$t['amount'];
    }
}
?>
<!DOCTYPE html>
<html lang="he" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>העברות בנקאיות | CRM</title>
    <style>
        * { box-sizing: border-box; margin: 0; padding: 0; }
        body {
            font-family: 'Segoe UI', Tahoma, Arial, sans-serif;
            background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
            min-height: 100vh;
            padding: 20px;
        }
        .container { max-width: 1400px; margin: 0 auto; }
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }
        h1 { color: white; }
        .back-btn {
            background: #6c757d;
            color: white;
            padding: 10px 20px;
            text-decoration: none;
            border-radius: 8px;
        }
        .back-btn:hover { background: #5a6268; }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(180px, 1fr));
            gap: 15px;
            margin-bottom: 20px;
        }
        .stat-card {
            background: white;
            padding: 20px;
            border-radius: 10px;
            text-align: center;
        }
        .stat-value { font-size: 1.8rem; font-weight: bold; direction: ltr; display: inline-block; }
        .stat-label { color: #666; font-size: 0.85rem; }
        .stat-pending .stat-value { color: #dc3545; }
        .stat-verified .stat-value { color: #28a745; }
        .stat-warning .stat-value { color: #ffc107; }
        
        .content { background: white; border-radius: 10px; padding: 20px; }
        
        .date-filter-form {
            background: #f8f9fa;
            padding: 15px;
            border-radius: 10px;
            margin-bottom: 15px;
        }
        .date-filter-row {
            display: flex;
            gap: 10px;
            align-items: center;
            flex-wrap: wrap;
        }
        .date-filter-row label {
            font-weight: 500;
            color: #555;
        }
        .date-filter-row input[type="date"] {
            padding: 8px 12px;
            border: 1px solid #ddd;
            border-radius: 6px;
            font-size: 0.9rem;
        }
        .btn-filter {
            background: #007bff;
            color: white;
            border: none;
            padding: 8px 20px;
            border-radius: 6px;
            cursor: pointer;
            font-size: 0.9rem;
        }
        .btn-filter:hover { background: #0056b3; }
        .btn-reset {
            background: #6c757d;
            color: white;
            text-decoration: none;
            padding: 8px 15px;
            border-radius: 6px;
            font-size: 0.85rem;
        }
        .btn-reset:hover { background: #5a6268; }
        
        .transfer-description {
            background: #e3f2fd;
            padding: 4px 8px;
            border-radius: 4px;
            color: #1565c0;
            font-weight: 500;
        }
        
        .filter-bar {
            display: flex;
            gap: 10px;
            margin-bottom: 20px;
            flex-wrap: wrap;
        }
        .filter-btn {
            padding: 10px 20px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            background: #e9ecef;
            text-decoration: none;
            color: #333;
            font-size: 0.9rem;
            transition: all 0.2s;
        }
        .filter-btn:hover { background: #dee2e6; }
        .filter-btn.active { background: #007bff; color: white; }
        .filter-btn.active.pending { background: #dc3545; }
        .filter-btn.active.verified { background: #28a745; }
        
        .alert {
            padding: 15px 20px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        .alert.success { background: #d4edda; color: #155724; }
        .alert.warning { background: #fff3cd; color: #856404; }
        
        .transfer-card {
            border: 1px solid #e0e0e0;
            border-radius: 12px;
            margin-bottom: 15px;
            overflow: hidden;
            transition: box-shadow 0.2s;
        }
        .transfer-card:hover { box-shadow: 0 4px 15px rgba(0,0,0,0.1); }
        
        .transfer-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 15px 20px;
            background: linear-gradient(135deg, #f8f9fa, #e9ecef);
        }
        .transfer-amount {
            font-size: 1.4rem;
            font-weight: bold;
            color: #28a745;
        }
        .transfer-date { color: #666; font-size: 0.9rem; }
        .transfer-status {
            padding: 5px 12px;
            border-radius: 20px;
            font-size: 0.85rem;
        }
        .status-pending { background: #fff3cd; color: #856404; }
        .status-verified { background: #d4edda; color: #155724; }
        .status-not-received { background: #f8d7da; color: #721c24; }
        
        .transfer-body {
            display: grid;
            grid-template-columns: 1fr 250px;
            gap: 20px;
            padding: 20px;
        }
        @media (max-width: 768px) {
            .transfer-body { grid-template-columns: 1fr; }
        }
        
        .transfer-details { }
        .transfer-row {
            display: flex;
            padding: 8px 0;
            border-bottom: 1px solid #f0f0f0;
        }
        .transfer-row:last-child { border-bottom: none; }
        .transfer-label {
            width: 140px;
            color: #666;
            font-size: 0.9rem;
        }
        .transfer-value {
            flex: 1;
            font-weight: 500;
        }
        
        .screenshot-container {
            background: #f8f9fa;
            border-radius: 8px;
            padding: 10px;
            text-align: center;
        }
        .screenshot-container img {
            max-width: 100%;
            max-height: 200px;
            border-radius: 6px;
            cursor: pointer;
            transition: transform 0.2s;
        }
        .screenshot-container img:hover { transform: scale(1.02); }
        .screenshot-container small { display: block; margin-top: 8px; color: #666; }
        
        .transfer-actions {
            padding: 15px 20px;
            background: #f8f9fa;
            border-top: 1px solid #e0e0e0;
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
        }
        
        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 0.9rem;
            transition: all 0.2s;
        }
        .btn-approve { background: #28a745; color: white; }
        .btn-approve:hover { background: #218838; }
        .btn-reject { background: #dc3545; color: white; }
        .btn-reject:hover { background: #c82333; }
        .btn-cancel { background: #6c757d; color: white; }
        .btn-cancel:hover { background: #5a6268; }
        
        .order-link {
            color: #007bff;
            text-decoration: none;
            font-weight: 600;
        }
        .order-link:hover { text-decoration: underline; }
        
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: #666;
        }
        .empty-state h2 { margin-bottom: 10px; }
        
        /* Modal לתמונה */
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.9);
            z-index: 1000;
            justify-content: center;
            align-items: center;
        }
        .modal.active { display: flex; }
        .modal img {
            max-width: 90%;
            max-height: 90%;
            border-radius: 8px;
        }
        .modal-close {
            position: absolute;
            top: 20px;
            right: 30px;
            color: white;
            font-size: 40px;
            cursor: pointer;
        }
        
        /* Modal לדחייה */
        .reject-modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.5);
            z-index: 1000;
            justify-content: center;
            align-items: center;
        }
        .reject-modal.active { display: flex; }
        .reject-modal-content {
            background: white;
            padding: 30px;
            border-radius: 12px;
            max-width: 400px;
            width: 90%;
        }
        .reject-modal-content h3 { margin-bottom: 15px; }
        .reject-modal-content textarea {
            width: 100%;
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 6px;
            margin-bottom: 15px;
            resize: vertical;
        }
        .reject-modal-buttons { display: flex; gap: 10px; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🏦 העברות בנקאיות לאימות</h1>
            <a href="reports.php" class="back-btn">← חזרה לדוחות</a>
        </div>

        <?= $message ?>

        <div class="stats-grid">
            <div class="stat-card stat-pending">
                <div class="stat-value"><?= $pendingCount ?></div>
                <div class="stat-label">ממתינות לאימות</div>
            </div>
            <div class="stat-card stat-pending">
                <div class="stat-value">₪<?= number_format($totalPending) ?></div>
                <div class="stat-label">סכום ממתין</div>
            </div>
            <div class="stat-card stat-verified">
                <div class="stat-value"><?= $verifiedCount ?></div>
                <div class="stat-label">אומתו</div>
            </div>
            <div class="stat-card stat-verified">
                <div class="stat-value">₪<?= number_format($totalVerified) ?></div>
                <div class="stat-label">סכום שאומת</div>
            </div>
            <?php if ($notReceivedCount > 0): ?>
            <div class="stat-card stat-warning">
                <div class="stat-value"><?= $notReceivedCount ?></div>
                <div class="stat-label">לא התקבלו</div>
            </div>
            <?php endif; ?>
        </div>

        <div class="content">
            <!-- פילטר תאריכים -->
            <form method="GET" class="date-filter-form">
                <input type="hidden" name="filter" value="<?= htmlspecialchars($currentFilter) ?>">
                <div class="date-filter-row">
                    <label>מתאריך:</label>
                    <input type="date" name="date_from" value="<?= $dateFrom ?>">
                    <label>עד תאריך:</label>
                    <input type="date" name="date_to" value="<?= $dateTo ?>">
                    <button type="submit" class="btn btn-filter">🔍 סנן</button>
                    <a href="?filter=<?= $currentFilter ?>&date_from=<?= date('Y-m-01') ?>&date_to=<?= date('Y-m-d') ?>" class="btn btn-reset">החודש</a>
                    <a href="?filter=<?= $currentFilter ?>&date_from=<?= date('Y-01-01') ?>&date_to=<?= date('Y-m-d') ?>" class="btn btn-reset">השנה</a>
                </div>
            </form>
            
            <div class="filter-bar">
                <a href="?filter=pending&date_from=<?= $dateFrom ?>&date_to=<?= $dateTo ?>" class="filter-btn <?= $currentFilter === 'pending' ? 'active pending' : '' ?>">
                    ⏳ ממתינות (<?= $pendingCount ?>)
                </a>
                <a href="?filter=verified&date_from=<?= $dateFrom ?>&date_to=<?= $dateTo ?>" class="filter-btn <?= $currentFilter === 'verified' ? 'active verified' : '' ?>">
                    ✅ אומתו (<?= $verifiedCount ?>)
                </a>
                <?php if ($notReceivedCount > 0): ?>
                <a href="?filter=not_received&date_from=<?= $dateFrom ?>&date_to=<?= $dateTo ?>" class="filter-btn <?= $currentFilter === 'not_received' ? 'active' : '' ?>">
                    ⚠️ לא התקבלו (<?= $notReceivedCount ?>)
                </a>
                <?php endif; ?>
                <a href="?filter=all&date_from=<?= $dateFrom ?>&date_to=<?= $dateTo ?>" class="filter-btn <?= $currentFilter === 'all' ? 'active' : '' ?>">
                    📋 הכל (<?= $totalTransfers ?>)
                </a>
            </div>

            <?php 
            $filteredTransfers = array_filter($transfers, function($t) use ($currentFilter) {
                $status = $t['status'];
                $isPending = (empty($status) || $status === 'pending');
                
                if ($currentFilter === 'all') return true;
                if ($currentFilter === 'pending') return $isPending;
                if ($currentFilter === 'verified') return $status === 'approved';
                if ($currentFilter === 'not_received') return $status === 'rejected';
                return true;
            });
            ?>

            <?php if (empty($filteredTransfers)): ?>
                <div class="empty-state">
                    <?php if ($currentFilter === 'pending'): ?>
                        <h2>🎉 אין העברות ממתינות</h2>
                        <p>כל ההעברות אומתו!</p>
                    <?php elseif ($currentFilter === 'verified'): ?>
                        <h2>📭 אין העברות מאומתות</h2>
                        <p>עדיין לא אימתת העברות</p>
                    <?php else: ?>
                        <h2>📭 אין העברות בנקאיות</h2>
                        <p>כל התשלומים בוצעו באשראי או במזומן</p>
                    <?php endif; ?>
                </div>
            <?php else: ?>
                <?php foreach ($filteredTransfers as $t): 
                    $status = $t['status'];
                    $isPending = (empty($status) || $status === 'pending');
                    
                    if ($isPending) {
                        $statusClass = 'status-pending';
                        $statusText = '⏳ ממתין לאימות';
                    } elseif ($status === 'approved') {
                        $statusClass = 'status-verified';
                        $statusText = '✅ אומת';
                    } elseif ($status === 'rejected') {
                        $statusClass = 'status-not-received';
                        $statusText = '⚠️ לא התקבל';
                    } else {
                        $statusClass = 'status-pending';
                        $statusText = '⏳ ממתין לאימות';
                    }
                ?>
                <div class="transfer-card">
                    <div class="transfer-header">
                        <div>
                            <span class="transfer-amount">₪<?= number_format($t['amount']) ?></span>
                            <span class="transfer-date">
                                <?= date('d/m/Y H:i', strtotime($t['created_at'])) ?>
                            </span>
                            <?php if (empty($t['draft_id'])): ?>
                            <span style="background:#e3f2fd; color:#1565c0; padding:2px 8px; border-radius:4px; font-size:0.75rem; margin-right:5px;">📝 ידני</span>
                            <?php endif; ?>
                        </div>
                        <span class="transfer-status <?= $statusClass ?>"><?= $statusText ?></span>
                    </div>
                    
                    <div class="transfer-body">
                        <div class="transfer-details">
                            <div class="transfer-row">
                                <span class="transfer-label">שם המעביר:</span>
                                <span class="transfer-value"><?= htmlspecialchars($t['transferrer_name'] ?? $t['payer_name'] ?? '-') ?></span>
                            </div>
                            <?php if (!empty($t['transferrer_phone'])): ?>
                            <div class="transfer-row">
                                <span class="transfer-label">טלפון:</span>
                                <span class="transfer-value">
                                    <a href="tel:<?= $t['transferrer_phone'] ?>"><?= htmlspecialchars($t['transferrer_phone']) ?></a>
                                </span>
                            </div>
                            <?php endif; ?>
                            <?php if (!empty($t['transferrer_email'])): ?>
                            <div class="transfer-row">
                                <span class="transfer-label">אימייל:</span>
                                <span class="transfer-value"><?= htmlspecialchars($t['transferrer_email']) ?></span>
                            </div>
                            <?php endif; ?>
                            <div class="transfer-row">
                                <span class="transfer-label">תאריך העברה:</span>
                                <span class="transfer-value"><?= $t['transfer_date'] ? date('d/m/Y', strtotime($t['transfer_date'])) : '-' ?></span>
                            </div>
                            <?php if (!empty($t['reference_number'])): ?>
                            <div class="transfer-row">
                                <span class="transfer-label">מס' אסמכתא:</span>
                                <span class="transfer-value" style="font-family: monospace; font-weight: bold;"><?= htmlspecialchars($t['reference_number']) ?></span>
                            </div>
                            <?php endif; ?>
                            <?php if (!empty($t['destination'])): ?>
                            <div class="transfer-row">
                                <span class="transfer-label">יעד:</span>
                                <span class="transfer-value"><?= htmlspecialchars($t['destination']) ?></span>
                            </div>
                            <?php endif; ?>
                            <?php 
                            // תיאור ההעברה - מה שנכתב בדראפט או בהערות
                            $transferDescription = $t['addon_description'] ?? $t['notes'] ?? '';
                            if (!empty($transferDescription)): 
                            ?>
                            <div class="transfer-row">
                                <span class="transfer-label">תיאור:</span>
                                <span class="transfer-value transfer-description"><?= htmlspecialchars($transferDescription) ?></span>
                            </div>
                            <?php endif; ?>
                            <?php 
                            // שם לקוח - מהטיוטה או מהתשלום הידני
                            $customerName = $t['draft_customer'] ?? $t['payer_name'] ?? '';
                            if (!empty($customerName)): 
                            ?>
                            <div class="transfer-row">
                                <span class="transfer-label">לקוח:</span>
                                <span class="transfer-value"><?= htmlspecialchars($customerName) ?></span>
                            </div>
                            <?php endif; ?>
                            <?php if (!empty($t['total_price'])): ?>
                            <div class="transfer-row">
                                <span class="transfer-label">סה"כ להזמנה:</span>
                                <span class="transfer-value">₪<?= number_format($t['total_price']) ?></span>
                            </div>
                            <?php endif; ?>
                            <?php if (!empty($t['notes'])): ?>
                            <div class="transfer-row">
                                <span class="transfer-label">הערות:</span>
                                <span class="transfer-value"><?= htmlspecialchars($t['notes']) ?></span>
                            </div>
                            <?php endif; ?>
                            <?php if (!empty($t['agent_name'])): ?>
                            <div class="transfer-row">
                                <span class="transfer-label">נציג:</span>
                                <span class="transfer-value"><?= htmlspecialchars($t['agent_name']) ?></span>
                            </div>
                            <?php endif; ?>
                            <?php if ($t['status'] === 'verified' && !empty($t['reviewer_name'])): ?>
                            <div class="transfer-row">
                                <span class="transfer-label">אומת ע"י:</span>
                                <span class="transfer-value"><?= htmlspecialchars($t['reviewer_name']) ?> (<?= date('d/m H:i', strtotime($t['reviewed_at'])) ?>)</span>
                            </div>
                            <?php endif; ?>
                            <?php if (!empty($t['rejection_reason'])): ?>
                            <div class="transfer-row">
                                <span class="transfer-label">סיבת דחייה:</span>
                                <span class="transfer-value" style="color:#dc3545;"><?= htmlspecialchars($t['rejection_reason']) ?></span>
                            </div>
                            <?php endif; ?>
                            <?php if (!empty($t['draft_token'])): ?>
                            <div class="transfer-row">
                                <span class="transfer-label">לינק תשלום:</span>
                                <span class="transfer-value">
                                    <a href="pay.php?token=<?= $t['draft_token'] ?>" target="_blank" class="order-link">פתח טופס</a>
                                </span>
                            </div>
                            <?php endif; ?>
                            <?php 
                            // קישור להזמנה - מנסה קודם מהטבלה, אחכ מהטיוטה
                            $orderId = $t['order_id'] ?? $t['draft_order_id'] ?? null;
                            if (!empty($orderId)): 
                            ?>
                            <div class="transfer-row">
                                <span class="transfer-label">🎫 הזמנה:</span>
                                <span class="transfer-value">
                                    <a href="view_order_new.php?id=<?= $orderId ?>" target="_blank" class="order-link" style="color:#28a745; font-weight:bold;">
                                        צפה בהזמנה #<?= $orderId ?>
                                    </a>
                                </span>
                            </div>
                            <?php endif; ?>
                        </div>
                        
                        <div class="screenshot-container">
                            <?php if (!empty($t['screenshot_path'])): ?>
                                <img src="<?= htmlspecialchars($t['screenshot_path']) ?>" alt="צילום אסמכתא" onclick="showImage(this.src)">
                                <small>📸 לחץ להגדלה</small>
                            <?php else: ?>
                                <p style="color:#999; padding: 40px 0;">אין צילום מסך</p>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <div class="transfer-actions">
                        <?php if ($isPending): ?>
                            <form method="POST" style="display:inline;">
                                <input type="hidden" name="transfer_id" value="<?= $t['id'] ?>">
                                <input type="hidden" name="action" value="verify">
                                <button type="submit" class="btn btn-approve" onclick="return confirm('לאשר שההעברה של ₪<?= number_format($t['amount']) ?> התקבלה בבנק?')">
                                    ✅ אמת - הגיע לבנק
                                </button>
                            </form>
                            <button type="button" class="btn btn-reject" onclick="showRejectModal(<?= $t['id'] ?>)">
                                ❌ לא התקבל
                            </button>
                        <?php elseif ($status === 'approved'): ?>
                            <form method="POST" style="display:inline;">
                                <input type="hidden" name="transfer_id" value="<?= $t['id'] ?>">
                                <input type="hidden" name="action" value="unverify">
                                <button type="submit" class="btn btn-cancel" onclick="return confirm('לבטל את האימות?')">
                                    ↩️ בטל אימות
                                </button>
                            </form>
                        <?php elseif ($status === 'rejected'): ?>
                            <form method="POST" style="display:inline;">
                                <input type="hidden" name="transfer_id" value="<?= $t['id'] ?>">
                                <input type="hidden" name="action" value="verify">
                                <button type="submit" class="btn btn-approve">
                                    ✅ בעצם כן הגיע
                                </button>
                            </form>
                        <?php endif; ?>
                    </div>
                </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
    </div>

    <!-- Modal לתמונה מוגדלת -->
    <div class="modal" id="imageModal" onclick="hideImage()">
        <span class="modal-close">&times;</span>
        <img id="modalImage" src="" alt="">
    </div>

    <!-- Modal לדחייה -->
    <div class="reject-modal" id="rejectModal">
        <div class="reject-modal-content">
            <h3>⚠️ סימון כלא התקבל</h3>
            <form method="POST" id="rejectForm">
                <input type="hidden" name="transfer_id" id="rejectTransferId">
                <input type="hidden" name="action" value="not_received">
                <textarea name="rejection_reason" rows="3" placeholder="סיבה (אופציונלי)..."></textarea>
                <div class="reject-modal-buttons">
                    <button type="submit" class="btn btn-reject">אשר</button>
                    <button type="button" class="btn btn-cancel" onclick="hideRejectModal()">ביטול</button>
                </div>
            </form>
        </div>
    </div>

    <script>
        function showImage(src) {
            document.getElementById('modalImage').src = src;
            document.getElementById('imageModal').classList.add('active');
        }
        
        function hideImage() {
            document.getElementById('imageModal').classList.remove('active');
        }
        
        function showRejectModal(transferId) {
            document.getElementById('rejectTransferId').value = transferId;
            document.getElementById('rejectModal').classList.add('active');
        }
        
        function hideRejectModal() {
            document.getElementById('rejectModal').classList.remove('active');
        }
        
        // סגירה עם ESC
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') {
                hideImage();
                hideRejectModal();
            }
        });
    </script>
</body>
</html>
