<?php
/**
 * Cardcom Callback - קבלת אישור תשלום + הפקת קבלה אוטומטית
 */
require_once 'includes/config.php';
require_once 'includes/invoice4u.php';

$pdo = getDB();
$pdo->exec("SET NAMES utf8mb4");

// לוג הבקשה
$logDir = __DIR__ . '/logs';
if (!is_dir($logDir)) @mkdir($logDir, 0755, true);
$logFile = $logDir . '/cardcom_' . date('Y-m-d') . '.log';

// === קבלת הנתונים ===
$data = [];
$rawInput = file_get_contents('php://input');

if (!empty($_POST)) {
    $data = $_POST;
    $source = 'POST';
} elseif (!empty($_GET)) {
    $data = $_GET;
    $source = 'GET';
} elseif ($rawInput) {
    $data = json_decode($rawInput, true) ?: [];
    $source = 'JSON';
}

@file_put_contents($logFile, date('Y-m-d H:i:s') . " - CALLBACK ($source)\n", FILE_APPEND);

if (empty($data)) {
    @file_put_contents($logFile, date('Y-m-d H:i:s') . " - ERROR: No data\n", FILE_APPEND);
    echo "No data"; exit;
}

try {
    // פרמטרים מ-Cardcom (API v11 JSON format)
    $responseCode = $data['ResponseCode'] ?? '';
    $returnValue = $data['ReturnValue'] ?? '';
    
    // קבלת פרטים מ-TranzactionInfo (המבנה החדש של Cardcom)
    $tranzInfo = $data['TranzactionInfo'] ?? [];
    $transactionId = $tranzInfo['TranzactionId'] ?? ($data['TranzactionId'] ?? '');
    $approvalNumber = $tranzInfo['ApprovalNumber'] ?? '';
    $sumCharged = floatval($tranzInfo['Amount'] ?? 0);
    $numOfPayments = intval($tranzInfo['NumberOfPayments'] ?? 1);
    $last4Digits = $tranzInfo['Last4CardDigitsString'] ?? '0000';
    $cardBrand = $tranzInfo['Brand'] ?? 'Visa';
    
    // חילוץ token, סכום, שם משלם ומייל מ-ReturnValue
    $token = '';
    $payerName = '';
    $payerEmail = '';
    if ($returnValue) {
        $parts = explode('|', $returnValue);
        $token = $parts[0] ?? '';
        // אם אין סכום מ-TranzactionInfo, קח מ-ReturnValue
        if (!$sumCharged && isset($parts[1])) {
            $sumCharged = floatval($parts[1]);
        }
    }
    
    // שליפת פרטי המשלם מהטבלה (שנשמרו לפני התשלום)
    if ($token) {
        try {
            $pendingStmt = $pdo->prepare("SELECT payer_name, payer_email FROM pending_payments WHERE token = ? ORDER BY created_at DESC LIMIT 1");
            $pendingStmt->execute([$token]);
            $pendingData = $pendingStmt->fetch();
            if ($pendingData) {
                $payerName = $pendingData['payer_name'] ?: '';
                $payerEmail = $pendingData['payer_email'] ?: '';
                @file_put_contents($logFile, date('Y-m-d H:i:s') . " - Found pending payment: Name=$payerName, Email=$payerEmail\n", FILE_APPEND);
            }
        } catch (Exception $e) {
            @file_put_contents($logFile, date('Y-m-d H:i:s') . " - pending_payments error: " . $e->getMessage() . "\n", FILE_APPEND);
        }
    }
    
    // אם אין שם, קח מקארדקום
    if (!$payerName) {
        $payerName = $tranzInfo['CardOwnerName'] ?? '';
    }
    
    @file_put_contents($logFile, date('Y-m-d H:i:s') . " - Token=$token, Code=$responseCode, Amount=$sumCharged, Approval=$approvalNumber, PayerName=$payerName, PayerEmail=$payerEmail\n", FILE_APPEND);
    
    if (!$token) {
        @file_put_contents($logFile, date('Y-m-d H:i:s') . " - ERROR: No token\n", FILE_APPEND);
        echo "No token"; exit;
    }
    
    // שליפת הטיוטה
    $stmt = $pdo->prepare("SELECT * FROM drafts WHERE token = ?");
    $stmt->execute([$token]);
    $draft = $stmt->fetch();
    
    if (!$draft) {
        @file_put_contents($logFile, date('Y-m-d H:i:s') . " - ERROR: Draft not found\n", FILE_APPEND);
        echo "Draft not found"; exit;
    }
    
    if ($responseCode === 0 || $responseCode === '0') {
        // תשלום הצליח!
        $currentPaid = floatval($draft['amount_paid'] ?? 0);
        $newPaid = $currentPaid + $sumCharged;
        $totalPrice = floatval($draft['total_price']);
        $newStatus = ($newPaid >= $totalPrice) ? 'paid' : 'partial';
        
        @file_put_contents($logFile, date('Y-m-d H:i:s') . " - Updating: current=$currentPaid + charged=$sumCharged = new=$newPaid (total=$totalPrice) -> status=$newStatus\n", FILE_APPEND);
        
        // עדכון הטיוטה
        $stmt = $pdo->prepare("
            UPDATE drafts SET 
                status = ?,
                amount_paid = ?,
                payment_method = 'cardcom',
                payment_transaction_id = ?,
                payment_date = NOW()
            WHERE token = ?
        ");
        $result = $stmt->execute([$newStatus, $newPaid, $transactionId, $token]);
        
        @file_put_contents($logFile, date('Y-m-d H:i:s') . " - Update result: " . ($result ? 'OK' : 'FAILED') . ", rows=" . $stmt->rowCount() . "\n", FILE_APPEND);
        
        // הוספה לטבלת draft_payments (רק אם לא קיים כבר)
        try {
            // וודא שעמודת payer_email קיימת
            try {
                $pdo->exec("ALTER TABLE draft_payments ADD COLUMN payer_email VARCHAR(255) AFTER payer_name");
            } catch (Exception $e) {
                // עמודה כבר קיימת - בסדר
            }
            
            // בדיקה אם כבר קיים תשלום עם אותו transaction_id
            $checkStmt = $pdo->prepare("SELECT id FROM draft_payments WHERE transaction_id = ?");
            $checkStmt->execute([$transactionId]);
            if ($checkStmt->fetch()) {
                @file_put_contents($logFile, date('Y-m-d H:i:s') . " - Payment already exists for transaction_id=$transactionId, skipping\n", FILE_APPEND);
            } else {
                $pdo->prepare("INSERT INTO draft_payments (draft_id, payer_name, payer_email, amount, payment_method, last_4_digits, card_brand, transaction_id, payment_date) VALUES (?, ?, ?, ?, 'credit_card', ?, ?, ?, NOW())")
                    ->execute([$draft['id'], $payerName ?: $draft['customer_name'], $payerEmail, $sumCharged, $last4, $cardBrand, $transactionId]);
            }
        } catch (Exception $e) {
            @file_put_contents($logFile, date('Y-m-d H:i:s') . " - draft_payments error (ignored): " . $e->getMessage() . "\n", FILE_APPEND);
        }
        
        // הזמנה תיווצר ידנית על ידי הנציגות דרך טופס complete_order.php
        // הטיוטה נשארת עם status=paid ובלי order_id עד שממלאים את הטופס המלא
        @file_put_contents($logFile, date('Y-m-d H:i:s') . " - Draft marked as $newStatus, awaiting order completion\n", FILE_APPEND);
        
        // אם זו טיוטה של תוספת - עדכון order_additions
        if (!empty($draft['parent_order_id'])) {
            try {
                $stmt = $pdo->prepare("UPDATE order_additions SET customer_paid = 1, customer_paid_date = NOW() WHERE draft_id = ?");
                $stmt->execute([$draft['id']]);
                @file_put_contents($logFile, date('Y-m-d H:i:s') . " - Updated order_additions for addition draft\n", FILE_APPEND);
            } catch (Exception $e) {
                @file_put_contents($logFile, date('Y-m-d H:i:s') . " - order_additions update error: " . $e->getMessage() . "\n", FILE_APPEND);
            }
        }
        
        // Invoice4U - הפקת קבלה
        @file_put_contents($logFile, date('Y-m-d H:i:s') . " - Creating receipt: Name=$payerName, Email=$payerEmail\n", FILE_APPEND);
        
        // תיאור לקבלה - שונה עבור תוספות
        if (!empty($draft['parent_order_id']) && !empty($draft['addon_description'])) {
            $receiptDescription = $draft['addon_description'];
        } else {
            // שליפת שם עברי של היעד
            $destParts = explode(',', $draft['destination']);
            $destName = trim($destParts[0] ?? '');
            $hebrewDest = $destName; // ברירת מחדל - השם באנגלית
            
            if ($destName) {
                $stmtDest = $pdo->prepare("SELECT name_he FROM settings_destinations WHERE name = ? AND name_he IS NOT NULL AND name_he != ''");
                $stmtDest->execute([$destName]);
                $destRow = $stmtDest->fetch();
                if ($destRow && $destRow['name_he']) {
                    $hebrewDest = $destRow['name_he'];
                }
            }
            
            // סוג ההזמנה
            $orderType = $draft['order_type'] ?? 'separate';
            if ($orderType === 'tour') {
                $receiptDescription = 'טיול מאורגן ל' . $hebrewDest;
            } else {
                $receiptDescription = 'חבילת נופש ל' . $hebrewDest;
            }
        }
        
        try {
            $invoice4u = new Invoice4U();
            $receiptResult = $invoice4u->createReceipt([
                'customer_name' => $payerName ?: $draft['customer_name'],
                'customer_email' => $payerEmail ?: ($draft['customer_email'] ?? ''),
                'customer_phone' => '', // לא לשלוח טלפון - כי המשלם לא בהכרח בעל הטלפון מהטיוטה
                'amount' => $sumCharged,
                'description' => $receiptDescription,
                'payment_type' => 'credit_card',
                'last_4_digits' => $last4Digits,
                'approval_number' => $approvalNumber,
                'installments' => $numOfPayments,
                'card_brand' => $cardBrand
            ]);
            @file_put_contents($logFile, date('Y-m-d H:i:s') . " - Invoice4U: " . ($receiptResult['success'] ? 'OK #'.$receiptResult['document_number'] : 'FAILED') . "\n", FILE_APPEND);
            
            if ($receiptResult['success']) {
                $pdo->prepare("UPDATE drafts SET receipt_number=?, receipt_url=? WHERE token=?")
                    ->execute([$receiptResult['document_number'], $receiptResult['pdf_link'], $token]);
            }
        } catch (Exception $e) {
            @file_put_contents($logFile, date('Y-m-d H:i:s') . " - Invoice4U error: " . $e->getMessage() . "\n", FILE_APPEND);
        }
        
        @file_put_contents($logFile, date('Y-m-d H:i:s') . " - SUCCESS!\n", FILE_APPEND);
        echo "OK";
    } else {
        @file_put_contents($logFile, date('Y-m-d H:i:s') . " - Payment failed with code: $responseCode\n", FILE_APPEND);
        echo "FAILED";
    }
    
} catch (Exception $e) {
    @file_put_contents($logFile, date('Y-m-d H:i:s') . " - EXCEPTION: " . $e->getMessage() . "\n", FILE_APPEND);
    echo "ERROR";
}
?>
