<?php
/**
 * יצירת טבלת order_hotels והעברת מלונות קיימים
 * מאפשר ריבוי מלונות להזמנה אחת
 */

require_once 'includes/config.php';

echo "<pre style='direction:ltr; font-family: monospace; background: #1e1e1e; color: #fff; padding: 20px; border-radius: 10px;'>";
echo "===========================================\n";
echo "   יצירת טבלת מלונות - order_hotels\n";
echo "===========================================\n\n";

try {
    $pdo = getDB();
    
    // שלב 1: יצירת הטבלה
    echo "📦 שלב 1: יצירת טבלה order_hotels...\n";
    
    $sql = "
    CREATE TABLE IF NOT EXISTS order_hotels (
        id INT PRIMARY KEY AUTO_INCREMENT,
        order_id INT NOT NULL,
        hotel_name VARCHAR(200) DEFAULT NULL COMMENT 'שם המלון',
        destination VARCHAR(100) DEFAULT NULL COMMENT 'עיר/אזור',
        check_in DATE DEFAULT NULL COMMENT 'תאריך כניסה',
        check_out DATE DEFAULT NULL COMMENT 'תאריך יציאה',
        nights INT DEFAULT NULL COMMENT 'מספר לילות',
        room_type VARCHAR(100) DEFAULT NULL COMMENT 'סוג חדר',
        meal_basis VARCHAR(50) DEFAULT NULL COMMENT 'בסיס אירוח',
        confirmation_code VARCHAR(100) DEFAULT NULL COMMENT 'קוד אישור',
        supplier ENUM('DIRECT','ORANGE','YELLOW','BLUE') DEFAULT 'DIRECT' COMMENT 'ספק מלון',
        supplier_cost DECIMAL(10,2) DEFAULT 0 COMMENT 'עלות לספק',
        supplier_currency ENUM('USD','EUR','ILS') DEFAULT 'USD' COMMENT 'מטבע',
        supplier_paid ENUM('paid','unpaid') DEFAULT 'unpaid' COMMENT 'שולם לספק?',
        voucher_path VARCHAR(500) DEFAULT NULL COMMENT 'נתיב וואוצ\'ר',
        sort_order INT DEFAULT 0 COMMENT 'סדר המלון בטיול',
        notes TEXT DEFAULT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        FOREIGN KEY (order_id) REFERENCES orders(id) ON DELETE CASCADE,
        INDEX idx_order (order_id),
        INDEX idx_supplier (supplier),
        INDEX idx_paid (supplier_paid)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ";
    
    $pdo->exec($sql);
    echo "   ✅ טבלה נוצרה בהצלחה!\n\n";
    
    // שלב 2: בדיקה אם יש מלונות קיימים להעברה
    echo "📊 שלב 2: בדיקת מלונות קיימים בטבלת orders...\n";
    
    // בדיקה אילו עמודות קיימות
    $columns = $pdo->query("SHOW COLUMNS FROM orders")->fetchAll(PDO::FETCH_COLUMN);
    $hasSupplierCols = in_array('hotel_supplier', $columns);
    
    if ($hasSupplierCols) {
        $stmt = $pdo->query("
            SELECT id, hotel_name, destination, hotel_checkin, hotel_checkout, 
                   hotel_confirmation,
                   hotel_supplier, hotel_supplier_cost, hotel_supplier_currency, hotel_supplier_paid
            FROM orders 
            WHERE hotel_name IS NOT NULL AND hotel_name != ''
        ");
    } else {
        $stmt = $pdo->query("
            SELECT id, hotel_name, destination, hotel_checkin, hotel_checkout, 
                   hotel_confirmation,
                   'DIRECT' as hotel_supplier, 0 as hotel_supplier_cost, 'USD' as hotel_supplier_currency, 'unpaid' as hotel_supplier_paid
            FROM orders 
            WHERE hotel_name IS NOT NULL AND hotel_name != ''
        ");
    }
    $existingHotels = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $count = count($existingHotels);
    echo "   📋 נמצאו $count הזמנות עם מלונות\n\n";
    
    if ($count > 0) {
        echo "🔄 שלב 3: העברת מלונות לטבלה החדשה...\n";
        
        // בדיקה שלא כבר הועברו
        $checkStmt = $pdo->query("SELECT COUNT(*) FROM order_hotels");
        $existingInNewTable = $checkStmt->fetchColumn();
        
        if ($existingInNewTable > 0) {
            echo "   ⚠️ כבר יש $existingInNewTable מלונות בטבלה החדשה - מדלג על מיגרציה\n";
        } else {
            $insertStmt = $pdo->prepare("
                INSERT INTO order_hotels 
                (order_id, hotel_name, destination, check_in, check_out, nights,
                 confirmation_code, supplier, supplier_cost, supplier_currency, supplier_paid, sort_order)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 1)
            ");
            
            $migrated = 0;
            foreach ($existingHotels as $hotel) {
                // חישוב לילות
                $nights = null;
                if ($hotel['hotel_checkin'] && $hotel['hotel_checkout']) {
                    $checkin = new DateTime($hotel['hotel_checkin']);
                    $checkout = new DateTime($hotel['hotel_checkout']);
                    $nights = $checkout->diff($checkin)->days;
                }
                
                // המרת ספק לפורמט החדש
                $supplier = 'DIRECT'; // ברירת מחדל
                $oldSupplier = strtoupper($hotel['hotel_supplier'] ?? '');
                if (in_array($oldSupplier, ['DIRECT', 'ORANGE', 'YELLOW', 'BLUE'])) {
                    $supplier = $oldSupplier;
                }
                
                // קוד אישור
                $confirmationCode = $hotel['hotel_confirmation'] ?? '';
                
                $insertStmt->execute([
                    $hotel['id'],
                    $hotel['hotel_name'],
                    $hotel['destination'],
                    $hotel['hotel_checkin'],
                    $hotel['hotel_checkout'],
                    $nights,
                    $confirmationCode,
                    $supplier,
                    $hotel['hotel_supplier_cost'] ?? 0,
                    $hotel['hotel_supplier_currency'] ?? 'USD',
                    $hotel['hotel_supplier_paid'] ?? 'unpaid'
                ]);
                $migrated++;
                echo "   ✓ הזמנה #{$hotel['id']} - {$hotel['hotel_name']}\n";
            }
            
            echo "\n   ✅ הועברו $migrated מלונות בהצלחה!\n";
        }
    } else {
        echo "ℹ️ אין מלונות להעברה\n";
    }
    
    echo "\n===========================================\n";
    echo "   ✅ הסקריפט הושלם בהצלחה!\n";
    echo "===========================================\n";
    
    // הצגת מבנה הטבלה
    echo "\n📋 מבנה הטבלה החדשה:\n";
    echo "-------------------------------------------\n";
    $columns = $pdo->query("DESCRIBE order_hotels")->fetchAll(PDO::FETCH_ASSOC);
    foreach ($columns as $col) {
        echo sprintf("   %-20s %-30s %s\n", 
            $col['Field'], 
            $col['Type'], 
            $col['Null'] === 'NO' ? 'NOT NULL' : ''
        );
    }
    
    echo "\n📊 ספקי מלונות זמינים:\n";
    echo "-------------------------------------------\n";
    echo "   • DIRECT  - ישירות מהמלון\n";
    echo "   • ORANGE  - סיטונאי\n";
    echo "   • YELLOW  - סיטונאי\n";
    echo "   • BLUE    - סיטונאי\n";
    
} catch (Exception $e) {
    echo "❌ שגיאה: " . $e->getMessage() . "\n";
}

echo "</pre>";
?>
