<?php
/**
 * עדכון שערי מט"ח אוטומטי - לשימוש עם cron job
 * משתמש ב-API הרשמי של בנק ישראל
 * 
 * הגדרת cron (פעם ביום בשעה 16:00):
 * 0 16 * * * /usr/bin/php /path/to/newprojectmillion/cron_update_rates.php
 */

define('LOG_FILE', __DIR__ . '/logs/exchange_rates.log');
define('MARGIN_PERCENT', 0.86);

function writeLog($message) {
    $timestamp = date('Y-m-d H:i:s');
    $logDir = dirname(LOG_FILE);
    if (!is_dir($logDir)) mkdir($logDir, 0755, true);
    file_put_contents(LOG_FILE, "[$timestamp] $message\n", FILE_APPEND);
    echo "[$timestamp] $message\n";
}

function fetchRates() {
    $url = 'https://boi.org.il/PublicApi/GetExchangeRates';
    
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_SSL_VERIFYPEER => false
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode !== 200) throw new Exception("HTTP $httpCode");
    
    $data = json_decode($response, true);
    if (!$data || !isset($data['exchangeRates'])) throw new Exception("Invalid JSON");
    
    $rates = [];
    foreach ($data['exchangeRates'] as $rate) {
        if ($rate['key'] === 'USD') $rates['USD'] = (float)$rate['currentExchangeRate'];
        if ($rate['key'] === 'EUR') $rates['EUR'] = (float)$rate['currentExchangeRate'];
    }
    
    if (empty($rates['USD']) || empty($rates['EUR'])) throw new Exception("USD/EUR not found");
    
    // הוספת מרווח
    $rates['USD'] = round($rates['USD'] * (1 + MARGIN_PERCENT / 100), 4);
    $rates['EUR'] = round($rates['EUR'] * (1 + MARGIN_PERCENT / 100), 4);
    
    return $rates;
}

writeLog("=== התחלת עדכון שערים ===");

try {
    $rates = fetchRates();
    writeLog("נמשכו (כולל מרווח " . MARGIN_PERCENT . "%): USD={$rates['USD']}, EUR={$rates['EUR']}");
    
    require_once __DIR__ . '/includes/config.php';
    $pdo = new PDO("mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4", DB_USER, DB_PASS);
    
    // וידוא עמודת source
    try {
        $stmt = $pdo->query("SHOW COLUMNS FROM exchange_rates LIKE 'source'");
        if ($stmt->rowCount() == 0) {
            $pdo->exec("ALTER TABLE exchange_rates ADD COLUMN source VARCHAR(100) DEFAULT 'ידני'");
        }
    } catch (Exception $e) {}
    
    $source = 'בנק ישראל + ' . MARGIN_PERCENT . '% (cron)';
    
    try {
        $stmt = $pdo->prepare("INSERT INTO exchange_rates (currency, rate, updated_at, updated_by, source) VALUES (?, ?, NOW(), 0, ?) ON DUPLICATE KEY UPDATE rate = VALUES(rate), updated_at = NOW(), source = VALUES(source)");
        $stmt->execute(['USD', $rates['USD'], $source]);
        $stmt->execute(['EUR', $rates['EUR'], $source]);
    } catch (Exception $e) {
        $stmt = $pdo->prepare("INSERT INTO exchange_rates (currency, rate, updated_at, updated_by) VALUES (?, ?, NOW(), 0) ON DUPLICATE KEY UPDATE rate = VALUES(rate), updated_at = NOW()");
        $stmt->execute(['USD', $rates['USD']]);
        $stmt->execute(['EUR', $rates['EUR']]);
    }
    
    writeLog("✅ נשמר!");
    exit(0);
} catch (Exception $e) {
    writeLog("❌ " . $e->getMessage());
    exit(1);
}
