<?php
// Callback לתשלומי דוקטים
require_once 'includes/config.php';

$pdo = getDB();
$pdo->exec("SET NAMES utf8mb4");

$logFile = __DIR__ . '/logs/cardcom_log.txt';

// קריאת הנתונים
$rawInput = file_get_contents('php://input');
$data = json_decode($rawInput, true);

if (!$data && !empty($_POST)) {
    $data = $_POST;
}

file_put_contents($logFile, date('Y-m-d H:i:s') . " - DOCKET CALLBACK\n", FILE_APPEND);

if (empty($data)) {
    file_put_contents($logFile, date('Y-m-d H:i:s') . " - ERROR: Empty data\n", FILE_APPEND);
    echo "OK";
    exit;
}

try {
    $responseCode = $data['ResponseCode'] ?? -1;
    $returnValue = $data['ReturnValue'] ?? '';
    $tranzactionInfo = $data['TranzactionInfo'] ?? [];
    
    // פענוח ReturnValue: token|amount|payer|docket|docketId|paymentId
    $parts = explode('|', $returnValue);
    
    if (count($parts) < 6 || $parts[3] !== 'docket') {
        file_put_contents($logFile, date('Y-m-d H:i:s') . " - ERROR: Invalid ReturnValue for docket\n", FILE_APPEND);
        echo "OK";
        exit;
    }
    
    $token = $parts[0];
    $amount = floatval($tranzactionInfo['Amount'] ?? $parts[1]);
    $payerName = urldecode($parts[2]);
    $docketId = intval($parts[4]);
    $paymentId = intval($parts[5]);
    
    $approvalNumber = $tranzactionInfo['ApprovalNumber'] ?? '';
    $transactionId = $tranzactionInfo['TranzactionId'] ?? '';
    
    file_put_contents($logFile, date('Y-m-d H:i:s') . " - Docket=$docketId, Payment=$paymentId, Amount=$amount, Code=$responseCode\n", FILE_APPEND);
    
    if ($responseCode !== 0) {
        // תשלום נכשל - מחיקת הרשומה
        $pdo->prepare("DELETE FROM docket_payments WHERE id = ? AND notes = 'pending'")->execute([$paymentId]);
        file_put_contents($logFile, date('Y-m-d H:i:s') . " - Payment failed, deleted pending record\n", FILE_APPEND);
        echo "OK";
        exit;
    }
    
    // עדכון רשומת התשלום
    $stmt = $pdo->prepare("UPDATE docket_payments SET 
        amount = ?,
        approval_number = ?,
        transaction_id = ?,
        payer_name = ?,
        notes = NULL,
        payment_date = NOW()
        WHERE id = ?");
    $stmt->execute([$amount, $approvalNumber, $transactionId, $payerName, $paymentId]);
    
    // עדכון total_paid בדוקט
    $stmt = $pdo->prepare("SELECT SUM(amount) as total FROM docket_payments WHERE docket_id = ? AND notes IS NULL");
    $stmt->execute([$docketId]);
    $totalPaid = $stmt->fetch()['total'] ?? 0;
    
    $pdo->prepare("UPDATE dockets SET total_paid = ? WHERE id = ?")->execute([$totalPaid, $docketId]);
    
    file_put_contents($logFile, date('Y-m-d H:i:s') . " - DOCKET SUCCESS! Total paid: $totalPaid\n", FILE_APPEND);
    
    // יצירת קבלה ב-Invoice4U
    $stmt = $pdo->prepare("SELECT * FROM dockets WHERE id = ?");
    $stmt->execute([$docketId]);
    $docket = $stmt->fetch();
    
    if ($docket) {
        // Invoice4U API call
        $invoice4uToken = '835ed34c-30bb-499b-8dd7-1460dba1343b';
        
        $invoiceData = [
            'CustomerName' => $docket['customer_name'],
            'CustomerEmail' => $docket['customer_email'],
            'CustomerPhone' => $docket['customer_phone'],
            'Items' => [[
                'Name' => "דוקט #{$docket['docket_number']} - {$docket['destination']}",
                'Price' => $amount,
                'Quantity' => 1
            ]],
            'PaymentType' => 1, // Credit card
            'PaymentSum' => $amount,
            'Comments' => "תשלום עבור דוקט #{$docket['docket_number']}"
        ];
        
        $soapRequest = '<?xml version="1.0" encoding="utf-8"?>
        <soap:Envelope xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/" xmlns:inv="http://invoice4u.co.il/">
            <soap:Body>
                <inv:CreateDocument>
                    <inv:token>' . $invoice4uToken . '</inv:token>
                    <inv:doc>
                        <inv:DocumentType>6</inv:DocumentType>
                        <inv:CustomerName>' . htmlspecialchars($docket['customer_name']) . '</inv:CustomerName>
                        <inv:CustomerEmail>' . htmlspecialchars($docket['customer_email'] ?? '') . '</inv:CustomerEmail>
                        <inv:CustomerPhone>' . htmlspecialchars($docket['customer_phone'] ?? '') . '</inv:CustomerPhone>
                        <inv:Items>
                            <inv:DocumentItem>
                                <inv:Name>דוקט #' . $docket['docket_number'] . ' - ' . htmlspecialchars($docket['destination'] ?? '') . '</inv:Name>
                                <inv:Price>' . $amount . '</inv:Price>
                                <inv:Quantity>1</inv:Quantity>
                            </inv:DocumentItem>
                        </inv:Items>
                        <inv:Payments>
                            <inv:PaymentItem>
                                <inv:PaymentType>1</inv:PaymentType>
                                <inv:PaymentSum>' . $amount . '</inv:PaymentSum>
                                <inv:PaymentCardLastDigits>' . ($tranzactionInfo['Last4CardDigitsString'] ?? '0000') . '</inv:PaymentCardLastDigits>
                                <inv:PaymentCardType>1</inv:PaymentCardType>
                                <inv:PaymentCardApprovalNumber>' . $approvalNumber . '</inv:PaymentCardApprovalNumber>
                            </inv:PaymentItem>
                        </inv:Payments>
                        <inv:Comments>תשלום עבור דוקט #' . $docket['docket_number'] . '</inv:Comments>
                    </inv:doc>
                </inv:CreateDocument>
            </soap:Body>
        </soap:Envelope>';
        
        $ch = curl_init('https://api.invoice4u.co.il/Services/ApiService.svc');
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [
                'Content-Type: text/xml; charset=utf-8',
                'SOAPAction: http://invoice4u.co.il/IApiService/CreateDocument'
            ],
            CURLOPT_POSTFIELDS => $soapRequest,
            CURLOPT_TIMEOUT => 30
        ]);
        
        $invoiceResponse = curl_exec($ch);
        curl_close($ch);
        
        if (preg_match('/<DocumentNumber>(\d+)<\/DocumentNumber>/', $invoiceResponse, $matches)) {
            $receiptNumber = $matches[1];
            
            // שמירת מספר קבלה בתשלום
            $pdo->prepare("UPDATE docket_payments SET receipt_number = ? WHERE id = ?")->execute([$receiptNumber, $paymentId]);
            
            file_put_contents($logFile, date('Y-m-d H:i:s') . " - Invoice4U: OK #$receiptNumber\n", FILE_APPEND);
        }
    }
    
    echo "OK";
    
} catch (Exception $e) {
    file_put_contents($logFile, date('Y-m-d H:i:s') . " - ERROR: " . $e->getMessage() . "\n", FILE_APPEND);
    echo "OK";
}
