<?php
require_once 'includes/config.php';
require_once 'includes/auth.php';
requireLogin();

$pdo = getDB();
$pdo->exec("SET NAMES utf8mb4");

$agentId = $_SESSION['agent_id'];
$docketId = $_GET['id'] ?? null;
$docket = null;
$items = [];

// טעינת ספקים
$suppliers = $pdo->query("SELECT * FROM settings_suppliers ORDER BY name")->fetchAll();

if ($docketId) {
    $stmt = $pdo->prepare("SELECT * FROM dockets WHERE id = ?");
    $stmt->execute([$docketId]);
    $docket = $stmt->fetch();
    
    $stmt = $pdo->prepare("SELECT * FROM docket_items WHERE docket_id = ? ORDER BY id");
    $stmt->execute([$docketId]);
    $items = $stmt->fetchAll();
}

// שמירה
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $pdo->beginTransaction();
        
        // יצירת מספר דוקט ייחודי
        if (!$docketId) {
            $year = date('y');
            $stmt = $pdo->query("SELECT MAX(CAST(SUBSTRING(docket_number, 4) AS UNSIGNED)) as max_num FROM dockets WHERE docket_number LIKE 'HB{$year}%'");
            $maxNum = $stmt->fetch()['max_num'] ?? 0;
            $docketNumber = 'HB' . $year . str_pad($maxNum + 1, 4, '0', STR_PAD_LEFT);
        }
        
        $data = [
            'agent_id' => $agentId,
            'customer_name' => $_POST['customer_name'],
            'customer_phone' => $_POST['customer_phone'],
            'customer_email' => $_POST['customer_email'],
            'destination' => $_POST['destination'],
            'departure_date' => $_POST['departure_date'] ?: null,
            'return_date' => $_POST['return_date'] ?: null,
            'adults' => (int)$_POST['adults'],
            'children' => (int)$_POST['children'],
            'currency' => $_POST['currency'],
            'status' => $_POST['status'],
            'notes' => $_POST['notes']
        ];
        
        if ($docketId) {
            // עדכון
            $sql = "UPDATE dockets SET " . implode(', ', array_map(fn($k) => "$k = ?", array_keys($data))) . " WHERE id = ?";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([...array_values($data), $docketId]);
        } else {
            // יצירה
            $data['docket_number'] = $docketNumber;
            $sql = "INSERT INTO dockets (" . implode(', ', array_keys($data)) . ") VALUES (" . implode(', ', array_fill(0, count($data), '?')) . ")";
            $stmt = $pdo->prepare($sql);
            $stmt->execute(array_values($data));
            $docketId = $pdo->lastInsertId();
        }
        
        // מחיקת פריטים קיימים ויצירה מחדש
        $pdo->prepare("DELETE FROM docket_items WHERE docket_id = ?")->execute([$docketId]);
        
        $totalCustomer = 0;
        $totalSupplier = 0;
        
        if (!empty($_POST['items'])) {
            $stmtItem = $pdo->prepare("INSERT INTO docket_items (docket_id, item_type, description, supplier_id, supplier_name, supplier_booking_code, supplier_cost, customer_price, supplier_paid, notes) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
            
            foreach ($_POST['items'] as $item) {
                if (empty($item['description'])) continue;
                
                $supplierName = '';
                if ($item['supplier_id']) {
                    foreach ($suppliers as $s) {
                        if ($s['id'] == $item['supplier_id']) {
                            $supplierName = $s['name'];
                            break;
                        }
                    }
                }
                
                $stmtItem->execute([
                    $docketId,
                    $item['item_type'],
                    $item['description'],
                    $item['supplier_id'] ?: null,
                    $supplierName,
                    $item['supplier_booking_code'],
                    floatval($item['supplier_cost']),
                    floatval($item['customer_price']),
                    isset($item['supplier_paid']) ? 1 : 0,
                    $item['notes'] ?? ''
                ]);
                
                $totalCustomer += floatval($item['customer_price']);
                $totalSupplier += floatval($item['supplier_cost']);
            }
        }
        
        // עדכון סה"כ
        $pdo->prepare("UPDATE dockets SET total_customer_price = ?, total_supplier_cost = ? WHERE id = ?")->execute([$totalCustomer, $totalSupplier, $docketId]);
        
        $pdo->commit();
        header("Location: docket_view.php?id=$docketId&saved=1");
        exit;
        
    } catch (Exception $e) {
        $pdo->rollBack();
        $error = $e->getMessage();
    }
}

$itemTypes = [
    'flight' => '✈️ טיסה',
    'hotel' => '🏨 מלון',
    'package' => '📦 חבילה',
    'luggage' => '🧳 כבודה',
    'insurance' => '🛡️ ביטוח',
    'tour' => '🎯 טיול/סיור',
    'transfer' => '🚐 העברה',
    'other' => '📋 אחר'
];
?>
<!DOCTYPE html>
<html lang="he" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $docket ? 'עריכת' : 'יצירת' ?> דוקט - חופשה במבצע</title>
    <style>
        * { box-sizing: border-box; margin: 0; padding: 0; }
        body { font-family: 'Segoe UI', Arial, sans-serif; background: #f0f2f5; min-height: 100vh; }
        
        .header { background: linear-gradient(135deg, #1e3c72, #2a5298); color: white; padding: 20px; display: flex; justify-content: space-between; align-items: center; }
        .header h1 { font-size: 1.5rem; }
        .header a { color: white; text-decoration: none; background: rgba(255,255,255,0.2); padding: 10px 20px; border-radius: 8px; }
        
        .container { max-width: 1200px; margin: 0 auto; padding: 20px; }
        
        .card { background: white; border-radius: 12px; padding: 25px; margin-bottom: 20px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .card h2 { margin-bottom: 20px; color: #1e3c72; display: flex; align-items: center; gap: 10px; }
        
        .form-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; }
        
        .form-group { margin-bottom: 15px; }
        .form-group label { display: block; margin-bottom: 5px; font-weight: 500; color: #333; }
        .form-group input, .form-group select, .form-group textarea { width: 100%; padding: 12px; border: 1px solid #ddd; border-radius: 8px; font-size: 1rem; }
        .form-group textarea { resize: vertical; min-height: 80px; }
        
        .items-section { margin-top: 20px; }
        .item-row { background: #f8f9fa; border-radius: 10px; padding: 15px; margin-bottom: 15px; position: relative; }
        .item-row .remove-btn { position: absolute; top: 10px; left: 10px; background: #dc3545; color: white; border: none; width: 30px; height: 30px; border-radius: 50%; cursor: pointer; font-size: 1.2rem; }
        .item-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 10px; }
        .item-grid input, .item-grid select { padding: 10px; border: 1px solid #ddd; border-radius: 6px; }
        
        .add-item-btn { background: #28a745; color: white; border: none; padding: 12px 25px; border-radius: 8px; cursor: pointer; font-size: 1rem; margin-top: 10px; }
        .add-item-btn:hover { background: #218838; }
        
        .checkbox-group { display: flex; align-items: center; gap: 8px; }
        .checkbox-group input { width: auto; }
        
        .btn-save { background: linear-gradient(135deg, #28a745, #20c997); color: white; border: none; padding: 15px 40px; border-radius: 10px; font-size: 1.1rem; cursor: pointer; }
        .btn-save:hover { transform: translateY(-2px); box-shadow: 0 5px 20px rgba(40,167,69,0.3); }
        
        .error { background: #f8d7da; color: #721c24; padding: 15px; border-radius: 8px; margin-bottom: 20px; }
        
        .totals { background: #e3f2fd; padding: 20px; border-radius: 10px; margin-top: 20px; }
        .totals-grid { display: grid; grid-template-columns: repeat(3, 1fr); gap: 20px; text-align: center; }
        .totals-grid .value { font-size: 1.5rem; font-weight: bold; color: #1e3c72; }
        .totals-grid .profit { color: #28a745; }
    </style>
</head>
<body>
    <div class="header">
        <h1>📁 <?= $docket ? 'עריכת דוקט #' . htmlspecialchars($docket['docket_number']) : 'דוקט חדש' ?></h1>
        <a href="dockets.php">← חזרה לרשימה</a>
    </div>
    
    <div class="container">
        <?php if (!empty($error)): ?>
        <div class="error"><?= htmlspecialchars($error) ?></div>
        <?php endif; ?>
        
        <form method="POST" id="docketForm">
            <div class="card">
                <h2>👤 פרטי לקוח</h2>
                <div class="form-grid">
                    <div class="form-group">
                        <label>שם לקוח *</label>
                        <input type="text" name="customer_name" required value="<?= htmlspecialchars($docket['customer_name'] ?? '') ?>">
                    </div>
                    <div class="form-group">
                        <label>טלפון</label>
                        <input type="text" name="customer_phone" value="<?= htmlspecialchars($docket['customer_phone'] ?? '') ?>">
                    </div>
                    <div class="form-group">
                        <label>אימייל</label>
                        <input type="email" name="customer_email" value="<?= htmlspecialchars($docket['customer_email'] ?? '') ?>">
                    </div>
                </div>
            </div>
            
            <div class="card">
                <h2>✈️ פרטי הנסיעה</h2>
                <div class="form-grid">
                    <div class="form-group">
                        <label>יעד</label>
                        <input type="text" name="destination" value="<?= htmlspecialchars($docket['destination'] ?? '') ?>">
                    </div>
                    <div class="form-group">
                        <label>תאריך יציאה</label>
                        <input type="date" name="departure_date" value="<?= $docket['departure_date'] ?? '' ?>">
                    </div>
                    <div class="form-group">
                        <label>תאריך חזרה</label>
                        <input type="date" name="return_date" value="<?= $docket['return_date'] ?? '' ?>">
                    </div>
                    <div class="form-group">
                        <label>מבוגרים</label>
                        <input type="number" name="adults" min="1" value="<?= $docket['adults'] ?? 1 ?>">
                    </div>
                    <div class="form-group">
                        <label>ילדים</label>
                        <input type="number" name="children" min="0" value="<?= $docket['children'] ?? 0 ?>">
                    </div>
                    <div class="form-group">
                        <label>מטבע</label>
                        <select name="currency">
                            <option value="ILS" <?= ($docket['currency'] ?? 'ILS') === 'ILS' ? 'selected' : '' ?>>₪ שקל</option>
                            <option value="USD" <?= ($docket['currency'] ?? '') === 'USD' ? 'selected' : '' ?>>$ דולר</option>
                            <option value="EUR" <?= ($docket['currency'] ?? '') === 'EUR' ? 'selected' : '' ?>>€ יורו</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label>סטטוס</label>
                        <select name="status">
                            <option value="active" <?= ($docket['status'] ?? 'active') === 'active' ? 'selected' : '' ?>>פעיל</option>
                            <option value="completed" <?= ($docket['status'] ?? '') === 'completed' ? 'selected' : '' ?>>הושלם</option>
                            <option value="cancelled" <?= ($docket['status'] ?? '') === 'cancelled' ? 'selected' : '' ?>>בוטל</option>
                        </select>
                    </div>
                </div>
                <div class="form-group">
                    <label>הערות</label>
                    <textarea name="notes"><?= htmlspecialchars($docket['notes'] ?? '') ?></textarea>
                </div>
            </div>
            
            <div class="card">
                <h2>📦 רכיבים (טיסה, מלון, תוספות...)</h2>
                
                <div id="itemsContainer">
                    <?php if (!empty($items)): ?>
                        <?php foreach ($items as $i => $item): ?>
                        <div class="item-row">
                            <button type="button" class="remove-btn" onclick="this.parentElement.remove(); updateTotals();">×</button>
                            <div class="item-grid">
                                <select name="items[<?= $i ?>][item_type]">
                                    <?php foreach ($itemTypes as $val => $label): ?>
                                    <option value="<?= $val ?>" <?= $item['item_type'] === $val ? 'selected' : '' ?>><?= $label ?></option>
                                    <?php endforeach; ?>
                                </select>
                                <input type="text" name="items[<?= $i ?>][description]" placeholder="תיאור" value="<?= htmlspecialchars($item['description']) ?>">
                                <select name="items[<?= $i ?>][supplier_id]">
                                    <option value="">-- ספק --</option>
                                    <?php foreach ($suppliers as $s): ?>
                                    <option value="<?= $s['id'] ?>" <?= $item['supplier_id'] == $s['id'] ? 'selected' : '' ?>><?= htmlspecialchars($s['name']) ?></option>
                                    <?php endforeach; ?>
                                </select>
                                <input type="text" name="items[<?= $i ?>][supplier_booking_code]" placeholder="מספר הזמנה ספק" value="<?= htmlspecialchars($item['supplier_booking_code']) ?>">
                                <input type="number" step="0.01" name="items[<?= $i ?>][supplier_cost]" placeholder="עלות לספק" value="<?= $item['supplier_cost'] ?>" onchange="updateTotals()">
                                <input type="number" step="0.01" name="items[<?= $i ?>][customer_price]" placeholder="מחיר ללקוח" value="<?= $item['customer_price'] ?>" onchange="updateTotals()">
                                <div class="checkbox-group">
                                    <input type="checkbox" name="items[<?= $i ?>][supplier_paid]" <?= $item['supplier_paid'] ? 'checked' : '' ?>>
                                    <label>שולם לספק</label>
                                </div>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <div class="item-row">
                            <button type="button" class="remove-btn" onclick="this.parentElement.remove(); updateTotals();">×</button>
                            <div class="item-grid">
                                <select name="items[0][item_type]">
                                    <?php foreach ($itemTypes as $val => $label): ?>
                                    <option value="<?= $val ?>"><?= $label ?></option>
                                    <?php endforeach; ?>
                                </select>
                                <input type="text" name="items[0][description]" placeholder="תיאור">
                                <select name="items[0][supplier_id]">
                                    <option value="">-- ספק --</option>
                                    <?php foreach ($suppliers as $s): ?>
                                    <option value="<?= $s['id'] ?>"><?= htmlspecialchars($s['name']) ?></option>
                                    <?php endforeach; ?>
                                </select>
                                <input type="text" name="items[0][supplier_booking_code]" placeholder="מספר הזמנה ספק">
                                <input type="number" step="0.01" name="items[0][supplier_cost]" placeholder="עלות לספק" onchange="updateTotals()">
                                <input type="number" step="0.01" name="items[0][customer_price]" placeholder="מחיר ללקוח" onchange="updateTotals()">
                                <div class="checkbox-group">
                                    <input type="checkbox" name="items[0][supplier_paid]">
                                    <label>שולם לספק</label>
                                </div>
                            </div>
                        </div>
                    <?php endif; ?>
                </div>
                
                <button type="button" class="add-item-btn" onclick="addItem()">➕ הוסף רכיב</button>
                
                <div class="totals">
                    <div class="totals-grid">
                        <div>
                            <div class="value" id="totalCustomer">₪0</div>
                            <div>מחיר ללקוח</div>
                        </div>
                        <div>
                            <div class="value" id="totalSupplier">₪0</div>
                            <div>עלות ספקים</div>
                        </div>
                        <div>
                            <div class="value profit" id="totalProfit">₪0</div>
                            <div>רווח</div>
                        </div>
                    </div>
                </div>
            </div>
            
            <div style="text-align: center; margin-top: 20px;">
                <button type="submit" class="btn-save">💾 שמור דוקט</button>
            </div>
        </form>
    </div>
    
    <script>
        let itemIndex = <?= max(count($items), 1) ?>;
        
        const itemTypes = <?= json_encode($itemTypes) ?>;
        const suppliers = <?= json_encode($suppliers) ?>;
        
        function addItem() {
            const container = document.getElementById('itemsContainer');
            const html = `
                <div class="item-row">
                    <button type="button" class="remove-btn" onclick="this.parentElement.remove(); updateTotals();">×</button>
                    <div class="item-grid">
                        <select name="items[${itemIndex}][item_type]">
                            ${Object.entries(itemTypes).map(([val, label]) => `<option value="${val}">${label}</option>`).join('')}
                        </select>
                        <input type="text" name="items[${itemIndex}][description]" placeholder="תיאור">
                        <select name="items[${itemIndex}][supplier_id]">
                            <option value="">-- ספק --</option>
                            ${suppliers.map(s => `<option value="${s.id}">${s.name}</option>`).join('')}
                        </select>
                        <input type="text" name="items[${itemIndex}][supplier_booking_code]" placeholder="מספר הזמנה ספק">
                        <input type="number" step="0.01" name="items[${itemIndex}][supplier_cost]" placeholder="עלות לספק" onchange="updateTotals()">
                        <input type="number" step="0.01" name="items[${itemIndex}][customer_price]" placeholder="מחיר ללקוח" onchange="updateTotals()">
                        <div class="checkbox-group">
                            <input type="checkbox" name="items[${itemIndex}][supplier_paid]">
                            <label>שולם לספק</label>
                        </div>
                    </div>
                </div>
            `;
            container.insertAdjacentHTML('beforeend', html);
            itemIndex++;
        }
        
        function updateTotals() {
            let customer = 0, supplier = 0;
            document.querySelectorAll('[name$="[customer_price]"]').forEach(el => customer += parseFloat(el.value) || 0);
            document.querySelectorAll('[name$="[supplier_cost]"]').forEach(el => supplier += parseFloat(el.value) || 0);
            
            document.getElementById('totalCustomer').textContent = '₪' + customer.toLocaleString();
            document.getElementById('totalSupplier').textContent = '₪' + supplier.toLocaleString();
            document.getElementById('totalProfit').textContent = '₪' + (customer - supplier).toLocaleString();
        }
        
        updateTotals();
    </script>
</body>
</html>
