<?php
// עריכת הזמנה
require_once 'includes/auth.php';
requireLogin();
blockAccountant();

$agent = getCurrentAgent();

// רק מנהל יכול לערוך הזמנות
if ($agent['role'] !== 'admin') {
    header('Location: orders_panel.php');
    exit;
}

$pdo = getDB();

// שליפת ההזמנה
$order_id = $_GET['id'] ?? null;
if (!$order_id) {
    header('Location: orders.php');
    exit;
}

$stmt = $pdo->prepare("SELECT * FROM orders WHERE id = ?");
$stmt->execute([$order_id]);
$order = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$order) {
    header('Location: orders.php');
    exit;
}

$passengers = json_decode($order['passengers'] ?? '[]', true);
$payments = json_decode($order['payments'] ?? '[]', true);
$extras = json_decode($order['extras'] ?? '[]', true);

// שליפת שערי מטבע
$rates = ['USD' => 3.65, 'EUR' => 3.95, 'ILS' => 1];
$stmt = $pdo->query("SELECT currency, rate FROM exchange_rates");
while ($row = $stmt->fetch()) {
    $rates[$row['currency']] = (float)$row['rate'];
}

// שליפת ספקים מההגדרות
$suppliers = $pdo->query("SELECT name FROM settings_suppliers WHERE active = 1 ORDER BY sort_order, name")->fetchAll(PDO::FETCH_COLUMN);

// שליפת חברות תעופה מההגדרות
$airlines = $pdo->query("SELECT name FROM settings_airlines WHERE active = 1 ORDER BY sort_order, name")->fetchAll(PDO::FETCH_COLUMN);
?>
<!DOCTYPE html>
<html lang="he" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>עריכת הזמנה #<?= $order['id'] ?> | <?= SITE_NAME ?></title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
            min-height: 100vh;
            padding: 0;
            color: #333;
        }
        .top-nav {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 15px 30px;
            background: white;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            flex-wrap: wrap;
            gap: 15px;
        }
        .logo {
            font-size: 1.3rem;
            font-weight: bold;
            color: #333;
        }
        .logo span {
            color: #667eea;
        }
        .main-nav {
            display: flex;
            gap: 5px;
        }
        .nav-link {
            padding: 10px 20px;
            text-decoration: none;
            color: #555;
            border-radius: 8px;
            font-weight: 500;
            transition: all 0.2s;
        }
        .nav-link:hover {
            background: #f0f0f0;
            color: #333;
        }
        .nav-link.active {
            background: #667eea;
            color: white;
        }
        .user-info {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        .user-name {
            color: #666;
        }
        .btn-logout {
            background: #f44336;
            color: white;
            padding: 8px 15px;
            border-radius: 5px;
            text-decoration: none;
            font-size: 0.85rem;
        }
        .container {
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
        }
        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 20px 30px;
            border-radius: 15px 15px 0 0;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .header h1 {
            font-size: 1.5rem;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        .header .badge {
            background: #ff9800;
            padding: 5px 12px;
            border-radius: 20px;
            font-size: 0.8rem;
        }
        .form-container {
            background: white;
            padding: 30px;
            border-radius: 0 0 15px 15px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.3);
        }
        .section {
            margin-bottom: 30px;
            padding-bottom: 25px;
            border-bottom: 2px solid #eee;
        }
        .section:last-child {
            border-bottom: none;
            margin-bottom: 0;
        }
        .section-title {
            font-size: 1.1rem;
            color: #667eea;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        .form-row {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-bottom: 15px;
        }
        .form-group {
            display: flex;
            flex-direction: column;
        }
        .form-group label {
            font-size: 0.85rem;
            color: #666;
            margin-bottom: 5px;
            font-weight: 500;
        }
        .form-group input,
        .form-group select {
            padding: 12px 15px;
            border: 2px solid #e0e0e0;
            border-radius: 10px;
            font-size: 1rem;
            transition: all 0.3s;
        }
        .form-group input:focus,
        .form-group select:focus {
            border-color: #667eea;
            outline: none;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }
        .form-group.full-width {
            grid-column: 1 / -1;
        }
        
        /* נוסעים */
        .passengers-list {
            display: flex;
            flex-direction: column;
            gap: 10px;
        }
        .passenger-input {
            display: flex;
            gap: 10px;
            align-items: center;
        }
        .passenger-input input {
            flex: 1;
            padding: 12px 15px;
            border: 2px solid #e0e0e0;
            border-radius: 10px;
            font-size: 1rem;
        }
        .passenger-input .passenger-num {
            background: #667eea;
            color: white;
            width: 30px;
            height: 30px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
            font-size: 0.85rem;
        }
        .btn-add-passenger {
            background: #e8f5e9;
            color: #4caf50;
            border: 2px dashed #4caf50;
            padding: 10px;
            border-radius: 10px;
            cursor: pointer;
            font-size: 0.9rem;
            transition: all 0.3s;
        }
        .btn-add-passenger:hover {
            background: #4caf50;
            color: white;
        }
        .btn-remove {
            background: #ffebee;
            color: #e53935;
            border: none;
            width: 30px;
            height: 30px;
            border-radius: 50%;
            cursor: pointer;
            font-size: 1.2rem;
        }
        
        /* ספק טיסה */
        .supplier-card {
            background: #f8f9fa;
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 15px;
            border: 2px solid #e0e0e0;
        }
        .supplier-card.return-supplier {
            background: #fff8e1;
            border-color: #ffcc80;
        }
        .supplier-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 15px;
        }
        .supplier-header h4 {
            color: #333;
            font-size: 1rem;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        /* העלאת קבצים */
        .upload-area {
            border: 2px dashed #ccc;
            border-radius: 10px;
            padding: 20px;
            text-align: center;
            cursor: pointer;
            transition: all 0.3s;
            background: #fafafa;
        }
        .upload-area:hover {
            border-color: #667eea;
            background: #f0f4ff;
        }
        .upload-area:not(.has-file):hover .upload-text::after {
            content: ' - עכשיו אפשר Ctrl+V!';
            color: #667eea;
            font-weight: 600;
        }
        .upload-area.has-file {
            border-color: #4caf50;
            background: #e8f5e9;
        }
        .upload-area.drag-over {
            border-color: #667eea;
            background: #e3f2fd;
            transform: scale(1.02);
        }
        .upload-area:focus {
            outline: 3px solid #667eea;
            outline-offset: 2px;
            background: #e8eaf6;
        }
        .upload-area:focus .upload-text {
            color: #667eea;
            font-weight: 600;
        }
        .upload-area input[type="file"] {
            display: none;
        }
        .upload-icon {
            font-size: 2rem;
            margin-bottom: 10px;
        }
        .upload-text {
            color: #666;
            font-size: 0.85rem;
        }
        .file-preview {
            margin-top: 10px;
            max-width: 100%;
            max-height: 150px;
            border-radius: 8px;
        }
        .file-name {
            margin-top: 10px;
            color: #4caf50;
            font-weight: 500;
        }
        .btn-remove-file {
            display: block;
            margin: 10px auto 0;
            padding: 5px 15px;
            background: #ffebee;
            color: #e53935;
            border: 1px solid #e53935;
            border-radius: 5px;
            cursor: pointer;
            font-size: 0.8rem;
        }
        .btn-remove-file:hover {
            background: #e53935;
            color: white;
        }
        
        /* קוביה מתקפלת - שונות */
        .collapsible-header {
            display: flex;
            align-items: center;
            gap: 10px;
            padding: 15px 20px;
            background: #f5f5f5;
            border: 2px dashed #ccc;
            border-radius: 10px;
            cursor: pointer;
            transition: all 0.3s;
        }
        .collapsible-header:hover {
            background: #e8f5e9;
            border-color: #4caf50;
        }
        .collapsible-header.open {
            background: #e3f2fd;
            border-color: #667eea;
            border-style: solid;
            border-radius: 10px 10px 0 0;
        }
        .collapsible-icon {
            font-size: 1.2rem;
            transition: transform 0.3s;
        }
        .collapsible-hint {
            margin-right: auto;
            color: #999;
            font-size: 0.85rem;
        }
        .collapsible-header.open .collapsible-hint {
            display: none;
        }
        .collapsible-content {
            background: #fafafa;
            border: 2px solid #667eea;
            border-top: none;
            border-radius: 0 0 10px 10px;
            padding: 20px;
        }
        .extra-item {
            background: white;
            border: 1px solid #e0e0e0;
            border-radius: 10px;
            padding: 15px;
            margin-bottom: 15px;
            position: relative;
        }
        .extra-item-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 1px solid #eee;
        }
        .extra-item-title {
            font-weight: 600;
            color: #333;
        }
        .btn-remove-extra {
            background: #ffebee;
            color: #e53935;
            border: none;
            width: 28px;
            height: 28px;
            border-radius: 50%;
            cursor: pointer;
            font-size: 1rem;
        }
        .btn-remove-extra:hover {
            background: #e53935;
            color: white;
        }
        .btn-add-extra {
            display: block;
            width: 100%;
            padding: 12px;
            background: #e8f5e9;
            color: #4caf50;
            border: 2px dashed #4caf50;
            border-radius: 10px;
            cursor: pointer;
            font-size: 0.95rem;
            font-weight: 500;
            transition: all 0.3s;
        }
        .btn-add-extra:hover {
            background: #4caf50;
            color: white;
        }
        .extra-item .upload-area {
            padding: 15px;
        }
        .extra-item .upload-icon {
            font-size: 1.5rem;
            margin-bottom: 5px;
        }
        .extra-item .upload-text {
            font-size: 0.8rem;
        }
        .extra-item .file-preview {
            max-height: 100px;
        }
        
        /* תשלומים */
        .payment-row {
            display: flex;
            gap: 10px;
            align-items: center;
            margin-bottom: 10px;
            flex-wrap: wrap;
        }
        .payment-row input, .payment-row select {
            flex: 1;
            min-width: 80px;
            padding: 10px;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            font-size: 0.95rem;
        }
        .payment-row .payment-name { flex: 2; min-width: 120px; }
        .payment-row .payment-amount { flex: 1; min-width: 80px; }
        .payment-row .payment-method { flex: 1; min-width: 100px; }
        .payment-row .payment-installments { flex: 0; width: 80px !important; min-width: 80px; }
        .btn-add {
            background: #28a745;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 8px;
            cursor: pointer;
            font-size: 0.9rem;
            margin-top: 10px;
        }
        .btn-add:hover {
            background: #218838;
        }
        .total-summary {
            margin-top: 20px;
            padding: 15px;
            background: #d4edda;
            border-radius: 10px;
        }
        .total-row {
            display: flex;
            justify-content: space-between;
            align-items: center;
            font-size: 1.2rem;
            font-weight: 600;
        }
        .total-amount {
            font-size: 1.5rem;
            color: #28a745;
        }
        
        /* סיכום רווח */
        .profit-section {
            background: #e8f5e9;
            border-radius: 15px;
            padding: 20px;
        }
        .profit-summary {
            background: white;
            padding: 15px;
            border-radius: 10px;
        }
        .profit-row {
            display: flex;
            justify-content: space-between;
            padding: 10px 0;
            border-bottom: 1px solid #e0e0e0;
            font-size: 1.05rem;
        }
        .profit-row:last-child {
            border-bottom: none;
        }
        .profit-row.total {
            border-top: 2px solid #28a745;
            margin-top: 10px;
            padding-top: 15px;
            font-weight: bold;
            font-size: 1.2rem;
        }
        .profit-value {
            font-weight: 600;
        }
        .profit-value.expense {
            color: #dc3545;
        }
        .profit-value.final {
            color: #28a745;
            font-size: 1.4rem;
        }
        .profit-value.negative {
            color: #dc3545 !important;
        }
        .rates-info {
            margin-top: 15px;
            padding: 10px 15px;
            background: #fff3cd;
            border-radius: 8px;
            font-size: 0.9rem;
            color: #856404;
        }
        .rates-info a {
            color: #856404;
        }
        
        /* הערות */
        .notes-textarea {
            width: 100%;
            padding: 12px;
            border: 2px solid #e0e0e0;
            border-radius: 10px;
            font-size: 1rem;
            resize: vertical;
            min-height: 80px;
        }
        .notes-textarea:focus {
            border-color: #667eea;
            outline: none;
        }
        
        /* אזור פלט - אישור הזמנה */
        .output-section {
            background: #e8f5e9;
            padding: 20px;
            border-radius: 15px;
            margin-top: 20px;
        }
        .email-output {
            background: white;
            padding: 20px;
            border-radius: 10px;
            border: 1px solid #ddd;
            white-space: pre-wrap;
            font-family: inherit;
            line-height: 1.8;
            max-height: 400px;
            overflow-y: auto;
        }
        .copy-btn {
            background: #007bff;
            color: white;
            border: none;
            padding: 12px 25px;
            border-radius: 8px;
            cursor: pointer;
            font-size: 1rem;
            margin-top: 15px;
            margin-left: 10px;
        }
        .copy-btn:hover {
            background: #0056b3;
        }
        .send-email-btn {
            background: #28a745;
            color: white;
            border: none;
            padding: 12px 25px;
            border-radius: 8px;
            cursor: pointer;
            font-size: 1rem;
        }
        .send-email-btn:hover {
            background: #1e7e34;
        }
        .send-email-btn:disabled {
            background: #6c757d;
            cursor: not-allowed;
        }
        .output-buttons {
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
            margin-top: 15px;
        }
        
        /* כפתורים */
        .btn {
            padding: 15px 30px;
            border: none;
            border-radius: 10px;
            font-size: 1rem;
            cursor: pointer;
            transition: all 0.3s;
            font-weight: 600;
        }
        .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 20px rgba(102, 126, 234, 0.4);
        }
        .btn-secondary {
            background: #f5f5f5;
            color: #333;
        }
        .btn-test {
            background: linear-gradient(135deg, #ff9800 0%, #f57c00 100%);
            color: white;
        }
        .btn-test:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 20px rgba(255, 152, 0, 0.4);
        }
        
        /* צ'קבוקס טיסת חזור שונה */
        .different-return-check {
            display: flex;
            align-items: center;
            gap: 10px;
            padding: 15px;
            background: #fff3e0;
            border-radius: 10px;
            margin-top: 15px;
            cursor: pointer;
        }
        .different-return-check input {
            width: 20px;
            height: 20px;
        }
        .different-return-check label {
            cursor: pointer;
            font-weight: 500;
            color: #e65100;
        }
        
        /* סטטוס תשלום */
        .payment-status {
            display: flex;
            gap: 10px;
        }
        .payment-status label {
            display: flex;
            align-items: center;
            gap: 5px;
            padding: 10px 15px;
            border: 2px solid #e0e0e0;
            border-radius: 10px;
            cursor: pointer;
            transition: all 0.3s;
        }
        .payment-status input[type="radio"] {
            display: none;
        }
        .payment-status input[type="radio"]:checked + label {
            border-color: #667eea;
            background: #f0f4ff;
        }
        .payment-status .paid {
            color: #4caf50;
        }
        .payment-status .unpaid {
            color: #ff9800;
        }
        
        /* Actions */
        .form-actions {
            display: flex;
            gap: 15px;
            justify-content: center;
            margin-top: 30px;
        }
        
        /* Back link */
        .back-link {
            display: inline-flex;
            align-items: center;
            gap: 5px;
            color: white;
            text-decoration: none;
            margin-bottom: 15px;
            opacity: 0.8;
        }
        .back-link:hover {
            opacity: 1;
        }
        
        /* Hidden */
        .hidden {
            display: none !important;
        }
    </style>
</head>
<body>
    <!-- תפריט עליון -->
    <header class="top-nav">
        <div class="logo">✈️ <span>חופשה</span> במבצע</div>
        <nav class="main-nav">
            <a href="complete_order_lite.php" class="nav-link active">⚡ הזמנה לייט</a>
            <a href="drafts_list.php" class="nav-link">📋 טיוטות</a>
            <a href="orders_panel.php" class="nav-link">🔍 פאנל מתקדם</a>
            <a href="reports.php" class="nav-link">📈 דוחות</a>
            <a href="settings.php" class="nav-link">⚙️ הגדרות</a>
        </nav>
        <div class="user-info">
            <span class="user-name">שלום, <?= sanitize($agent['full_name']) ?></span>
            <a href="logout.php" class="btn-logout">התנתק</a>
        </div>
    </header>

    <div class="container">
        <div class="header">
            <h1>✏️ עריכת הזמנה #<?= $order['id'] ?></h1>
            <a href="view_order_new.php?id=<?= $order['id'] ?>" class="badge" style="text-decoration:none;">← חזרה לצפייה</a>
        </div>
        
        <div class="form-container">
            <form id="orderForm">
                <input type="hidden" name="order_id" value="<?= $order['id'] ?>">
                
                <!-- פרטי הזמנה בסיסיים -->
                <div class="section">
                    <h3 class="section-title">📋 פרטי הזמנה</h3>
                    <div class="form-row">
                        <div class="form-group">
                            <label>תאריך ביצוע ההזמנה</label>
                            <input type="date" name="order_date" id="order_date" value="<?= $order['order_date'] ?? date('Y-m-d') ?>" required>
                        </div>
                        <div class="form-group">
                            <label>יעד (באנגלית)</label>
                            <input type="text" name="destination" id="destination" value="<?= htmlspecialchars($order['destination'] ?? '') ?>" placeholder="Budapest, Hungary" required>
                        </div>
                    </div>
                    <div class="form-row">
                        <div class="form-group">
                            <label>📅 תאריך טיסה - הלוך</label>
                            <input type="date" name="departure_date" id="departure_date" value="<?= $order['departure_date'] ?? '' ?>" required>
                        </div>
                        <div class="form-group">
                            <label>📅 תאריך טיסה - חזור</label>
                            <input type="date" name="return_date" id="return_date" value="<?= $order['return_date'] ?? '' ?>" required>
                        </div>
                    </div>
                </div>
                
                <!-- נוסעים -->
                <div class="section">
                    <h3 class="section-title">👥 נוסעים</h3>
                    <div class="passengers-list" id="passengers-list">
                        <?php foreach ($passengers as $i => $passenger): ?>
                        <div class="passenger-input">
                            <span class="passenger-num"><?= $i + 1 ?></span>
                            <input type="text" name="passengers[]" value="<?= htmlspecialchars($passenger) ?>" placeholder="שם מלא באנגלית (כמו בדרכון)" required>
                            <?php if ($i > 0): ?><button type="button" class="btn-remove-passenger" onclick="this.parentElement.remove(); updatePassengerNumbers();">✕</button><?php endif; ?>
                        </div>
                        <?php endforeach; ?>
                        <?php if (empty($passengers)): ?>
                        <div class="passenger-input">
                            <span class="passenger-num">1</span>
                            <input type="text" name="passengers[]" placeholder="שם מלא באנגלית (כמו בדרכון)" required>
                        </div>
                        <?php endif; ?>
                    </div>
                    <button type="button" class="btn-add-passenger" onclick="addPassenger()">
                        + הוסף נוסע
                    </button>
                </div>
                
                <!-- פרטי קשר -->
                <div class="section">
                    <h3 class="section-title">📞 פרטי קשר</h3>
                    <div class="form-row">
                        <div class="form-group">
                            <label>טלפון ראש ההזמנה</label>
                            <input type="tel" name="customer_phone" id="customer_phone" value="<?= htmlspecialchars($order['customer_phone'] ?? '') ?>" placeholder="050-1234567" required>
                        </div>
                        <div class="form-group">
                            <label>אימייל</label>
                            <input type="email" name="customer_email" id="customer_email" value="<?= htmlspecialchars($order['customer_email'] ?? '') ?>" placeholder="email@example.com">
                        </div>
                    </div>
                </div>
                
                <!-- ספק טיסה - הלוך -->
                <div class="section">
                    <h3 class="section-title">✈️ פרטי טיסה</h3>
                    
                    <div class="supplier-card" id="outbound-supplier-card">
                        <div class="supplier-header">
                            <h4>🛫 טיסת הלוך</h4>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label>ספק</label>
                                <select name="flight_supplier" id="flight_supplier" required>
                                    <option value="DIRECT">DIRECT (ישיר מחברת התעופה)</option>
                                    <?php foreach ($suppliers as $supplier): ?>
                                        <option value="<?= htmlspecialchars($supplier) ?>"><?= htmlspecialchars($supplier) ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="form-group">
                                <label>חברת תעופה</label>
                                <select name="airline_outbound" id="airline_outbound" required>
                                    <option value="">בחר חברה</option>
                                    <?php foreach ($airlines as $airline): ?>
                                        <option value="<?= htmlspecialchars($airline) ?>"><?= htmlspecialchars($airline) ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label>קוד הזמנה</label>
                                <input type="text" name="booking_code_outbound" id="booking_code_outbound" placeholder="ABC123">
                            </div>
                            <div class="form-group">
                                <label>עלות לספק</label>
                                <input type="number" step="0.01" name="flight_supplier_cost" id="flight_supplier_cost" placeholder="500.00">
                            </div>
                            <div class="form-group">
                                <label>מטבע</label>
                                <select name="flight_supplier_currency" id="flight_supplier_currency">
                                    <option value="USD">$ USD</option>
                                    <option value="EUR">€ EUR</option>
                                    <option value="ILS">₪ ILS</option>
                                </select>
                            </div>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label>סטטוס תשלום לספק</label>
                                <div class="payment-status">
                                    <input type="radio" name="flight_supplier_paid" id="flight_paid_yes" value="paid">
                                    <label for="flight_paid_yes" class="paid">✓ שולם</label>
                                    
                                    <input type="radio" name="flight_supplier_paid" id="flight_paid_no" value="unpaid" checked>
                                    <label for="flight_paid_no" class="unpaid">⏳ טרם שולם</label>
                                </div>
                            </div>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group full-width">
                                <label>וואצ'ר / אישור הזמנה מהספק</label>
                                <div class="upload-area" onclick="clickUpload(event, 'voucher_outbound')">
                                    <input type="file" name="voucher_outbound" id="voucher_outbound" accept="image/*,.pdf" onchange="handleFileUpload(this, 'outbound')">
                                    <div class="upload-icon">📄</div>
                                    <div class="upload-text">לחץ כאן ואז הדבק (Ctrl+V) או בחר קובץ</div>
                                    <div id="preview-outbound"></div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- צ'קבוקס לטיסת חזור שונה -->
                    <div class="different-return-check">
                        <input type="checkbox" id="different_return" onchange="toggleReturnFlight()">
                        <label for="different_return">🔄 טיסת החזור עם ספק או חברת תעופה שונה</label>
                    </div>
                    
                    <!-- ספק טיסה - חזור (מוסתר כברירת מחדל) -->
                    <div class="supplier-card return-supplier hidden" id="return-supplier-card">
                        <div class="supplier-header">
                            <h4>🛬 טיסת חזור</h4>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label>ספק</label>
                                <select name="return_flight_supplier" id="return_flight_supplier">
                                    <option value="DIRECT">DIRECT (ישיר מחברת התעופה)</option>
                                    <?php foreach ($suppliers as $supplier): ?>
                                        <option value="<?= htmlspecialchars($supplier) ?>"><?= htmlspecialchars($supplier) ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="form-group">
                                <label>חברת תעופה</label>
                                <select name="airline_return" id="airline_return">
                                    <option value="">בחר חברה</option>
                                    <?php foreach ($airlines as $airline): ?>
                                        <option value="<?= htmlspecialchars($airline) ?>"><?= htmlspecialchars($airline) ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label>קוד הזמנה</label>
                                <input type="text" name="booking_code_return" id="booking_code_return" placeholder="XYZ789">
                            </div>
                            <div class="form-group">
                                <label>עלות לספק</label>
                                <input type="number" step="0.01" name="return_flight_supplier_cost" id="return_flight_supplier_cost" placeholder="500.00">
                            </div>
                            <div class="form-group">
                                <label>מטבע</label>
                                <select name="return_flight_supplier_currency" id="return_flight_supplier_currency">
                                    <option value="USD">$ USD</option>
                                    <option value="EUR">€ EUR</option>
                                    <option value="ILS">₪ ILS</option>
                                </select>
                            </div>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label>סטטוס תשלום לספק</label>
                                <div class="payment-status">
                                    <input type="radio" name="return_flight_supplier_paid" id="return_paid_yes" value="paid">
                                    <label for="return_paid_yes" class="paid">✓ שולם</label>
                                    
                                    <input type="radio" name="return_flight_supplier_paid" id="return_paid_no" value="unpaid" checked>
                                    <label for="return_paid_no" class="unpaid">⏳ טרם שולם</label>
                                </div>
                            </div>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group full-width">
                                <label>וואצ'ר / אישור הזמנה מהספק</label>
                                <div class="upload-area" onclick="clickUpload(event, 'voucher_return')">
                                    <input type="file" name="voucher_return" id="voucher_return" accept="image/*,.pdf" onchange="handleFileUpload(this, 'return')">
                                    <div class="upload-icon">📄</div>
                                    <div class="upload-text">לחץ כאן ואז הדבק (Ctrl+V) או בחר קובץ</div>
                                    <div id="preview-return"></div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- פרטי מלון -->
                <div class="section">
                    <h3 class="section-title">🏨 פרטי מלון</h3>
                    
                    <div class="supplier-card">
                        <div class="form-row">
                            <div class="form-group">
                                <label>ספק מלון</label>
                                <select name="hotel_supplier" id="hotel_supplier" required>
                                    <option value="DIRECT">DIRECT (ישירות מהמלון)</option>
                                    <option value="ORANGE">ORANGE</option>
                                    <option value="BLUE">BLUE</option>
                                    <option value="YELLOW">YELLOW</option>
                                </select>
                            </div>
                            <div class="form-group">
                                <label>קוד הזמנה</label>
                                <input type="text" name="hotel_booking_code" id="hotel_booking_code" placeholder="HTL123456">
                            </div>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label>עלות לספק</label>
                                <input type="number" step="0.01" name="hotel_supplier_cost" id="hotel_supplier_cost" placeholder="300.00">
                            </div>
                            <div class="form-group">
                                <label>מטבע</label>
                                <select name="hotel_supplier_currency" id="hotel_supplier_currency">
                                    <option value="USD">$ USD</option>
                                    <option value="EUR">€ EUR</option>
                                    <option value="ILS">₪ ILS</option>
                                </select>
                            </div>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label>סטטוס תשלום לספק</label>
                                <div class="payment-status">
                                    <input type="radio" name="hotel_supplier_paid" id="hotel_paid_yes" value="paid">
                                    <label for="hotel_paid_yes" class="paid">✓ שולם</label>
                                    
                                    <input type="radio" name="hotel_supplier_paid" id="hotel_paid_no" value="unpaid" checked>
                                    <label for="hotel_paid_no" class="unpaid">⏳ טרם שולם</label>
                                </div>
                            </div>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group full-width">
                                <label>אישור הזמנה מהמלון / ספק</label>
                                <div class="upload-area" onclick="clickUpload(event, 'voucher_hotel')">
                                    <input type="file" name="voucher_hotel" id="voucher_hotel" accept="image/*,.pdf" onchange="handleFileUpload(this, 'hotel')">
                                    <div class="upload-icon">🏨</div>
                                    <div class="upload-text">לחץ כאן ואז הדבק (Ctrl+V) או בחר קובץ</div>
                                    <div id="preview-hotel"></div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- פרטי כבודה -->
                <div class="section">
                    <h3 class="section-title">🧳 פרטי כבודה</h3>
                    <div class="form-group">
                        <label>פירוט הכבודה שמגיעה ללקוח (בעברית)</label>
                        <textarea name="luggage_text" id="luggage_text" class="notes-textarea" rows="3" placeholder="לדוגמה:
לכל נוסע מגיע תיק גב 40x20x25 ס״מ + טרולי 8 ק״ג
מזוודה 20 ק״ג משותפת לזוג"></textarea>
                    </div>
                </div>
                
                <!-- שונות - מתקפל -->
                <div class="section">
                    <div class="collapsible-header" onclick="toggleExtras()">
                        <span class="collapsible-icon" id="extras-icon">➕</span>
                        <h3 class="section-title" style="margin:0;">🎫 שונות (כרטיסים, העברות, אטרקציות...)</h3>
                        <span class="collapsible-hint">לחץ להוספה</span>
                    </div>
                    
                    <div class="collapsible-content hidden" id="extras-content">
                        <div id="extras-list">
                            <!-- פריטים יתווספו כאן דינמית -->
                        </div>
                        
                        <button type="button" class="btn-add-extra" onclick="addExtraItem()">
                            ➕ הוסף פריט
                        </button>
                    </div>
                </div>
                
                <!-- תשלומים מלקוחות -->
                <div class="section">
                    <h3 class="section-title">💳 פרטי תשלומים מלקוחות</h3>
                    
                    <div class="payment-headers" style="display:flex; gap:10px; margin-bottom:5px; font-size:0.85rem; color:#666;">
                        <span style="flex:2; min-width:120px;">שם משלם</span>
                        <span style="flex:1; min-width:80px;">סכום ₪</span>
                        <span style="flex:1; min-width:100px;">אמצעי תשלום</span>
                        <span style="width:80px;">תשלומים</span>
                        <span style="width:30px;"></span>
                    </div>
                    <div id="payments-container">
                        <div class="payment-row">
                            <input type="text" name="payment_name[]" placeholder="שם המשלם" class="payment-name">
                            <input type="number" name="payment_amount[]" placeholder="סכום" class="payment-amount" oninput="calculateTotal()">
                            <select name="payment_method[]" class="payment-method">
                                <option value="אשראי">אשראי</option>
                                <option value="העברה בנקאית">העברה בנקאית</option>
                                <option value="מזומן">מזומן</option>
                                <option value="ביט">ביט</option>
                            </select>
                            <input type="number" name="payment_installments[]" placeholder="תשלומים" class="payment-installments" min="1" value="1">
                            <button type="button" class="btn-remove" onclick="removePaymentRow(this)" style="visibility:hidden;">✕</button>
                        </div>
                    </div>
                    <button type="button" class="btn-add" onclick="addPayment()">+ הוסף תשלום</button>
                    
                    <div class="total-summary">
                        <div class="total-row">
                            <span>סה"כ שולם:</span>
                            <span id="total-paid" class="total-amount">₪0</span>
                        </div>
                    </div>
                </div>
                
                <!-- סיכום רווח -->
                <div class="section profit-section">
                    <h3 class="section-title">💰 סיכום רווח</h3>
                    <div class="profit-summary">
                        <div class="profit-row">
                            <span>סה"כ מהלקוח:</span>
                            <span id="summary-customer" class="profit-value">₪0</span>
                        </div>
                        <div class="profit-row">
                            <span>עלות טיסה לספק:</span>
                            <span id="summary-flight" class="profit-value expense">-₪0</span>
                        </div>
                        <div class="profit-row">
                            <span>עלות מלון לספק:</span>
                            <span id="summary-hotel" class="profit-value expense">-₪0</span>
                        </div>
                        <div class="profit-row" id="extras-cost-row" style="display:none;">
                            <span>עלות שונות:</span>
                            <span id="summary-extras" class="profit-value expense">-₪0</span>
                        </div>
                        <div class="profit-row total">
                            <span>רווח סופי:</span>
                            <span id="summary-profit" class="profit-value final">₪0</span>
                        </div>
                    </div>
                    <input type="hidden" name="profit" id="profit" value="0">
                    <input type="hidden" name="usd_rate" id="usd_rate" value="<?= $rates['USD'] ?>">
                    <input type="hidden" name="eur_rate" id="eur_rate" value="<?= $rates['EUR'] ?>">
                    <div class="rates-info">
                        💱 שערי מטבע: $1 = ₪<?= $rates['USD'] ?> | €1 = ₪<?= $rates['EUR'] ?>
                        <a href="exchange_rates.php" style="margin-right:10px;">[עדכן שערים]</a>
                    </div>
                </div>
                
                <!-- הערות -->
                <div class="section">
                    <h3 class="section-title">📝 הערות</h3>
                    <div class="form-group">
                        <label>הערות חשובות להזמנה</label>
                        <textarea name="notes" id="notes" class="notes-textarea" rows="3" placeholder="הערות פנימיות..."></textarea>
                    </div>
                </div>
                
                <!-- כפתורי פעולה -->
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" onclick="window.location='view_order_new.php?id=<?= $order['id'] ?>'">ביטול</button>
                    <button type="button" class="btn btn-success" onclick="saveOrder()" style="background:#28a745;">💾 שמור שינויים</button>
                    <button type="button" class="btn btn-primary" onclick="generateEmail()">✨ צור אישור הזמנה למייל</button>
                </div>
                
            </form>
            
            <!-- תוצאה - אישור הזמנה מוכן -->
            <div id="output-section" class="output-section hidden">
                <h3 class="section-title">📧 אישור הזמנה מוכן לשליחה</h3>
                <div id="email-output" class="email-output"></div>
                <div class="output-buttons">
                    <button type="button" class="copy-btn" onclick="copyEmail()">📋 העתק למייל</button>
                    <button type="button" class="send-email-btn" onclick="sendEmailToCustomer()">📧 שלח מייל ללקוח</button>
                    <button type="button" class="btn btn-primary" onclick="saveOrder()">💾 עדכן הזמנה</button>
                </div>
            </div>
        </div>
    </div>
    
    <script>
        let passengerCount = 1;
        let extraItemCount = 0;
        
        // נתוני ההזמנה לעריכה
        const orderData = <?= json_encode($order, JSON_UNESCAPED_UNICODE) ?>;
        const orderPayments = <?= json_encode($payments, JSON_UNESCAPED_UNICODE) ?>;
        const orderExtras = <?= json_encode($extras, JSON_UNESCAPED_UNICODE) ?>;
        
        // פתיחה/סגירה של קוביית שונות
        function toggleExtras() {
            const content = document.getElementById('extras-content');
            const icon = document.getElementById('extras-icon');
            const header = document.querySelector('.collapsible-header');
            
            if (content.classList.contains('hidden')) {
                content.classList.remove('hidden');
                icon.textContent = '➖';
                header.classList.add('open');
                // אם אין פריטים, הוסף אחד אוטומטית
                if (extraItemCount === 0) {
                    addExtraItem();
                }
            } else {
                content.classList.add('hidden');
                icon.textContent = '➕';
                header.classList.remove('open');
            }
        }
        
        // הוספת פריט שונות
        function addExtraItem() {
            extraItemCount++;
            const list = document.getElementById('extras-list');
            const div = document.createElement('div');
            div.className = 'extra-item';
            div.id = 'extra-item-' + extraItemCount;
            div.innerHTML = `
                <div class="extra-item-header">
                    <span class="extra-item-title">🎫 פריט #${extraItemCount}</span>
                    <button type="button" class="btn-remove-extra" onclick="removeExtraItem(${extraItemCount})">×</button>
                </div>
                <div class="form-row">
                    <div class="form-group" style="flex:2;">
                        <label>תיאור</label>
                        <input type="text" name="extra_description[]" placeholder="כרטיסים למשחק ברצלונה / העברות משדה / פארק מים...">
                    </div>
                    <div class="form-group">
                        <label>קוד הזמנה</label>
                        <input type="text" name="extra_booking_code[]" placeholder="TKT123">
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label>סכום</label>
                        <input type="number" step="0.01" name="extra_cost[]" placeholder="150.00">
                    </div>
                    <div class="form-group">
                        <label>מטבע</label>
                        <select name="extra_currency[]">
                            <option value="USD">$ USD</option>
                            <option value="EUR">€ EUR</option>
                            <option value="ILS">₪ ILS</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label>סטטוס תשלום</label>
                        <div class="payment-status">
                            <input type="radio" name="extra_paid_${extraItemCount}" id="extra_paid_yes_${extraItemCount}" value="paid">
                            <label for="extra_paid_yes_${extraItemCount}" class="paid">✓ שולם</label>
                            
                            <input type="radio" name="extra_paid_${extraItemCount}" id="extra_paid_no_${extraItemCount}" value="unpaid" checked>
                            <label for="extra_paid_no_${extraItemCount}" class="unpaid">⏳ טרם</label>
                        </div>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group full-width">
                        <label>אישור הזמנה / צילום מסך</label>
                        <div class="upload-area upload-area-extra" data-extra-id="${extraItemCount}" onclick="clickUpload(event, 'voucher_extra_${extraItemCount}')" tabindex="0">
                            <input type="file" name="voucher_extra_${extraItemCount}" id="voucher_extra_${extraItemCount}" accept="image/*,.pdf" onchange="handleFileUpload(this, 'extra_${extraItemCount}')">
                            <div class="upload-icon">📎</div>
                            <div class="upload-text">לחץ כאן ואז הדבק (Ctrl+V) או בחר קובץ</div>
                            <div id="preview-extra_${extraItemCount}"></div>
                        </div>
                    </div>
                </div>
            `;
            list.appendChild(div);
            
            // הוסף event listeners לאזור ההעלאה החדש
            const newUploadArea = div.querySelector('.upload-area-extra');
            const currentExtraId = extraItemCount; // שמור את הערך הנוכחי
            if (newUploadArea) {
                newUploadArea.addEventListener('mouseenter', function() {
                    hoveredUploadArea = this;
                });
                newUploadArea.addEventListener('mouseleave', function() {
                    if (hoveredUploadArea === this) {
                        hoveredUploadArea = null;
                    }
                });
                newUploadArea.addEventListener('paste', function(e) {
                    const extraId = this.dataset.extraId;
                    handlePaste(e, 'extra_' + extraId);
                });
            }
        }
        
        // הסרת פריט שונות
        function removeExtraItem(id) {
            const item = document.getElementById('extra-item-' + id);
            if (item) {
                item.remove();
            }
        }
        
        // לחיצה על אזור העלאה - פותח בוחר קבצים רק אם לא לחצו על כפתור
        function clickUpload(event, inputId) {
            // אם לחצו על כפתור הסרה או על התמונה - לא לפתוח בוחר קבצים
            if (event.target.classList.contains('btn-remove-file') || 
                event.target.tagName === 'IMG' ||
                event.target.classList.contains('file-name')) {
                return;
            }
            document.getElementById(inputId).click();
        }
        
        // הוספת נוסע
        function addPassenger() {
            passengerCount++;
            const list = document.getElementById('passengers-list');
            const div = document.createElement('div');
            div.className = 'passenger-input';
            div.innerHTML = `
                <span class="passenger-num">${passengerCount}</span>
                <input type="text" name="passengers[]" placeholder="שם מלא באנגלית (כמו בדרכון)" required>
                <button type="button" class="btn-remove" onclick="removePassenger(this)">×</button>
            `;
            list.appendChild(div);
        }
        
        // הסרת נוסע
        function removePassenger(btn) {
            btn.parentElement.remove();
            updatePassengerNumbers();
        }
        
        // עדכון מספרי נוסעים
        function updatePassengerNumbers() {
            const nums = document.querySelectorAll('.passenger-num');
            nums.forEach((num, index) => {
                num.textContent = index + 1;
            });
            passengerCount = nums.length;
        }
        
        // הצג/הסתר טיסת חזור שונה
        function toggleReturnFlight() {
            const checkbox = document.getElementById('different_return');
            const returnCard = document.getElementById('return-supplier-card');
            
            if (checkbox.checked) {
                returnCard.classList.remove('hidden');
            } else {
                returnCard.classList.add('hidden');
            }
        }
        
        // טיפול בהעלאת קובץ
        function handleFileUpload(input, direction) {
            const preview = document.getElementById('preview-' + direction);
            const uploadArea = input.closest('.upload-area');
            
            if (input.files && input.files[0]) {
                showFilePreview(input.files[0], preview, uploadArea, direction);
            }
        }
        
        // הצגת תצוגה מקדימה של קובץ
        function showFilePreview(file, preview, uploadArea, direction) {
            uploadArea.classList.add('has-file');
            
            if (file.type.startsWith('image/')) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    preview.innerHTML = `
                        <img src="${e.target.result}" class="file-preview">
                        <div class="file-name">📎 ${file.name || 'צילום מסך'}</div>
                        <button type="button" class="btn-remove-file" onclick="removeFile('${direction}')">✕ הסר</button>
                    `;
                    // שמור את ה-data URL לשימוש בשליחה
                    uploadArea.dataset.imageData = e.target.result;
                };
                reader.readAsDataURL(file);
            } else {
                preview.innerHTML = `
                    <div class="file-name">📎 ${file.name}</div>
                    <button type="button" class="btn-remove-file" onclick="removeFile('${direction}')">✕ הסר</button>
                `;
            }
        }
        
        // הסרת קובץ
        function removeFile(direction) {
            const preview = document.getElementById('preview-' + direction);
            const uploadArea = preview.closest('.upload-area');
            const input = uploadArea.querySelector('input[type="file"]');
            
            preview.innerHTML = '';
            uploadArea.classList.remove('has-file');
            uploadArea.dataset.imageData = '';
            if (input) input.value = '';
        }
        
        // טיפול בהדבקה מ-clipboard (Ctrl+V)
        function handlePaste(e, direction) {
            const items = e.clipboardData?.items;
            if (!items) return;
            
            for (let item of items) {
                if (item.type.startsWith('image/')) {
                    e.preventDefault();
                    const file = item.getAsFile();
                    const preview = document.getElementById('preview-' + direction);
                    const uploadArea = preview.closest('.upload-area');
                    showFilePreview(file, preview, uploadArea, direction);
                    break;
                }
            }
        }
        
        // הוספת event listeners להדבקה על כל אזורי ההעלאה
        let hoveredUploadArea = null; // עוקב אחרי איזה אזור העכבר מעליו
        
        document.addEventListener('DOMContentLoaded', function() {
            document.querySelectorAll('.upload-area').forEach(area => {
                // מאפשר focus על האזור
                area.setAttribute('tabindex', '0');
                
                // עוקב אחרי hover
                area.addEventListener('mouseenter', function() {
                    hoveredUploadArea = this;
                });
                area.addEventListener('mouseleave', function() {
                    if (hoveredUploadArea === this) {
                        hoveredUploadArea = null;
                    }
                });
                
                // האזנה להדבקה על האזור עצמו (כשיש פוקוס)
                area.addEventListener('paste', function(e) {
                    const input = this.querySelector('input[type="file"]');
                    const direction = input.id.replace('voucher_', '');
                    handlePaste(e, direction);
                });
                
                // האזנה ל-drag & drop
                area.addEventListener('dragover', function(e) {
                    e.preventDefault();
                    this.classList.add('drag-over');
                });
                
                area.addEventListener('dragleave', function(e) {
                    this.classList.remove('drag-over');
                });
                
                area.addEventListener('drop', function(e) {
                    e.preventDefault();
                    this.classList.remove('drag-over');
                    const input = this.querySelector('input[type="file"]');
                    const direction = input.id.replace('voucher_', '');
                    if (e.dataTransfer.files.length > 0) {
                        const preview = document.getElementById('preview-' + direction);
                        showFilePreview(e.dataTransfer.files[0], preview, this, direction);
                    }
                });
            });
            
            // האזנה גלובלית ל-paste - עובד כשהעכבר מעל אזור העלאה
            document.addEventListener('paste', function(e) {
                if (hoveredUploadArea) {
                    const input = hoveredUploadArea.querySelector('input[type="file"]');
                    const direction = input.id.replace('voucher_', '');
                    handlePaste(e, direction);
                }
            });
            
            // האזנה לשינויים בעלויות ספקים
            const costInputs = [
                'flight_supplier_cost', 'flight_supplier_currency',
                'return_flight_supplier_cost', 'return_flight_supplier_currency',
                'hotel_supplier_cost', 'hotel_supplier_currency'
            ];
            costInputs.forEach(id => {
                const el = document.getElementById(id);
                if (el) {
                    el.addEventListener('change', calculateTotal);
                    el.addEventListener('input', calculateTotal);
                }
            });
            
            // האזנה לשינויים בשונות (עלויות נוספות)
            document.getElementById('extras-list')?.addEventListener('input', function(e) {
                if (e.target.name === 'extra_cost[]' || e.target.name === 'extra_currency[]') {
                    calculateTotal();
                }
            });
            document.getElementById('extras-list')?.addEventListener('change', function(e) {
                if (e.target.name === 'extra_currency[]') {
                    calculateTotal();
                }
            });
            
            // הפעלה ראשונית
            calculateTotal();
            
            // מילוי טופס מנתוני ההזמנה
            fillFormFromOrder();
        });
        
        // מילוי הטופס מנתוני ההזמנה
        function fillFormFromOrder() {
            if (!orderData) return;
            
            console.log('Loading order data:', orderData);
            
            // ספקי טיסה
            setSelectValue('flight_supplier', orderData.flight_supplier);
            setSelectValue('airline_outbound', orderData.airline_outbound);
            document.getElementById('booking_code_outbound').value = orderData.booking_code_outbound || '';
            document.getElementById('flight_supplier_cost').value = orderData.flight_supplier_cost || '';
            setSelectValue('flight_supplier_currency', orderData.flight_supplier_currency);
            setRadioValue('flight_supplier_paid', orderData.flight_supplier_paid || 'unpaid');
            
            // טיסת חזור
            setSelectValue('return_flight_supplier', orderData.return_flight_supplier);
            setSelectValue('airline_return', orderData.airline_return);
            document.getElementById('booking_code_return').value = orderData.booking_code_return || '';
            document.getElementById('return_flight_supplier_cost').value = orderData.return_flight_supplier_cost || '';
            setSelectValue('return_flight_supplier_currency', orderData.return_flight_supplier_currency);
            setRadioValue('return_flight_supplier_paid', orderData.return_flight_supplier_paid || 'unpaid');
            
            // מלון
            setSelectValue('hotel_supplier', orderData.hotel_supplier);
            document.getElementById('hotel_booking_code').value = orderData.hotel_supplier_booking || '';
            document.getElementById('hotel_supplier_cost').value = orderData.hotel_supplier_cost || '';
            setSelectValue('hotel_supplier_currency', orderData.hotel_supplier_currency);
            setRadioValue('hotel_supplier_paid', orderData.hotel_supplier_paid || 'unpaid');
            
            // כבודה והערות
            const luggageField = document.getElementById('luggage_text');
            if (luggageField) luggageField.value = orderData.luggage_details || '';
            
            const notesField = document.getElementById('notes');
            if (notesField) notesField.value = orderData.notes || '';
            
            // תשלומים
            if (orderPayments && orderPayments.length > 0) {
                const container = document.getElementById('payments-container');
                container.innerHTML = '';
                
                orderPayments.forEach((payment, index) => {
                    const row = document.createElement('div');
                    row.className = 'payment-row';
                    row.innerHTML = `
                        <input type="text" name="payment_name[]" value="${payment.name || ''}" placeholder="שם המשלם" class="payment-name">
                        <input type="number" name="payment_amount[]" value="${payment.amount || ''}" placeholder="סכום" class="payment-amount" oninput="calculateTotal()">
                        <select name="payment_method[]" class="payment-method">
                            <option value="אשראי" ${payment.method === 'אשראי' ? 'selected' : ''}>אשראי</option>
                            <option value="מזומן" ${payment.method === 'מזומן' ? 'selected' : ''}>מזומן</option>
                            <option value="העברה בנקאית" ${payment.method === 'העברה בנקאית' ? 'selected' : ''}>העברה בנקאית</option>
                            <option value="ביט" ${payment.method === 'ביט' ? 'selected' : ''}>ביט</option>
                            <option value="פייבוקס" ${payment.method === 'פייבוקס' ? 'selected' : ''}>פייבוקס</option>
                        </select>
                        <input type="number" name="payment_installments[]" value="${payment.installments || 1}" placeholder="תשלומים" class="payment-installments" min="1">
                        <button type="button" class="btn-remove" onclick="removePaymentRow(this)">✕</button>
                    `;
                    container.appendChild(row);
                });
            }
            
            // שונות/תוספות
            if (orderExtras && orderExtras.length > 0) {
                orderExtras.forEach(extra => {
                    addExtraItem();
                    const items = document.querySelectorAll('.extra-item');
                    const lastItem = items[items.length - 1];
                    if (lastItem) {
                        lastItem.querySelector('input[name="extra_description[]"]').value = extra.description || '';
                        lastItem.querySelector('input[name="extra_booking_code[]"]').value = extra.booking_code || '';
                        lastItem.querySelector('input[name="extra_cost[]"]').value = extra.cost || '';
                        setSelectValueInElement(lastItem, 'select[name="extra_currency[]"]', extra.currency);
                    }
                });
            }
            
            calculateTotal();
            
            // בדיקה אם יש טיסת חזור שונה - פתיחה אוטומטית
            const hasDifferentReturn = (
                (orderData.return_flight_supplier && orderData.return_flight_supplier !== orderData.flight_supplier) ||
                (orderData.airline_return && orderData.airline_return !== orderData.airline_outbound) ||
                (orderData.return_flight_supplier_cost && parseFloat(orderData.return_flight_supplier_cost) > 0)
            );
            
            if (hasDifferentReturn) {
                const checkbox = document.getElementById('different_return');
                if (checkbox) {
                    checkbox.checked = true;
                    toggleReturnFlight(); // פותח את הקטע
                }
            }
            
            console.log('Order data loaded successfully');
        }
        
        function setSelectValue(id, value) {
            const select = document.getElementById(id);
            if (select && value) {
                for (let option of select.options) {
                    if (option.value === value) {
                        option.selected = true;
                        break;
                    }
                }
            }
        }
        
        function setSelectValueInElement(element, selector, value) {
            const select = element.querySelector(selector);
            if (select && value) {
                for (let option of select.options) {
                    if (option.value === value) {
                        option.selected = true;
                        break;
                    }
                }
            }
        }
        
        function setRadioValue(name, value) {
            const radio = document.querySelector(`input[name="${name}"][value="${value}"]`);
            if (radio) radio.checked = true;
        }
        
        // הוספת תשלום
        function addPayment() {
            const container = document.getElementById('payments-container');
            const row = document.createElement('div');
            row.className = 'payment-row';
            row.innerHTML = `
                <input type="text" name="payment_name[]" placeholder="שם המשלם" class="payment-name">
                <input type="number" name="payment_amount[]" placeholder="סכום" class="payment-amount" oninput="calculateTotal()">
                <select name="payment_method[]" class="payment-method">
                    <option value="אשראי">אשראי</option>
                    <option value="העברה בנקאית">העברה בנקאית</option>
                    <option value="מזומן">מזומן</option>
                    <option value="ביט">ביט</option>
                </select>
                <input type="number" name="payment_installments[]" placeholder="תשלומים" class="payment-installments" min="1" value="1" style="width:80px;">
                <button type="button" class="btn-remove" onclick="removePaymentRow(this)">✕</button>
            `;
            container.appendChild(row);
        }
        
        // הסרת שורת תשלום
        function removePaymentRow(btn) {
            btn.parentElement.remove();
            calculateTotal();
        }
        
        // שערי מטבע מה-PHP
        const exchangeRates = {
            ILS: 1,
            USD: <?= $rates['USD'] ?>,
            EUR: <?= $rates['EUR'] ?>
        };
        
        // המרה לשקלים
        function convertToILS(amount, currency) {
            const rate = exchangeRates[currency] || 1;
            return amount * rate;
        }
        
        // חישוב סה"כ תשלומים ורווח
        function calculateTotal() {
            // סכום מהלקוח
            const amounts = document.querySelectorAll('input[name="payment_amount[]"]');
            let customerTotal = 0;
            amounts.forEach(input => {
                const val = parseFloat(input.value) || 0;
                customerTotal += val;
            });
            document.getElementById('total-paid').textContent = '₪' + customerTotal.toLocaleString();
            document.getElementById('summary-customer').textContent = '₪' + customerTotal.toLocaleString();

            let totalSupplierCost = 0;
            
            // עלות טיסה לספק
            const flightCost = parseFloat(document.getElementById('flight_supplier_cost')?.value) || 0;
            const flightCurrency = document.getElementById('flight_supplier_currency')?.value || 'ILS';
            const flightCostILS = convertToILS(flightCost, flightCurrency);
            
            // בדיקה אם יש טיסת חזור נפרדת
            const differentReturn = document.getElementById('different_return')?.checked;
            let returnFlightCostILS = 0;
            if (differentReturn) {
                const returnFlightCost = parseFloat(document.getElementById('return_flight_supplier_cost')?.value) || 0;
                const returnFlightCurrency = document.getElementById('return_flight_supplier_currency')?.value || 'ILS';
                returnFlightCostILS = convertToILS(returnFlightCost, returnFlightCurrency);
            }
            
            const totalFlightCostILS = flightCostILS + returnFlightCostILS;
            let flightDisplay = '-₪' + Math.round(totalFlightCostILS).toLocaleString();
            document.getElementById('summary-flight').textContent = flightDisplay;

            // עלות מלון לספק
            const hotelCost = parseFloat(document.getElementById('hotel_supplier_cost')?.value) || 0;
            const hotelCurrency = document.getElementById('hotel_supplier_currency')?.value || 'ILS';
            const hotelCostILS = convertToILS(hotelCost, hotelCurrency);
            
            let hotelDisplay = '-₪' + Math.round(hotelCostILS).toLocaleString();
            document.getElementById('summary-hotel').textContent = hotelDisplay;
            
            // עלות שונות
            let extrasCostILS = 0;
            const extraCosts = document.querySelectorAll('input[name="extra_cost[]"]');
            const extraCurrencies = document.querySelectorAll('select[name="extra_currency[]"]');
            extraCosts.forEach((input, index) => {
                const cost = parseFloat(input.value) || 0;
                const currency = extraCurrencies[index]?.value || 'ILS';
                extrasCostILS += convertToILS(cost, currency);
            });
            
            const extrasRow = document.getElementById('extras-cost-row');
            if (extrasCostILS > 0) {
                extrasRow.style.display = '';
                document.getElementById('summary-extras').textContent = '-₪' + Math.round(extrasCostILS).toLocaleString();
            } else {
                extrasRow.style.display = 'none';
            }
            
            totalSupplierCost = totalFlightCostILS + hotelCostILS + extrasCostILS;

            // רווח סופי
            const profit = customerTotal - totalSupplierCost;
            const profitEl = document.getElementById('summary-profit');
            profitEl.textContent = '₪' + Math.round(profit).toLocaleString();
            profitEl.classList.toggle('negative', profit < 0);
            
            // עדכון שדה hidden
            document.getElementById('profit').value = Math.round(profit);
        }
        
        // שליחת הטופס
        // יצירת אישור הזמנה למייל
        function generateEmail() {
            const passengers = Array.from(document.querySelectorAll('input[name="passengers[]"]'))
                .map(input => input.value).filter(v => v);
            
            const destination = document.getElementById('destination').value;
            const airline = document.getElementById('airline_outbound').value.trim();
            const departureDate = document.getElementById('departure_date').value;
            const returnDate = document.getElementById('return_date').value;
            const isDifferentReturn = document.getElementById('different_return')?.checked;
            const airlineReturn = isDifferentReturn ? document.getElementById('airline_return')?.value?.trim() : '';
            
            // מילון תרגום יעדים
            const destinationTranslations = {
                'budapest': 'בודפשט', 'prague': 'פראג', 'rome': 'רומא', 'paris': 'פריז',
                'barcelona': 'ברצלונה', 'amsterdam': 'אמסטרדם', 'london': 'לונדון', 'vienna': 'וינה',
                'berlin': 'ברלין', 'athens': 'אתונה', 'lisbon': 'ליסבון', 'madrid': 'מדריד',
                'milan': 'מילאנו', 'venice': 'ונציה', 'florence': 'פירנצה', 'krakow': 'קרקוב',
                'warsaw': 'ורשה', 'dublin': 'דבלין', 'brussels': 'בריסל', 'munich': 'מינכן',
                'salzburg': 'זלצבורג', 'rhodes': 'רודוס', 'crete': 'כרתים', 'cyprus': 'קפריסין',
                'larnaca': 'לרנקה', 'paphos': 'פאפוס', 'sofia': 'סופיה', 'bucharest': 'בוקרשט',
                'belgrade': 'בלגרד', 'tirana': 'טירנה', 'batumi': 'באטומי', 'tbilisi': 'טביליסי',
                'istanbul': 'איסטנבול', 'antalya': 'אנטליה'
            };
            
            // תרגום היעד לעברית
            const destLower = destination.toLowerCase().trim();
            let hebrewDestination = destination;
            for (const [eng, heb] of Object.entries(destinationTranslations)) {
                if (destLower.includes(eng)) {
                    hebrewDestination = heb;
                    break;
                }
            }
            
            // חברות עם צ'ק אין ידני (24 שעות לפני)
            const manualCheckinAirlines = ['wizzair', 'wizz air', 'wizz', 'ryanair', 'ryan air', 'flyone', 'fly one'];
            const airlineLower = airline.toLowerCase();
            const airlineReturnLower = (airlineReturn || airline).toLowerCase(); // אם אין חזור שונה, זו אותה חברה
            
            // בדיקה לכל כיוון בנפרד
            const isOutboundManual = manualCheckinAirlines.some(a => airlineLower.includes(a));
            const isReturnManual = manualCheckinAirlines.some(a => airlineReturnLower.includes(a));
            
            // בדיקה אם הטיסה מעל 3 שבועות
            const today = new Date();
            const flightDate = new Date(departureDate);
            const diffDays = Math.floor((flightDate - today) / (1000 * 60 * 60 * 24));
            const showCancellationInsurance = diffDays > 21;
            
            // בניית טקסט חברת תעופה
            let airlineText = '';
            if (isDifferentReturn && airlineReturn) {
                airlineText = `הטיסה בהלוך עם חברת ${airline}, בחזור עם חברת ${airlineReturn}. הטיסה בהלוך אמורה לצאת מטרמינל 3`;
            } else {
                airlineText = `הטיסות הן עם חברת ${airline}. הטיסה הלוך אמורה לצאת מטרמינל 3`;
            }
            
            // בניית טקסט כרטיסי טיסה - לפי שילוב החברות
            let ticketsText = '';
            if (isOutboundManual && isReturnManual) {
                // שתי הטיסות עם חברות צ'ק אין ידני
                ticketsText = `כרטיסי טיסה תקבלו מאיתנו בוואטסאפ בטווח של 24 שעות מכל טיסה אחרי שנבצע לכם את הצ'ק אין.`;
            } else if (!isOutboundManual && !isReturnManual) {
                // שתי הטיסות עם חברות רגילות
                ticketsText = `כרטיסי טיסה תקבלו בדלפק של חברת התעופה ביום הטיסה שם גם יבוצע לכם הצ'ק אין בחינם.`;
            } else if (isOutboundManual && !isReturnManual) {
                // רק הלוך עם חברה ידנית
                ticketsText = `בטיסת ההלוך - כרטיס הטיסה יישלח אליכם בוואטסאפ בטווח של 24 שעות מהטיסה אחרי שנבצע לכם את הצ'ק אין.
בטיסת החזור - תקבלו את כרטיס הטיסה בדלפק של חברת התעופה ביום הטיסה שם גם יבוצע לכם הצ'ק אין בחינם.`;
            } else {
                // רק חזור עם חברה ידנית
                ticketsText = `בטיסת ההלוך - תקבלו את כרטיס הטיסה בדלפק של חברת התעופה ביום הטיסה שם גם יבוצע לכם הצ'ק אין בחינם.
בטיסת החזור - כרטיס הטיסה יישלח אליכם בוואטסאפ בטווח של 24 שעות מהטיסה אחרי שנבצע לכם את הצ'ק אין.`;
            }
            
            // טקסט כבודה
            const luggageText = document.getElementById('luggage_text')?.value?.trim() || '';
            let luggageSection = '';
            if (luggageText) {
                luggageSection = `

🧳 כבודה
${luggageText}`;
            }
            
            // בניית טקסט ביטוח ביטול
            let cancellationInsuranceText = '';
            if (showCancellationInsurance) {
                cancellationInsuranceText = `

━━━━━━━━━━━━━━━━━━━━━━━━━

🛡️ ביטוח ביטול נסיעה

במידה ומועד החופשה שלכם רחוק אנו ממליצים בחום לרכוש ביטוח לביטול מכל סיבה של חברת טריפ גרנטי (מבית הפניקס) שיאפשר לכם לקבל 80 אחוז בחזרה אפילו עד 6 שעות ממועד הטיסה.
ניתן לרכוש את הביטוח עד 7 ימים ממועד הזמנת החופשה ובתנאי שטווח הזמן עד הנסיעה גדול מ- 15 ימים. לפרטים מלאים ולרכישת הביטוח יש ללחוץ על הקישור הבא:
https://www.tripguaranty.co.il/InsurancePurchase/?agentCode=228`;
            }

            const emailText = `שלום רב,

מצורפים בזאת אישורי הזמנה של הטיסות והמלון ב${hebrewDestination}.
אנא וודאו שהשמות רשומים בדיוק כפי שהם מופיעים אצלכם בדרכון באנגלית.

━━━━━━━━━━━━━━━━━━━━━━━━━

✈️ הנחיות חשובות מאוד לטיסה

${airlineText} אך יש לוודא זאת ביום הטיסה למקרה ויהיו שינויים בקישור הבא:
https://www.iaa.gov.il/airports/ben-gurion/flight-board/

יש להגיע לשדה לפחות 3 וחצי שעות לפני הטיסה.

${ticketsText}
${luggageSection}${cancellationInsuranceText}

━━━━━━━━━━━━━━━━━━━━━━━━━

🛡️ ביטוח נסיעות

טרם הנסיעה מומלץ מאוד לעשות ביטוח נסיעות לחו״ל שיכסה הוצאות רפואיות ומקרי חירום.
אנו ממליצים בחום על PassportCard, חברת ביטוח הנסיעות שלטעמנו היא המובילה והאמינה בישראל. ניתן לרכוש את הביטוח שלהם בקלות דרך הלינק הבא:
https://buy.passportcard.co.il/?AffiliateId=xEOn9OJj4ELo8mVndfdKjQ%3D%3D

יתרונות בולטים של PassportCard:
• אין הוצאות מהכיס בחו״ל - ללא צורך בהחזרים!
• מקבלים כרטיס פספורטכארד שמטעינים במקום ומאפשר לשלם ישר לרופא/בית חולים.
• שירות רופאים אונליין 24/7, כולל מרשמים דיגיטליים.
• שירות לקוחות זמין בוואטסאפ.
• מעקב ותפעול תביעה ישירות מהאפליקציה, ללא טפסים.
• אפשרות להרחבות כמו ביטול נסיעה, ציוד יקר, ספורט אתגרי ועוד.

━━━━━━━━━━━━━━━━━━━━━━━━━

למקרים דחופים נשמח לעמוד לרשותכם בוואטסאפ 050-6101031

מאחלים לכם חופשה נעימה! ✈️🌴`;

            document.getElementById('email-output').textContent = emailText;
            document.getElementById('output-section').classList.remove('hidden');
            document.getElementById('output-section').scrollIntoView({ behavior: 'smooth' });
        }

        // העתקה ללוח
        function copyEmail() {
            const text = document.getElementById('email-output').textContent;
            navigator.clipboard.writeText(text).then(() => {
                alert('✅ הטקסט הועתק! אפשר להדביק במייל');
            });
        }

        // משתנה לשמירת מזהה ההזמנה האחרונה
        let lastSavedOrderId = null;

        // שליחת מייל ללקוח
        async function sendEmailToCustomer() {
            const customerEmail = document.getElementById('customer_email')?.value;
            const emailBody = document.getElementById('email-output')?.textContent;
            const destination = document.getElementById('destination')?.value || '';
            
            // בדיקות
            if (!customerEmail) {
                alert('❌ לא הוזנה כתובת מייל של הלקוח');
                return;
            }
            
            if (!emailBody) {
                alert('❌ אין תוכן מייל לשליחה. לחץ קודם על "צור אישור הזמנה"');
                return;
            }
            
            // אם ההזמנה עדיין לא נשמרה, שומר קודם
            if (!lastSavedOrderId) {
                const shouldSave = confirm('ההזמנה עדיין לא נשמרה במערכת.\nהאם לשמור קודם ואז לשלוח מייל עם הקבצים המצורפים?');
                if (shouldSave) {
                    await saveOrder();
                    // ממתין קצת לסיום השמירה
                    await new Promise(resolve => setTimeout(resolve, 500));
                }
            }
            
            // אישור לפני שליחה
            if (!confirm(`לשלוח מייל לכתובת:\n${customerEmail}?`)) {
                return;
            }
            
            // הצגת טעינה
            const btn = document.querySelector('.send-email-btn');
            const originalText = btn.innerHTML;
            btn.innerHTML = '⏳ שולח...';
            btn.disabled = true;
            
            try {
                const formData = new FormData();
                formData.append('customer_email', customerEmail);
                formData.append('email_subject', `אישור הזמנה - ${destination} - חופשה במבצע`);
                formData.append('email_body', emailBody);
                if (lastSavedOrderId) {
                    formData.append('order_id', lastSavedOrderId);
                }
                
                const response = await fetch('api/send_order_email.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    alert('✅ ' + result.message);
                } else {
                    alert('❌ ' + result.error);
                }
            } catch (error) {
                alert('❌ שגיאה בשליחה: ' + error.message);
            } finally {
                btn.innerHTML = originalText;
                btn.disabled = false;
            }
        }

        // שמירה במערכת
        async function saveOrder() {
            const form = document.getElementById('orderForm');
            const formData = new FormData(form);
            
            // איסוף נוסעים
            const passengers = Array.from(document.querySelectorAll('input[name="passengers[]"]'))
                .map(input => input.value).filter(v => v);
            formData.set('passengers', JSON.stringify(passengers));
            
            // איסוף תשלומים
            const paymentNames = Array.from(document.querySelectorAll('input[name="payment_name[]"]'))
                .map(input => input.value);
            const paymentAmounts = Array.from(document.querySelectorAll('input[name="payment_amount[]"]'))
                .map(input => input.value);
            const paymentMethods = Array.from(document.querySelectorAll('select[name="payment_method[]"]'))
                .map(select => select.value);
            const paymentInstallments = Array.from(document.querySelectorAll('input[name="payment_installments[]"]'))
                .map(input => input.value || '1');
            
            const payments = paymentNames.map((name, i) => ({
                name: name,
                amount: paymentAmounts[i],
                method: paymentMethods[i],
                installments: paymentInstallments[i],
                date: document.getElementById('order_date').value // תאריך ההזמנה
            })).filter(p => p.name || p.amount);
            
            formData.set('payments', JSON.stringify(payments));
            
            // חישוב סה"כ מחיר מהתשלומים
            const totalPrice = payments.reduce((sum, p) => sum + (parseFloat(p.amount) || 0), 0);
            formData.set('total_price', totalPrice);
            
            // סוג הזמנה - תמיד separate בגרסה לייט
            formData.set('order_type', 'separate');
            
            // סטטוס תשלום לספקים
            formData.set('flight_supplier_paid', document.querySelector('input[name="flight_supplier_paid"]:checked')?.value || 'unpaid');
            formData.set('hotel_supplier_paid', document.querySelector('input[name="hotel_supplier_paid"]:checked')?.value || 'unpaid');
            
            // איסוף שונות (extras)
            const extraDescriptions = Array.from(document.querySelectorAll('input[name="extra_description[]"]'))
                .map(input => input.value);
            const extraCodes = Array.from(document.querySelectorAll('input[name="extra_booking_code[]"]'))
                .map(input => input.value);
            const extraCosts = Array.from(document.querySelectorAll('input[name="extra_cost[]"]'))
                .map(input => input.value);
            const extraCurrencies = Array.from(document.querySelectorAll('select[name="extra_currency[]"]'))
                .map(select => select.value);
            
            const extras = extraDescriptions.map((desc, i) => ({
                description: desc,
                booking_code: extraCodes[i],
                cost: extraCosts[i],
                currency: extraCurrencies[i]
            })).filter(e => e.description || e.cost);
            
            formData.set('extras', JSON.stringify(extras));
            
            // כבודה
            formData.set('luggage_details', document.getElementById('luggage_text')?.value || '');
            
            // איסוף צילומי מסך (base64 data)
            const outboundUpload = document.querySelector('#voucher_outbound')?.closest('.upload-area');
            if (outboundUpload?.dataset.imageData) {
                formData.set('voucher_outbound_data', outboundUpload.dataset.imageData);
            }
            
            const returnUpload = document.querySelector('#voucher_return')?.closest('.upload-area');
            if (returnUpload?.dataset.imageData) {
                formData.set('voucher_return_data', returnUpload.dataset.imageData);
            }
            
            const hotelUpload = document.querySelector('#voucher_hotel')?.closest('.upload-area');
            if (hotelUpload?.dataset.imageData) {
                formData.set('voucher_hotel_data', hotelUpload.dataset.imageData);
            }
            
            // איסוף צילומי מסך של שונות
            document.querySelectorAll('.upload-area-extra').forEach((area, index) => {
                if (area.dataset.imageData) {
                    formData.set('voucher_extra_' + (index + 1) + '_data', area.dataset.imageData);
                }
            });
            
            try {
                const response = await fetch('api/update_order.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    alert('✅ ההזמנה עודכנה בהצלחה!');
                    window.location.href = 'view_order_new.php?id=' + result.order_id;
                } else {
                    alert('❌ שגיאה: ' + result.error);
                }
            } catch (error) {
                alert('❌ שגיאת תקשורת: ' + error.message);
            }
        }

        // מילוי נתונים אקראיים לטסט
        function fillRandomData() {
            const destinations = ['Budapest, Hungary', 'Prague, Czech Republic', 'Barcelona, Spain', 'Rome, Italy', 'Paris, France', 'Athens, Greece', 'Vienna, Austria', 'Amsterdam, Netherlands'];
            const names = ['David Cohen', 'Sarah Levi', 'Michael Ben-David', 'Rachel Mizrahi', 'Yosef Peretz', 'Miriam Katz', 'Daniel Goldberg', 'Esther Friedman'];
            const airlines = document.querySelectorAll('#airline_outbound option');
            const suppliers = document.querySelectorAll('#flight_supplier option');
            const hotelSuppliers = document.querySelectorAll('#hotel_supplier option');
            
            // פונקציות עזר
            const random = (arr) => arr[Math.floor(Math.random() * arr.length)];
            const randomNum = (min, max) => Math.floor(Math.random() * (max - min + 1)) + min;
            const randomDate = (daysFromNow) => {
                const date = new Date();
                date.setDate(date.getDate() + daysFromNow);
                return date.toISOString().split('T')[0];
            };
            
            // יעד
            document.getElementById('destination').value = random(destinations);
            
            // תאריכים
            const departDays = randomNum(14, 60);
            document.getElementById('departure_date').value = randomDate(departDays);
            document.getElementById('return_date').value = randomDate(departDays + randomNum(3, 10));
            
            // נוסעים
            const passengerInputs = document.querySelectorAll('input[name="passengers[]"]');
            passengerInputs[0].value = random(names);
            if (Math.random() > 0.5) {
                addPassenger();
                document.querySelectorAll('input[name="passengers[]"]')[1].value = random(names);
            }
            
            // טלפון ומייל
            document.getElementById('customer_phone').value = '05' + randomNum(0, 9) + '-' + randomNum(1000000, 9999999);
            document.getElementById('customer_email').value = 'test' + randomNum(100, 999) + '@gmail.com';
            
            // טיסת הלוך
            if (suppliers.length > 1) {
                document.getElementById('flight_supplier').value = suppliers[randomNum(1, suppliers.length - 1)].value;
            }
            if (airlines.length > 1) {
                document.getElementById('airline_outbound').value = airlines[randomNum(1, airlines.length - 1)].value;
            }
            document.getElementById('booking_code_outbound').value = 'FLT' + randomNum(100000, 999999);
            document.getElementById('flight_supplier_cost').value = randomNum(200, 800);
            document.getElementById('flight_supplier_currency').value = random(['USD', 'EUR']);
            document.querySelector('input[name="flight_supplier_paid"][value="' + random(['paid', 'unpaid']) + '"]').checked = true;
            
            // מלון
            if (hotelSuppliers.length > 0) {
                document.getElementById('hotel_supplier').value = hotelSuppliers[randomNum(0, hotelSuppliers.length - 1)].value;
            }
            document.getElementById('hotel_booking_code').value = 'HTL' + randomNum(100000, 999999);
            document.getElementById('hotel_supplier_cost').value = randomNum(300, 1500);
            document.getElementById('hotel_supplier_currency').value = random(['USD', 'EUR']);
            document.querySelector('input[name="hotel_supplier_paid"][value="' + random(['paid', 'unpaid']) + '"]').checked = true;
            
            // תשלום מלקוח
            const paymentNames = document.querySelectorAll('input[name="payment_name[]"]');
            const paymentAmounts = document.querySelectorAll('input[name="payment_amount[]"]');
            paymentNames[0].value = random(names);
            paymentAmounts[0].value = randomNum(2000, 6000);
            
            // הערות
            // כבודה
            const luggageOptions = [
                'לכל נוסע מגיע תיק גב 40x20x25 ס״מ + טרולי 8 ק״ג',
                'לכל נוסע מגיע תיק גב + מזוודה 20 ק״ג',
                'תיק גב לכל נוסע + מזוודה 23 ק״ג משותפת לזוג',
                'לכל נוסע: תיק אישי + טרולי 10 ק״ג + מזוודה 20 ק״ג'
            ];
            document.getElementById('luggage_text').value = random(luggageOptions);
            
            // הערות
            document.getElementById('notes').value = 'הזמנת טסט - ' + new Date().toLocaleString('he-IL');
            
            // חישוב רווח
            calculateTotal();
            
            // הודעה
            alert('✅ נתוני טסט מולאו בהצלחה!');
        }
        
        document.getElementById('orderForm').addEventListener('submit', function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            
            // TODO: שליחה לשרת
            alert('הזמנה נשמרה! (בקרוב יהיה חיבור לשרת)');
            console.log('Form data:', Object.fromEntries(formData));
        });
    </script>
</body>
</html>
