<?php
// מערכת אימות משתמשים
// ============================================

require_once __DIR__ . '/config.php';

// התחלת סשן מאובטח
function startSecureSession() {
    if (session_status() === PHP_SESSION_NONE) {
        // הגדרת זמן חיי הסשן בשרת (24 שעות)
        ini_set('session.gc_maxlifetime', SESSION_LIFETIME);
        
        // תיקייה ייעודית לסשנים (מונע מחיקה בשרת משותף)
        $sessionPath = __DIR__ . '/../sessions';
        if (!file_exists($sessionPath)) {
            @mkdir($sessionPath, 0700, true);
        }
        if (is_writable($sessionPath)) {
            ini_set('session.save_path', $sessionPath);
        }
        
        // הגדרת העוגיה בצד הלקוח
        session_set_cookie_params([
            'lifetime' => SESSION_LIFETIME,
            'path' => '/',
            'secure' => SECURE_COOKIES,
            'httponly' => true,
            'samesite' => 'Strict'
        ]);
        
        session_start();
        
        // בדיקה ידנית של זמן הסשן (למקרה שהשרת מוחק sessions)
        if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity'] > SESSION_LIFETIME)) {
            // סשן פג תוקף - התנתק
            session_unset();
            session_destroy();
            session_start();
        }
        $_SESSION['last_activity'] = time(); // עדכון זמן פעילות אחרון
    }
}

// בדיקה אם המשתמש מחובר
function isLoggedIn() {
    startSecureSession();
    return isset($_SESSION['agent_id']) && !empty($_SESSION['agent_id']);
}

// קבלת פרטי המשתמש המחובר
function getCurrentAgent() {
    if (!isLoggedIn()) {
        return null;
    }
    
    $pdo = getDB();
    $stmt = $pdo->prepare("SELECT id, username, full_name, role FROM agents WHERE id = ?");
    $stmt->execute([$_SESSION['agent_id']]);
    return $stmt->fetch();
}

// התחברות
function login($username, $password) {
    $pdo = getDB();
    $stmt = $pdo->prepare("SELECT id, username, password, full_name, role FROM agents WHERE username = ?");
    $stmt->execute([$username]);
    $agent = $stmt->fetch();
    
    if ($agent && password_verify($password, $agent['password'])) {
        startSecureSession();
        session_regenerate_id(true);
        $_SESSION['agent_id'] = $agent['id'];
        $_SESSION['agent_name'] = $agent['full_name'];
        $_SESSION['agent_role'] = $agent['role'];
        return true;
    }
    
    return false;
}

// התנתקות
function logout() {
    startSecureSession();
    $_SESSION = [];
    session_destroy();
}

// הפניה אם לא מחובר
function requireLogin() {
    if (!isLoggedIn()) {
        header('Location: login.php');
        exit;
    }
}

// חסימת גישה לרואי חשבון - מפנה אותם לדוח שלהם
function blockAccountant() {
    $agent = getCurrentAgent();
    if ($agent && $agent['role'] === 'accountant') {
        header('Location: reports.php?report=accountant');
        exit;
    }
}

// הצפנת סיסמה חדשה
function hashPassword($password) {
    return password_hash($password, PASSWORD_BCRYPT, ['cost' => 10]);
}
