<?php
/**
 * Invoice4U SOAP API Integration
 * הפקת קבלות אוטומטית
 */

class Invoice4U {
    private $token;
    private $wsdlUrl = 'https://api.invoice4u.co.il/Services/ApiService.svc?singleWsdl';
    private $client;
    
    // סוגי מסמכים
    const DOC_INVOICE = 1;           // חשבונית מס
    const DOC_RECEIPT = 2;           // קבלה
    const DOC_INVOICE_RECEIPT = 3;   // חשבונית מס קבלה
    const DOC_CREDIT = 4;            // זיכוי
    const DOC_PROFORMA = 5;          // חשבון עסקה
    const DOC_ORDER = 6;             // הזמנה
    const DOC_QUOTE = 7;             // הצעת מחיר
    
    // סוגי תשלום
    const PAY_CREDIT_CARD = 1;
    const PAY_CHECK = 2;
    const PAY_TRANSFER = 3;
    const PAY_CASH = 4;
    const PAY_CREDIT = 5;
    const PAY_OTHER = 7;
    const PAY_BIT = 8;
    
    public function __construct($token = null) {
        $this->token = $token ?: '835ed34c-30bb-499b-8dd7-1460dba1343b';
    }
    
    /**
     * יצירת חיבור SOAP
     */
    private function getClient() {
        if (!$this->client) {
            try {
                $options = [
                    'trace' => true,
                    'exceptions' => true,
                    'cache_wsdl' => WSDL_CACHE_NONE,
                    'connection_timeout' => 30,
                    'stream_context' => stream_context_create([
                        'ssl' => [
                            'verify_peer' => false,
                            'verify_peer_name' => false
                        ]
                    ])
                ];
                $this->client = new SoapClient($this->wsdlUrl, $options);
            } catch (Exception $e) {
                $this->log("SOAP Error: " . $e->getMessage());
                return null;
            }
        }
        return $this->client;
    }
    
    /**
     * בדיקת תקינות Token
     */
    public function isAuthenticated() {
        $client = $this->getClient();
        if (!$client) {
            return ['success' => false, 'error' => 'Could not create SOAP client'];
        }
        
        try {
            $result = $client->IsAuthenticated(['token' => $this->token]);
            $this->log("IsAuthenticated Response: " . json_encode($result));
            
            if (isset($result->IsAuthenticatedResult)) {
                return [
                    'success' => true,
                    'user' => $result->IsAuthenticatedResult
                ];
            }
            return ['success' => false, 'error' => 'Invalid token'];
            
        } catch (SoapFault $e) {
            $this->log("IsAuthenticated Error: " . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * יצירת קבלה (Receipt)
     */
    public function createReceipt($data) {
        return $this->createDocument($data, self::DOC_RECEIPT);
    }
    
    /**
     * יצירת חשבונית מס קבלה (Invoice Receipt)
     */
    public function createInvoiceReceipt($data) {
        return $this->createDocument($data, self::DOC_INVOICE_RECEIPT);
    }
    
    /**
     * יצירת מסמך
     */
    public function createDocument($data, $documentType = self::DOC_RECEIPT) {
        $client = $this->getClient();
        if (!$client) {
            return ['success' => false, 'error' => 'Could not create SOAP client'];
        }
        
        try {
            // בניית אובייקט המסמך
            $doc = [
                'DocumentType' => $documentType,
                'Currency' => 'ILS',
                'TaxIncluded' => true,
                'TaxPercentage' => 18,
                'Subject' => $data['description'] ?? 'תשלום עבור שירותי תיירות',
                'Language' => 1, // Hebrew
            ];
            
            // לקוח - חובה להשתמש ב-ClientID לקבלות
            if (!empty($data['client_id'])) {
                $doc['ClientID'] = $data['client_id'];
            } else {
                // חיפוש או יצירת לקוח
                $customerName = $data['customer_name'] ?? 'לקוח כללי';
                $customerEmail = $data['customer_email'] ?? '';
                $customerPhone = $data['customer_phone'] ?? '';
                $customerId = $data['customer_id'] ?? '';
                
                $clientId = $this->getOrCreateCustomer($customerName, $customerEmail, $customerPhone, $customerId);
                
                if ($clientId) {
                    $doc['ClientID'] = $clientId;
                } else {
                    return ['success' => false, 'error' => 'Could not create or find customer'];
                }
            }
            
            // פריטים
            $doc['Items'] = [
                'DocumentItem' => [
                    'Code' => '001',
                    'Name' => $data['description'] ?? 'תשלום עבור שירותי תיירות',
                    'Quantity' => 1,
                    'Price' => floatval($data['amount'] ?? 0)
                ]
            ];
            
            // תשלומים (נדרש לקבלה ולחשבונית מס קבלה)
            if ($documentType == self::DOC_RECEIPT || $documentType == self::DOC_INVOICE_RECEIPT) {
                $payment = [
                    'PaymentType' => $this->getPaymentType($data['payment_type'] ?? 'credit_card'),
                    'Amount' => floatval($data['amount'] ?? 0),
                    'Date' => date('c') // ISO 8601 format
                ];
                
                // פרטי כרטיס אשראי
                if ($payment['PaymentType'] == self::PAY_CREDIT_CARD) {
                    $cardName = $data['card_brand'] ?? 'Visa';
                    if (!empty($data['last_4_digits'])) {
                        $cardName .= ' ****' . $data['last_4_digits'];
                    }
                    $payment['CreditCardName'] = $cardName;
                    $payment['PaymentNumber'] = $data['approval_number'] ?? '';
                    $payment['NumberOfPayments'] = intval($data['installments'] ?? 1);
                    $payment['ExpirationDate'] = '';
                }
                
                $doc['Payments'] = ['Payment' => $payment];
            }
            
            // אימיילים - גם לשיוך וגם לשליחה
            $emails = [];
            if (!empty($data['customer_email'])) {
                // שליחה ללקוח
                $emails[] = [
                    'Mail' => $data['customer_email'],
                    'IsUserMail' => false
                ];
            }
            // שליחה גם למייל החברה (עותק)
            $emails[] = [
                'Mail' => 'hufshabemivtza@gmail.com',
                'IsUserMail' => true
            ];
            
            if (!empty($emails)) {
                $doc['AssociatedEmails'] = ['AssociatedEmail' => $emails];
                $doc['SendByEmail'] = true; // שליחת המסמך במייל ללקוח
            }
            
            // מזהה ייחודי
            $doc['ApiIdentifier'] = $data['api_identifier'] ?? $this->generateGuid();
            
            $this->log("CreateDocument Request: " . json_encode($doc, JSON_UNESCAPED_UNICODE));
            
            // שליחת הבקשה
            $result = $client->CreateDocument([
                'doc' => $doc,
                'token' => $this->token
            ]);
            
            $this->log("CreateDocument Response: " . json_encode($result));
            
            if (isset($result->CreateDocumentResult)) {
                $docResult = $result->CreateDocumentResult;
                
                // בדיקת שגיאות - רק אם יש שגיאות ממשיות
                if (!empty($docResult->Errors) && isset($docResult->Errors->CommonError)) {
                    $error = $docResult->Errors->CommonError->Error ?? 'Unknown error';
                    return ['success' => false, 'error' => $error, 'raw' => $docResult];
                }
                
                // בדיקת הצלחה - לפי מספר מסמך או קישור PDF
                if (!empty($docResult->DocumentNumber) || !empty($docResult->PrintOriginalPDFLink)) {
                    return [
                        'success' => true,
                        'document_number' => $docResult->DocumentNumber ?? '',
                        'document_id' => $docResult->ID ?? '',
                        'total' => $docResult->Total ?? 0,
                        'pdf_link' => $docResult->PrintOriginalPDFLink ?? '',
                        'pdf_copy_link' => $docResult->PrintCertifiedCopyPDFLink ?? '',
                        'client_id' => $docResult->ClientID ?? '',
                        'client_name' => $docResult->ClientName ?? '',
                        'raw' => $docResult
                    ];
                }
                
                return ['success' => false, 'error' => 'Document creation failed', 'raw' => $docResult];
            }
            
            return ['success' => false, 'error' => 'No result returned'];
            
        } catch (SoapFault $e) {
            $this->log("CreateDocument Error: " . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * קבלת סוג תשלום
     */
    private function getPaymentType($type) {
        $types = [
            'credit_card' => self::PAY_CREDIT_CARD,
            'check' => self::PAY_CHECK,
            'transfer' => self::PAY_TRANSFER,
            'cash' => self::PAY_CASH,
            'credit' => self::PAY_CREDIT,
            'bit' => self::PAY_BIT,
            'other' => self::PAY_OTHER
        ];
        return $types[$type] ?? self::PAY_CREDIT_CARD;
    }
    
    /**
     * יצירת GUID
     */
    private function generateGuid() {
        return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff), mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
        );
    }
    
    /**
     * בדיקת חיבור
     */
    public function testConnection() {
        return $this->isAuthenticated();
    }
    
    /**
     * חיפוש או יצירת לקוח
     */
    public function getOrCreateCustomer($name, $email = '', $phone = '', $uniqueId = '') {
        $client = $this->getClient();
        if (!$client) {
            return null;
        }
        
        try {
            // חיפוש לקוח קיים לפי שם
            $searchCustomer = ['Name' => $name, 'Active' => true];
            $result = $client->GetCustomers([
                'cust' => $searchCustomer,
                'token' => $this->token
            ]);
            
            $this->log("GetCustomers Response: " . json_encode($result));
            
            if (isset($result->GetCustomersResult->Response->Customer)) {
                $customers = $result->GetCustomersResult->Response->Customer;
                $existingCustomer = is_array($customers) ? $customers[0] : $customers;
                $existingId = $existingCustomer->ID;
                
                // עדכון המייל של הלקוח הקיים אם יש מייל חדש
                if (!empty($email)) {
                    try {
                        $updateCustomer = [
                            'ID' => $existingId,
                            'Name' => $name,
                            'Email' => $email,
                            'Phone' => '', // מנקה טלפון
                            'Active' => true
                        ];
                        $client->UpdateCustomer([
                            'cu' => $updateCustomer,
                            'token' => $this->token
                        ]);
                        $this->log("Updated customer $existingId with email: $email");
                    } catch (Exception $e) {
                        $this->log("Update customer email failed: " . $e->getMessage());
                    }
                }
                
                return $existingId;
            }
            
            // לקוח לא קיים - צור חדש
            $newCustomer = [
                'Name' => $name,
                'Active' => true
            ];
            
            // הוסף מייל רק אם קיים
            if (!empty($email)) {
                $newCustomer['Email'] = $email;
            }
            
            // הוסף טלפון רק אם קיים
            if (!empty($phone)) {
                $newCustomer['Phone'] = $phone;
            }
            
            // הוסף מזהה ייחודי רק אם קיים
            if (!empty($uniqueId)) {
                $newCustomer['UniqueID'] = $uniqueId;
            }
            
            $createResult = $client->CreateCustomer([
                'cu' => $newCustomer,
                'token' => $this->token
            ]);
            
            $this->log("CreateCustomer Response: " . json_encode($createResult));
            
            if (isset($createResult->CreateCustomerResult->ID)) {
                return $createResult->CreateCustomerResult->ID;
            }
            
            return null;
            
        } catch (SoapFault $e) {
            $this->log("Customer Error: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * לוג
     */
    private function log($message) {
        $logDir = __DIR__ . '/../logs';
        if (!is_dir($logDir)) @mkdir($logDir, 0755, true);
        
        $logFile = $logDir . '/invoice4u_' . date('Y-m-d') . '.log';
        @file_put_contents($logFile, date('Y-m-d H:i:s') . " - $message\n", FILE_APPEND);
    }
}
?>