<?php
/**
 * Invoice4U API Integration
 * הפקת קבלות אוטומטית
 */

class Invoice4U {
    private $apiKey;
    private $httpPostUrl = 'https://www.invoice4u.co.il/public/HttpPost.aspx';
    
    public function __construct($apiKey = null) {
        $this->apiKey = $apiKey ?: '835ed34c-30bb-499b-8dd7-1460dba1343b';
    }
    
    /**
     * יצירת קבלה (Receipt)
     */
    public function createReceipt($data) {
        $postData = [
            'TransType' => 'R:CREATE',
            'ApiKey' => $this->apiKey,
            
            // פרטי לקוח
            'ClientName' => $data['customer_name'] ?? 'לקוח',
            'ClientEmail' => $data['customer_email'] ?? '',
            'ClientPhone' => $data['customer_phone'] ?? '',
            
            // פרטי פריט
            'InvoiceItemCode' => '001',
            'InvoiceItemDescription' => $data['description'] ?? 'תשלום עבור שירותי תיירות',
            'InvoiceItemPrice' => floatval($data['amount'] ?? 0),
            'InvoiceItemQuantity' => 1,
            
            // פרטי תשלום
            'PaymentType' => $this->getPaymentType($data['payment_method'] ?? 'credit_card'),
            'PaymentSum' => floatval($data['amount'] ?? 0),
            'PaymentCardLastDigits' => $data['last_4_digits'] ?? '',
            'PaymentCardApprovalNumber' => $data['approval_number'] ?? '',
            'PaymentNumOfPayments' => intval($data['installments'] ?? 1),
            
            // שליחה במייל
            'SendEmail' => !empty($data['customer_email']) ? 'true' : 'false'
        ];
        
        return $this->sendHttpPost($postData);
    }
    
    /**
     * יצירת חשבונית מס קבלה
     */
    public function createTaxInvoiceReceipt($data) {
        $postData = [
            'TransType' => 'IR:CREATE',
            'ApiKey' => $this->apiKey,
            
            'ClientName' => $data['customer_name'] ?? 'לקוח',
            'ClientEmail' => $data['customer_email'] ?? '',
            'ClientPhone' => $data['customer_phone'] ?? '',
            
            'InvoiceItemCode' => '001',
            'InvoiceItemDescription' => $data['description'] ?? 'תשלום עבור שירותי תיירות',
            'InvoiceItemPrice' => floatval($data['amount'] ?? 0),
            'InvoiceItemQuantity' => 1,
            
            'PaymentType' => $this->getPaymentType($data['payment_method'] ?? 'credit_card'),
            'PaymentSum' => floatval($data['amount'] ?? 0),
            'PaymentCardLastDigits' => $data['last_4_digits'] ?? '',
            'PaymentCardApprovalNumber' => $data['approval_number'] ?? '',
            'PaymentNumOfPayments' => intval($data['installments'] ?? 1),
            
            'SendEmail' => !empty($data['customer_email']) ? 'true' : 'false'
        ];
        
        return $this->sendHttpPost($postData);
    }
    
    /**
     * סוג תשלום
     */
    private function getPaymentType($method) {
        $types = [
            'credit_card' => 1,
            'cash' => 2,
            'check' => 3,
            'transfer' => 4,
            'other' => 10
        ];
        return $types[$method] ?? 1;
    }
    
    /**
     * שליחת HTTP POST
     */
    private function sendHttpPost($data) {
        $postFields = http_build_query($data);
        
        $ch = curl_init();
        
        curl_setopt_array($ch, [
            CURLOPT_URL => $this->httpPostUrl,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $postFields,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_FOLLOWLOCATION => true,  // עקוב אחרי redirects
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/x-www-form-urlencoded; charset=utf-8'
            ]
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $finalUrl = curl_getinfo($ch, CURLINFO_EFFECTIVE_URL);
        $error = curl_error($ch);
        curl_close($ch);
        
        // לוג
        $this->log("URL: " . $this->httpPostUrl . " -> $finalUrl");
        $this->log("Request: " . json_encode($data, JSON_UNESCAPED_UNICODE));
        $this->log("Response ($httpCode): " . $response);
        
        if ($error) {
            return ['success' => false, 'error' => $error];
        }
        
        return $this->parseResponse($response);
    }
    
    /**
     * פרסור תגובה
     */
    private function parseResponse($response) {
        // נסה לפרסר כ-query string
        parse_str($response, $parsed);
        
        if (!empty($parsed) && isset($parsed['Status'])) {
            $success = strtoupper($parsed['Status'] ?? '') === 'OK' ||
                      ($parsed['ResponseCode'] ?? '') === '0';
            
            return [
                'success' => $success,
                'document_number' => $parsed['DocumentNumber'] ?? $parsed['DocNumber'] ?? '',
                'document_url' => $parsed['DocumentURL'] ?? $parsed['DocURL'] ?? '',
                'status' => $parsed['Status'] ?? '',
                'message' => $parsed['Message'] ?? $parsed['ErrorMessage'] ?? '',
                'raw' => $response
            ];
        }
        
        // אולי זה XML
        if (strpos($response, '<?xml') !== false || strpos($response, '<') === 0) {
            return $this->parseXmlResponse($response);
        }
        
        return [
            'success' => false,
            'error' => 'Could not parse response',
            'raw' => $response
        ];
    }
    
    /**
     * פרסור XML
     */
    private function parseXmlResponse($response) {
        try {
            $xml = @simplexml_load_string($response);
            if ($xml) {
                return [
                    'success' => true,
                    'document_number' => (string)($xml->DocumentNumber ?? $xml->DocNumber ?? ''),
                    'document_url' => (string)($xml->DocumentURL ?? ''),
                    'raw' => $response
                ];
            }
        } catch (Exception $e) {
            // ignore
        }
        
        return ['success' => false, 'error' => 'XML parse failed', 'raw' => $response];
    }
    
    /**
     * לוג
     */
    private function log($message) {
        $logDir = __DIR__ . '/../logs';
        if (!is_dir($logDir)) @mkdir($logDir, 0755, true);
        
        $logFile = $logDir . '/invoice4u_' . date('Y-m-d') . '.log';
        @file_put_contents($logFile, date('Y-m-d H:i:s') . " - $message\n", FILE_APPEND);
    }
    
    /**
     * בדיקת חיבור
     */
    public function testConnection() {
        return ['success' => true, 'message' => 'Use createReceipt to test'];
    }
}
?>