<?php
// ניהול הזמנות מתקדם
require_once 'includes/auth.php';
requireLogin();

$agent = getCurrentAgent();
$pdo = getDB();

// קבלת פרמטרים
$dateType = $_GET['date_type'] ?? 'departure'; // departure, return, order, created
$dateFrom = $_GET['date_from'] ?? '';
$dateTo = $_GET['date_to'] ?? '';
$quickFilter = $_GET['quick'] ?? '';
$agentFilter = $_GET['agent'] ?? '';
$statusFilter = $_GET['status'] ?? '';
$search = $_GET['search'] ?? '';

// חישוב תאריכים לפילטרים מהירים
$today = date('Y-m-d');
$tomorrow = date('Y-m-d', strtotime('+1 day'));
$dayAfter = date('Y-m-d', strtotime('+2 days'));
$weekEnd = date('Y-m-d', strtotime('+7 days'));
$monthEnd = date('Y-m-d', strtotime('+30 days'));
$thisMonthStart = date('Y-m-01');
$thisMonthEnd = date('Y-m-t');

// טיפול בפילטרים מהירים
if ($quickFilter) {
    switch ($quickFilter) {
        case 'today':
            $dateFrom = $today;
            $dateTo = $today;
            break;
        case 'tomorrow':
            $dateFrom = $tomorrow;
            $dateTo = $tomorrow;
            break;
        case 'day_after':
            $dateFrom = $dayAfter;
            $dateTo = $dayAfter;
            break;
        case 'this_week':
            $dateFrom = $today;
            $dateTo = $weekEnd;
            break;
        case 'this_month':
            $dateFrom = $thisMonthStart;
            $dateTo = $thisMonthEnd;
            break;
        case 'next_month':
            $dateFrom = date('Y-m-01', strtotime('+1 month'));
            $dateTo = date('Y-m-t', strtotime('+1 month'));
            break;
    }
}

// מיפוי סוג תאריך לעמודה
$dateColumns = [
    'departure' => 'departure_date',
    'return' => 'return_date',
    'order' => 'order_date',
    'created' => 'DATE(created_at)'
];
$dateColumn = $dateColumns[$dateType] ?? 'departure_date';

// בניית השאילתא
$where = [];
$params = [];

if ($dateFrom) {
    $where[] = "$dateColumn >= ?";
    $params[] = $dateFrom;
}
if ($dateTo) {
    $where[] = "$dateColumn <= ?";
    $params[] = $dateTo;
}
if ($agentFilter) {
    $where[] = "o.agent_id = ?";
    $params[] = $agentFilter;
}
if ($statusFilter) {
    $where[] = "o.status = ?";
    $params[] = $statusFilter;
}
if ($search) {
    $where[] = "(o.destination LIKE ? OR o.customer_phone LIKE ? OR o.hotel_name LIKE ? OR o.id = ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = $search;
}

$whereClause = $where ? 'WHERE ' . implode(' AND ', $where) : '';

// שליפת ההזמנות
$sql = "SELECT o.*, a.full_name as agent_name 
        FROM orders o 
        LEFT JOIN agents a ON o.agent_id = a.id 
        $whereClause 
        ORDER BY $dateColumn ASC, o.created_at DESC";
$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$orders = $stmt->fetchAll();

// קבלת רשימת נציגים
$agents = $pdo->query("SELECT id, full_name FROM agents ORDER BY full_name")->fetchAll();

// סטטוסים
$statuses = [
    'draft' => ['name' => 'טיוטה', 'color' => '#6c757d'],
    'sent' => ['name' => 'נשלח', 'color' => '#17a2b8'],
    'paid' => ['name' => 'שולם', 'color' => '#28a745'],
    'confirmed' => ['name' => 'אושר', 'color' => '#007bff'],
    'cancelled' => ['name' => 'בוטל', 'color' => '#dc3545']
];

// סוגי תאריכים
$dateTypes = [
    'departure' => 'תאריך יציאה',
    'return' => 'תאריך חזרה',
    'order' => 'תאריך הזמנה',
    'created' => 'תאריך יצירה'
];

// חישוב סטטיסטיקות
$totalOrders = count($orders);
$totalRevenue = array_sum(array_column($orders, 'total_price'));
$totalProfit = array_sum(array_column($orders, 'profit'));

// פונקציה לחישוב ימים עד יציאה
function daysUntil($date) {
    if (!$date) return null;
    $diff = (strtotime($date) - strtotime(date('Y-m-d'))) / 86400;
    return (int)$diff;
}

// תווית לפילטר הנוכחי
$currentFilterLabel = '';
if ($quickFilter) {
    $quickLabels = [
        'today' => 'היום',
        'tomorrow' => 'מחר',
        'day_after' => 'מחרתיים',
        'this_week' => 'השבוע הקרוב',
        'this_month' => 'החודש',
        'next_month' => 'החודש הבא'
    ];
    $currentFilterLabel = $quickLabels[$quickFilter] ?? '';
} elseif ($dateFrom || $dateTo) {
    $currentFilterLabel = ($dateFrom ? date('d/m', strtotime($dateFrom)) : '') . 
                         ' - ' . 
                         ($dateTo ? date('d/m', strtotime($dateTo)) : '');
}
?>
<!DOCTYPE html>
<html lang="he" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ניהול הזמנות | CRM</title>
    <style>
        * { box-sizing: border-box; margin: 0; padding: 0; }
        body {
            font-family: 'Segoe UI', Tahoma, Arial, sans-serif;
            background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
            min-height: 100vh;
            padding: 20px;
        }
        .container { max-width: 1400px; margin: 0 auto; }
        
        /* Header */
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            flex-wrap: wrap;
            gap: 10px;
        }
        .header h1 { color: white; font-size: 1.8rem; }
        .header-actions { display: flex; gap: 10px; }
        .header-btn {
            padding: 10px 20px;
            border-radius: 8px;
            text-decoration: none;
            font-weight: 500;
            border: none;
            cursor: pointer;
        }
        .btn-primary { background: #667eea; color: white; }
        .btn-secondary { background: #6c757d; color: white; }
        
        /* Quick Filters */
        .quick-filters {
            background: white;
            padding: 15px 20px;
            border-radius: 10px;
            margin-bottom: 15px;
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
            align-items: center;
        }
        .quick-filters label {
            font-weight: 600;
            color: #333;
            margin-left: 10px;
        }
        .quick-btn {
            padding: 8px 16px;
            border: 2px solid #e0e0e0;
            border-radius: 20px;
            background: white;
            cursor: pointer;
            font-size: 0.9rem;
            transition: all 0.2s;
        }
        .quick-btn:hover { border-color: #667eea; color: #667eea; }
        .quick-btn.active { background: #667eea; color: white; border-color: #667eea; }
        
        /* Advanced Filters */
        .filters-panel {
            background: white;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 15px;
        }
        .filters-row {
            display: flex;
            gap: 15px;
            flex-wrap: wrap;
            align-items: flex-end;
        }
        .filter-group {
            display: flex;
            flex-direction: column;
            gap: 5px;
        }
        .filter-group label {
            font-size: 0.85rem;
            color: #666;
            font-weight: 500;
        }
        .filter-group select,
        .filter-group input {
            padding: 10px 15px;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            font-size: 0.95rem;
            min-width: 150px;
        }
        .filter-group select:focus,
        .filter-group input:focus {
            border-color: #667eea;
            outline: none;
        }
        .btn-search {
            padding: 10px 25px;
            background: #667eea;
            color: white;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-size: 0.95rem;
            font-weight: 500;
        }
        .btn-clear {
            padding: 10px 20px;
            background: #f8f9fa;
            color: #666;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            cursor: pointer;
            text-decoration: none;
            font-size: 0.95rem;
        }
        
        /* Stats */
        .stats-bar {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(180px, 1fr));
            gap: 15px;
            margin-bottom: 15px;
        }
        .stat-card {
            background: white;
            padding: 20px;
            border-radius: 10px;
            text-align: center;
        }
        .stat-value { font-size: 1.8rem; font-weight: bold; color: #333; }
        .stat-label { font-size: 0.85rem; color: #666; margin-top: 5px; }
        .stat-card.purple .stat-value { color: #667eea; }
        .stat-card.green .stat-value { color: #28a745; }
        .stat-card.blue .stat-value { color: #17a2b8; }
        
        /* Current Filter Badge */
        .current-filter {
            background: #e8f4fd;
            color: #1976d2;
            padding: 8px 15px;
            border-radius: 20px;
            font-size: 0.9rem;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }
        .current-filter .label { font-weight: 600; }
        
        /* Table */
        .table-container {
            background: white;
            border-radius: 10px;
            overflow: hidden;
        }
        .table-header {
            padding: 15px 20px;
            border-bottom: 1px solid #eee;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .table-header h2 { font-size: 1.1rem; color: #333; }
        table {
            width: 100%;
            border-collapse: collapse;
        }
        th, td {
            padding: 12px 15px;
            text-align: right;
            border-bottom: 1px solid #eee;
        }
        th {
            background: #f8f9fa;
            font-weight: 600;
            color: #333;
            font-size: 0.9rem;
        }
        tr:hover { background: #f8f9fa; }
        
        /* Status Badge */
        .status-badge {
            padding: 4px 12px;
            border-radius: 15px;
            color: white;
            font-size: 0.8rem;
            font-weight: 500;
        }
        
        /* Days Badge */
        .days-badge {
            padding: 4px 10px;
            border-radius: 10px;
            font-size: 0.8rem;
            font-weight: 600;
        }
        .days-today { background: #ff6b6b; color: white; }
        .days-tomorrow { background: #ffa94d; color: white; }
        .days-soon { background: #ffd43b; color: #333; }
        .days-week { background: #69db7c; color: #333; }
        .days-later { background: #e9ecef; color: #666; }
        .days-past { background: #dee2e6; color: #666; }
        
        /* Action Links */
        .action-link {
            color: #667eea;
            text-decoration: none;
            margin-left: 10px;
        }
        .action-link:hover { text-decoration: underline; }
        
        /* Sortable Headers */
        th.sortable {
            cursor: pointer;
            user-select: none;
            position: relative;
            padding-left: 25px;
        }
        th.sortable:hover {
            background: #e9ecef;
        }
        th.sortable::after {
            content: '⇅';
            position: absolute;
            left: 8px;
            opacity: 0.3;
            font-size: 0.8rem;
        }
        th.sortable.asc::after {
            content: '↑';
            opacity: 1;
            color: #667eea;
        }
        th.sortable.desc::after {
            content: '↓';
            opacity: 1;
            color: #667eea;
        }
        
        /* Order ID */
        .order-id {
            font-weight: 600;
            color: #333;
        }
        .order-id a { color: #667eea; text-decoration: none; }
        .order-id a:hover { text-decoration: underline; }
        
        /* Amount */
        .amount { font-weight: 600; color: #28a745; }
        .profit { color: #17a2b8; }
        
        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: #666;
        }
        .empty-state h3 { margin-bottom: 10px; color: #333; }
        
        /* Responsive */
        @media (max-width: 768px) {
            .filters-row { flex-direction: column; }
            .filter-group { width: 100%; }
            .filter-group select,
            .filter-group input { width: 100%; }
            table { font-size: 0.85rem; }
            th, td { padding: 8px 10px; }
        }
    </style>
</head>
<body>
    <div class="container">
        <!-- Header -->
        <div class="header">
            <h1>📋 ניהול הזמנות</h1>
            <div class="header-actions">
                <a href="complete_order.php" class="header-btn btn-primary">+ הזמנה חדשה</a>
                <a href="orders_new.php" class="header-btn btn-secondary">תצוגה רגילה</a>
                <a href="index.php" class="header-btn btn-secondary">דשבורד</a>
            </div>
        </div>

        <!-- Quick Filters -->
        <div class="quick-filters">
            <label>🚀 מהיר:</label>
            <a href="?quick=today&date_type=<?= $dateType ?>" class="quick-btn <?= $quickFilter === 'today' ? 'active' : '' ?>">היום</a>
            <a href="?quick=tomorrow&date_type=<?= $dateType ?>" class="quick-btn <?= $quickFilter === 'tomorrow' ? 'active' : '' ?>">מחר</a>
            <a href="?quick=day_after&date_type=<?= $dateType ?>" class="quick-btn <?= $quickFilter === 'day_after' ? 'active' : '' ?>">מחרתיים</a>
            <a href="?quick=this_week&date_type=<?= $dateType ?>" class="quick-btn <?= $quickFilter === 'this_week' ? 'active' : '' ?>">7 ימים הקרובים</a>
            <a href="?quick=this_month&date_type=<?= $dateType ?>" class="quick-btn <?= $quickFilter === 'this_month' ? 'active' : '' ?>">החודש</a>
            <a href="?quick=next_month&date_type=<?= $dateType ?>" class="quick-btn <?= $quickFilter === 'next_month' ? 'active' : '' ?>">החודש הבא</a>
            <a href="?" class="quick-btn">הכל</a>
        </div>

        <!-- Advanced Filters -->
        <form method="GET" class="filters-panel">
            <div class="filters-row">
                <div class="filter-group">
                    <label>סנן לפי:</label>
                    <select name="date_type">
                        <?php foreach ($dateTypes as $key => $label): ?>
                            <option value="<?= $key ?>" <?= $dateType === $key ? 'selected' : '' ?>><?= $label ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="filter-group">
                    <label>מתאריך:</label>
                    <input type="date" name="date_from" value="<?= $dateFrom ?>">
                </div>
                
                <div class="filter-group">
                    <label>עד תאריך:</label>
                    <input type="date" name="date_to" value="<?= $dateTo ?>">
                </div>
                
                <div class="filter-group">
                    <label>סוכן:</label>
                    <select name="agent">
                        <option value="">כל הסוכנים</option>
                        <?php foreach ($agents as $a): ?>
                            <option value="<?= $a['id'] ?>" <?= $agentFilter == $a['id'] ? 'selected' : '' ?>>
                                <?= htmlspecialchars($a['full_name']) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="filter-group">
                    <label>סטטוס:</label>
                    <select name="status">
                        <option value="">כל הסטטוסים</option>
                        <?php foreach ($statuses as $key => $s): ?>
                            <option value="<?= $key ?>" <?= $statusFilter === $key ? 'selected' : '' ?>><?= $s['name'] ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="filter-group">
                    <label>חיפוש:</label>
                    <input type="text" name="search" value="<?= htmlspecialchars($search) ?>" placeholder="יעד, טלפון, מלון...">
                </div>
                
                <button type="submit" class="btn-search">🔍 חפש</button>
                <a href="?" class="btn-clear">נקה</a>
            </div>
        </form>

        <!-- Stats -->
        <div class="stats-bar">
            <div class="stat-card purple">
                <div class="stat-value"><?= $totalOrders ?></div>
                <div class="stat-label">הזמנות נמצאו</div>
            </div>
            <div class="stat-card green">
                <div class="stat-value">₪<?= number_format($totalRevenue) ?></div>
                <div class="stat-label">סה"כ מכירות</div>
            </div>
            <div class="stat-card blue">
                <div class="stat-value">₪<?= number_format($totalProfit) ?></div>
                <div class="stat-label">סה"כ רווח</div>
            </div>
            <?php if ($currentFilterLabel): ?>
            <div class="stat-card">
                <div class="current-filter">
                    <span class="label"><?= $dateTypes[$dateType] ?>:</span>
                    <span><?= $currentFilterLabel ?></span>
                </div>
            </div>
            <?php endif; ?>
        </div>

        <!-- Orders Table -->
        <div class="table-container">
            <div class="table-header">
                <h2>רשימת הזמנות</h2>
            </div>
            
            <?php if (empty($orders)): ?>
                <div class="empty-state">
                    <h3>🔍 לא נמצאו הזמנות</h3>
                    <p>נסה לשנות את הפילטרים או לחפש משהו אחר</p>
                </div>
            <?php else: ?>
                <table id="ordersTable">
                    <thead>
                        <tr>
                            <th class="sortable" data-sort="number" data-col="0">#</th>
                            <th class="sortable" data-sort="string" data-col="1">יעד</th>
                            <th class="sortable" data-sort="date" data-col="2">תאריך יציאה</th>
                            <th class="sortable" data-sort="number" data-col="3">ימים ליציאה</th>
                            <th class="sortable" data-sort="date" data-col="4">תאריך חזרה</th>
                            <th class="sortable" data-sort="string" data-col="5">מלון</th>
                            <th class="sortable" data-sort="number" data-col="6">סה"כ</th>
                            <th class="sortable" data-sort="number" data-col="7">רווח</th>
                            <th class="sortable" data-sort="string" data-col="8">סוכן</th>
                            <th class="sortable" data-sort="string" data-col="9">סטטוס</th>
                            <th>פעולות</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($orders as $order): 
                            $days = daysUntil($order['departure_date']);
                            $daysClass = 'days-later';
                            $daysText = $days !== null ? $days . ' ימים' : '-';
                            
                            if ($days === null) {
                                $daysClass = '';
                                $daysText = '-';
                            } elseif ($days < 0) {
                                $daysClass = 'days-past';
                                $daysText = 'עבר';
                            } elseif ($days === 0) {
                                $daysClass = 'days-today';
                                $daysText = 'היום!';
                            } elseif ($days === 1) {
                                $daysClass = 'days-tomorrow';
                                $daysText = 'מחר!';
                            } elseif ($days === 2) {
                                $daysClass = 'days-soon';
                                $daysText = 'מחרתיים';
                            } elseif ($days <= 7) {
                                $daysClass = 'days-week';
                                $daysText = $days . ' ימים';
                            }
                        ?>
                        <tr>
                            <td class="order-id" data-sort-value="<?= $order['id'] ?>">
                                <a href="view_order.php?id=<?= $order['id'] ?>">#<?= $order['id'] ?></a>
                            </td>
                            <td data-sort-value="<?= htmlspecialchars($order['destination'] ?? '') ?>"><strong><?= htmlspecialchars($order['destination'] ?? '-') ?></strong></td>
                            <td data-sort-value="<?= $order['departure_date'] ?? '9999-99-99' ?>"><?= $order['departure_date'] ? date('d/m/Y', strtotime($order['departure_date'])) : '-' ?></td>
                            <td data-sort-value="<?= $days ?? 9999 ?>">
                                <?php if ($daysClass): ?>
                                    <span class="days-badge <?= $daysClass ?>"><?= $daysText ?></span>
                                <?php else: ?>
                                    -
                                <?php endif; ?>
                            </td>
                            <td data-sort-value="<?= $order['return_date'] ?? '9999-99-99' ?>"><?= $order['return_date'] ? date('d/m/Y', strtotime($order['return_date'])) : '-' ?></td>
                            <td data-sort-value="<?= htmlspecialchars($order['hotel_name'] ?? '') ?>"><?= htmlspecialchars($order['hotel_name'] ?? '-') ?></td>
                            <td class="amount" data-sort-value="<?= $order['total_price'] ?? 0 ?>">₪<?= number_format($order['total_price'] ?? 0) ?></td>
                            <td class="profit" data-sort-value="<?= $order['profit'] ?? 0 ?>">₪<?= number_format($order['profit'] ?? 0) ?></td>
                            <td data-sort-value="<?= htmlspecialchars($order['agent_name'] ?? '') ?>"><?= htmlspecialchars($order['agent_name'] ?? '-') ?></td>
                            <td data-sort-value="<?= $order['status'] ?>">
                                <?php $st = $statuses[$order['status']] ?? ['name' => $order['status'], 'color' => '#999']; ?>
                                <span class="status-badge" style="background:<?= $st['color'] ?>">
                                    <?= $st['name'] ?>
                                </span>
                            </td>
                            <td>
                                <a href="view_order.php?id=<?= $order['id'] ?>" class="action-link">צפה</a>
                                <a href="complete_order.php?edit=<?= $order['id'] ?>" class="action-link">ערוך</a>
                                <a href="voucher.php?id=<?= $order['id'] ?>" class="action-link" target="_blank">וואוצ'ר</a>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
    </div>

    <script>
    document.addEventListener('DOMContentLoaded', function() {
        const table = document.getElementById('ordersTable');
        if (!table) return;
        
        const headers = table.querySelectorAll('th.sortable');
        let currentSort = { col: null, dir: 'asc' };
        
        headers.forEach(header => {
            header.addEventListener('click', function() {
                const col = parseInt(this.dataset.col);
                const sortType = this.dataset.sort;
                
                // Toggle direction
                if (currentSort.col === col) {
                    currentSort.dir = currentSort.dir === 'asc' ? 'desc' : 'asc';
                } else {
                    currentSort.col = col;
                    currentSort.dir = 'asc';
                }
                
                // Update header classes
                headers.forEach(h => h.classList.remove('asc', 'desc'));
                this.classList.add(currentSort.dir);
                
                // Sort the table
                sortTable(col, sortType, currentSort.dir);
            });
        });
        
        function sortTable(col, sortType, dir) {
            const tbody = table.querySelector('tbody');
            const rows = Array.from(tbody.querySelectorAll('tr'));
            
            rows.sort((a, b) => {
                const aVal = a.cells[col].dataset.sortValue || a.cells[col].textContent.trim();
                const bVal = b.cells[col].dataset.sortValue || b.cells[col].textContent.trim();
                
                let comparison = 0;
                
                if (sortType === 'number') {
                    const aNum = parseFloat(aVal) || 0;
                    const bNum = parseFloat(bVal) || 0;
                    comparison = aNum - bNum;
                } else if (sortType === 'date') {
                    comparison = aVal.localeCompare(bVal);
                } else {
                    comparison = aVal.localeCompare(bVal, 'he');
                }
                
                return dir === 'asc' ? comparison : -comparison;
            });
            
            // Re-append rows in sorted order
            rows.forEach(row => tbody.appendChild(row));
        }
    });
    </script>
</body>
</html>
