<?php
// פאנל ניהול הזמנות מקיף
require_once 'includes/auth.php';
requireLogin();
blockAccountant();

$pdo = getDB();
$error = '';
$orders = [];

// פרטי המשתמש המחובר
$currentAgent = getCurrentAgent();
$isAdmin = ($currentAgent && isset($currentAgent['role']) && $currentAgent['role'] === 'admin');

// פרמטרי סינון (לא צריכים DB)
$filters = [
    'order_id' => $_GET['order_id'] ?? '',
    'customer_name' => $_GET['customer_name'] ?? '',
    'destination' => $_GET['destination'] ?? '',
    'flight_supplier' => $_GET['flight_supplier'] ?? '',
    'flight_airline' => $_GET['flight_airline'] ?? '',
    'agent_id' => $_GET['agent_id'] ?? '',
    'payment_status' => $_GET['payment_status'] ?? '',
    'order_status' => $_GET['order_status'] ?? '',
    'order_type' => $_GET['order_type'] ?? '',
    'date_from' => $_GET['date_from'] ?? '',
    'date_to' => $_GET['date_to'] ?? '',
    'flight_from' => $_GET['flight_from'] ?? '',
    'flight_to' => $_GET['flight_to'] ?? '',
    'sort' => $_GET['sort'] ?? 'created_at',
    'order' => $_GET['order'] ?? 'DESC'
];

// אם נציג - כופה לראות רק את ההזמנות שלו
if (!$isAdmin) {
    $filters['agent_id'] = $currentAgent['id'];
}

try {

// בניית שאילתה
$sql = "SELECT o.*, a.full_name as agent_name
        FROM orders o
        LEFT JOIN agents a ON o.agent_id = a.id
        WHERE 1=1";
$params = [];

if ($filters['order_id']) {
    $sql .= " AND o.id = ?";
    $params[] = $filters['order_id'];
}
if ($filters['customer_name']) {
    $searchTerm = '%' . $filters['customer_name'] . '%';
    $normalizedSearch = '%' . preg_replace('/[-\s]/', '', $filters['customer_name']) . '%';
    
    $sql .= " AND (o.passengers LIKE ? OR o.customer_phone LIKE ? OR REPLACE(REPLACE(o.customer_phone, '-', ''), ' ', '') LIKE ? OR o.customer_email LIKE ?)";
    $params[] = $searchTerm;
    $params[] = $searchTerm;
    $params[] = $normalizedSearch;
    $params[] = $searchTerm;
}
if ($filters['destination']) {
    $sql .= " AND o.destination LIKE ?";
    $params[] = '%' . $filters['destination'] . '%';
}
if ($filters['flight_supplier']) {
    $sql .= " AND (o.flight_supplier = ? OR o.return_flight_supplier = ?)";
    $params[] = $filters['flight_supplier'];
    $params[] = $filters['flight_supplier'];
}
if ($filters['flight_airline']) {
    $sql .= " AND (o.airline_outbound LIKE ? OR o.airline_return LIKE ?)";
    $params[] = '%' . $filters['flight_airline'] . '%';
    $params[] = '%' . $filters['flight_airline'] . '%';
}
if ($filters['agent_id']) {
    $sql .= " AND o.agent_id = ?";
    $params[] = $filters['agent_id'];
}
if ($filters['order_status']) {
    $sql .= " AND o.status = ?";
    $params[] = $filters['order_status'];
}
if ($filters['date_from']) {
    $sql .= " AND DATE(o.created_at) >= ?";
    $params[] = $filters['date_from'];
}
if ($filters['date_to']) {
    $sql .= " AND DATE(o.created_at) <= ?";
    $params[] = $filters['date_to'];
}
if ($filters['flight_from']) {
    $sql .= " AND DATE(o.departure_date) >= ?";
    $params[] = $filters['flight_from'];
}
if ($filters['flight_to']) {
    $sql .= " AND DATE(o.departure_date) <= ?";
    $params[] = $filters['flight_to'];
}
if ($filters['order_type']) {
    $sql .= " AND o.order_type = ?";
    $params[] = $filters['order_type'];
}

// מיון
$allowedSorts = ['id', 'created_at', 'customer_name', 'destination', 'departure_date', 'return_date', 'total_price', 'profit'];
$sort = in_array($filters['sort'], $allowedSorts) ? $filters['sort'] : 'created_at';
$order = $filters['order'] === 'ASC' ? 'ASC' : 'DESC';
$sql .= " ORDER BY o.$sort $order";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$orders = $stmt->fetchAll(PDO::FETCH_ASSOC);

// חישוב total_paid לכל הזמנה
foreach ($orders as &$o) {
    $payments = json_decode($o['customer_payments'] ?? '[]', true);
    $totalPaid = 0;
    if (is_array($payments)) {
        foreach ($payments as $p) {
            $totalPaid += floatval($p['amount'] ?? 0);
        }
    }
    $o['total_paid'] = $totalPaid;
}
unset($o);

// סינון סטטוס תשלום
if ($filters['payment_status']) {
    $orders = array_filter($orders, function($o) use ($filters) {
        $totalPrice = floatval($o['total_price'] ?? 0);
        $totalPaid = floatval($o['total_paid'] ?? 0);
        
        if ($filters['payment_status'] === 'paid') {
            return $totalPaid >= $totalPrice && $totalPrice > 0;
        } elseif ($filters['payment_status'] === 'partial') {
            return $totalPaid > 0 && $totalPaid < $totalPrice;
        } elseif ($filters['payment_status'] === 'unpaid') {
            return $totalPaid == 0 || $totalPaid < $totalPrice * 0.1;
        }
        return true;
    });
}

// שליפת נתונים לדרופדאונים
$agents = [];
$suppliers = [];
$airlines = [];
try {
    $agents = $pdo->query("SELECT id, full_name FROM agents ORDER BY full_name")->fetchAll();
} catch (Exception $e) {}
try {
    $suppliers = $pdo->query("
        SELECT DISTINCT supplier FROM (
            SELECT flight_supplier as supplier FROM orders WHERE flight_supplier IS NOT NULL AND flight_supplier != ''
            UNION
            SELECT return_flight_supplier as supplier FROM orders WHERE return_flight_supplier IS NOT NULL AND return_flight_supplier != ''
        ) as all_suppliers ORDER BY supplier
    ")->fetchAll(PDO::FETCH_COLUMN);
} catch (Exception $e) {}
try {
    $airlines = $pdo->query("
        SELECT DISTINCT airline FROM (
            SELECT airline_outbound as airline FROM orders WHERE airline_outbound IS NOT NULL AND airline_outbound != ''
            UNION
            SELECT airline_return as airline FROM orders WHERE airline_return IS NOT NULL AND airline_return != ''
        ) as all_airlines ORDER BY airline
    ")->fetchAll(PDO::FETCH_COLUMN);
} catch (Exception $e) {}

// סטטיסטיקות
$totalOrders = count($orders);
$totalRevenue = array_sum(array_column($orders, 'total_price'));
$totalProfit = array_sum(array_column($orders, 'profit'));
$totalPaid = array_sum(array_column($orders, 'total_paid'));

} catch (Exception $e) {
    $orders = [];
    $agents = [];
    $suppliers = [];
    $airlines = [];
    $totalOrders = 0;
    $totalRevenue = 0;
    $totalProfit = 0;
    $totalPaid = 0;
    $error = $e->getMessage();
}
?>
<!DOCTYPE html>
<html lang="he" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>פאנל ניהול הזמנות</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: 'Segoe UI', Tahoma, sans-serif; background: #f0f2f5; min-height: 100vh; }
        
        .header {
            background: linear-gradient(135deg, #1e3c72, #2a5298);
            color: white;
            padding: 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-wrap: wrap;
            gap: 15px;
        }
        .header h1 { font-size: 1.5rem; }
        .header-actions { display: flex; gap: 10px; }
        .header-actions a {
            background: rgba(255,255,255,0.2);
            color: white;
            padding: 10px 20px;
            border-radius: 8px;
            text-decoration: none;
            font-size: 0.9rem;
        }
        .header-actions a:hover { background: rgba(255,255,255,0.3); }
        
        .container { max-width: 1800px; margin: 0 auto; padding: 20px; }
        
        /* סטטיסטיקות */
        .stats-row {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-bottom: 20px;
        }
        .stat-card {
            background: white;
            padding: 20px;
            border-radius: 12px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.08);
            text-align: center;
        }
        .stat-card .number { font-size: 1.8rem; font-weight: bold; color: #1e3c72; direction: ltr; display: inline-block; }
        .stat-card .label { color: #666; font-size: 0.9rem; margin-top: 5px; }
        
        /* פילטרים */
        .filters-card {
            background: white;
            padding: 20px;
            border-radius: 12px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.08);
            margin-bottom: 20px;
        }
        .filters-card h3 { margin-bottom: 15px; color: #333; }
        .filters-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 12px;
        }
        .filter-group { display: flex; flex-direction: column; }
        .filter-group label { font-size: 0.8rem; color: #666; margin-bottom: 4px; }
        .filter-group input, .filter-group select {
            padding: 8px 10px;
            border: 1px solid #ddd;
            border-radius: 6px;
            font-size: 0.9rem;
        }
        .filter-group input:focus, .filter-group select:focus {
            border-color: #1e3c72;
            outline: none;
        }
        .filter-actions {
            display: flex;
            gap: 10px;
            margin-top: 15px;
            justify-content: flex-end;
        }
        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-size: 0.9rem;
            font-weight: 500;
        }
        .btn-primary { background: #1e3c72; color: white; }
        .btn-secondary { background: #e0e0e0; color: #333; }
        .btn-success { background: #28a745; color: white; }
        .btn:hover { opacity: 0.9; }
        
        /* טבלה */
        .table-card {
            background: white;
            border-radius: 12px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.08);
        }
        .table-header {
            padding: 15px 20px;
            background: #f8f9fa;
            border-bottom: 1px solid #eee;
            display: flex;
            justify-content: space-between;
            align-items: center;
            gap: 15px;
            flex-wrap: wrap;
            border-radius: 12px 12px 0 0;
        }
        .table-wrapper { overflow-x: auto; overflow-y: visible; }
        
        /* טבלה - auto width עם מקסימום */
        table { 
            border-collapse: collapse; 
            font-size: 0.85rem; 
            width: auto;
            min-width: 100%;
        }
        th, td { 
            padding: 10px 14px; 
            text-align: right; 
            border-bottom: 1px solid #eee;
            border-left: 1px solid #e0e0e0;
            white-space: nowrap;
            position: relative;
            max-width: 180px;
            text-overflow: ellipsis;
        }
        th:last-child, td:last-child { border-left: none; }
        th { 
            background: #f8f9fa; 
            font-weight: 600; 
            color: #333;
            position: sticky;
            top: 0;
        }
        th a { color: inherit; text-decoration: none; }
        th a:hover { color: #1e3c72; }
        tr:hover { background: #f8f9fa; }
        
        /* עמודות מוסתרות */
        .col-hidden { display: none !important; }
        
        .order-id { font-weight: bold; color: #1e3c72; }
        .customer-name { font-weight: 500; overflow: visible; }
        .destination { color: #28a745; }
        .supplier { 
            background: #e3f2fd; 
            padding: 3px 8px; 
            border-radius: 4px; 
            font-size: 0.75rem;
            font-weight: 600;
        }
        .airline {
            background: #fff3e0;
            padding: 3px 8px;
            border-radius: 4px;
            font-size: 0.75rem;
        }
        .booking-code {
            font-family: monospace;
            font-size: 0.8rem;
        }
        .actions { white-space: nowrap; }
        .actions a { 
            margin: 0 3px; 
            text-decoration: none;
            font-size: 1rem;
        }
        
        /* אייקון הערות */
        .notes-indicator {
            display: inline-flex;
            align-items: center;
            cursor: help;
            margin-left: 5px;
            position: relative;
            font-size: 0.9rem;
        }
        .notes-indicator .tooltip {
            visibility: hidden;
            opacity: 0;
            position: absolute;
            background: #333;
            color: white;
            padding: 10px 15px;
            border-radius: 8px;
            font-size: 0.85rem;
            white-space: pre-wrap;
            max-width: 300px;
            min-width: 200px;
            z-index: 9999;
            top: 25px;
            right: -10px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.3);
            line-height: 1.5;
            font-weight: normal;
            transition: opacity 0.2s, visibility 0.2s;
        }
        .notes-indicator .tooltip::before {
            content: '';
            position: absolute;
            top: -8px;
            right: 15px;
            border-width: 0 8px 8px 8px;
            border-style: solid;
            border-color: transparent transparent #333 transparent;
        }
        .notes-indicator:hover .tooltip {
            visibility: visible;
            opacity: 1;
        }
        
        /* מודאל התאמת עמודות */
        .columns-btn {
            background: #667eea;
            color: white;
            padding: 8px 15px;
            border-radius: 6px;
            cursor: pointer;
            font-size: 0.85rem;
            border: none;
            display: flex;
            align-items: center;
            gap: 5px;
        }
        .columns-btn:hover { background: #5a6fd6; }
        
        .modal-overlay {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0,0,0,0.5);
            z-index: 1000;
            justify-content: center;
            align-items: center;
        }
        .modal-overlay.show { display: flex; }
        
        .modal-content {
            background: white;
            border-radius: 12px;
            padding: 25px;
            max-width: 400px;
            width: 90%;
            max-height: 80vh;
            overflow-y: auto;
        }
        .modal-content h3 {
            margin-bottom: 20px;
            color: #333;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .modal-close {
            background: none;
            border: none;
            font-size: 1.5rem;
            cursor: pointer;
            color: #999;
        }
        .modal-close:hover { color: #333; }
        
        .column-option {
            display: flex;
            align-items: center;
            padding: 10px;
            border-bottom: 1px solid #eee;
            cursor: pointer;
        }
        .column-option:hover { background: #f8f9fa; }
        .column-option input { margin-left: 10px; }
        .column-option label { cursor: pointer; flex: 1; }
        
        .modal-actions {
            margin-top: 20px;
            display: flex;
            gap: 10px;
            justify-content: flex-end;
        }
        
        /* Resizer */
        .resizer {
            position: absolute;
            left: 0;
            top: 0;
            height: 100%;
            width: 5px;
            cursor: col-resize;
            background: transparent;
        }
        .resizer:hover { background: #1e3c72; }
    </style>
</head>
<body>
    <div class="header">
        <h1>📊 פאנל ניהול הזמנות</h1>
        <div class="header-actions">
            <a href="complete_order_lite.php">⚡ הזמנה חדשה</a>
            <a href="drafts_list.php">📋 טיוטות</a>
            <?php if ($isAdmin): ?>
            <a href="reports.php">📈 דוחות</a>
            <a href="settings.php">⚙️ הגדרות</a>
            <?php endif; ?>
        </div>
    </div>
    
    <div class="container">
        <!-- סטטיסטיקות -->
        <div class="stats-row">
            <div class="stat-card">
                <div class="number"><?= $totalOrders ?></div>
                <div class="label">סה"כ הזמנות</div>
            </div>
            <div class="stat-card">
                <div class="number">₪<?= number_format($totalRevenue) ?></div>
                <div class="label">סה"כ הכנסות</div>
            </div>
            <div class="stat-card">
                <div class="number"><?= $totalProfit < 0 ? '-₪' . number_format(abs($totalProfit)) : '₪' . number_format($totalProfit) ?></div>
                <div class="label">סה"כ רווח</div>
            </div>
            <div class="stat-card">
                <div class="number">₪<?= number_format($totalPaid) ?></div>
                <div class="label">סה"כ שולם</div>
            </div>
        </div>
        
        <!-- פילטרים -->
        <div class="filters-card">
            <h3>🔍 סינון וחיפוש</h3>
            <form method="GET">
                <div class="filters-grid">
                    <div class="filter-group">
                        <label>מספר הזמנה</label>
                        <input type="text" name="order_id" value="<?= htmlspecialchars($filters['order_id']) ?>" placeholder="#">
                    </div>
                    <div class="filter-group">
                        <label>שם/טלפון/מייל</label>
                        <input type="text" name="customer_name" value="<?= htmlspecialchars($filters['customer_name']) ?>" placeholder="חיפוש...">
                    </div>
                    <div class="filter-group">
                        <label>יעד</label>
                        <input type="text" name="destination" value="<?= htmlspecialchars($filters['destination']) ?>" placeholder="יעד...">
                    </div>
                    <div class="filter-group">
                        <label>ספק טיסה</label>
                        <select name="flight_supplier">
                            <option value="">הכל</option>
                            <?php foreach ($suppliers as $s): ?>
                            <option value="<?= htmlspecialchars($s) ?>" <?= $filters['flight_supplier'] === $s ? 'selected' : '' ?>><?= htmlspecialchars(strtoupper($s)) ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="filter-group">
                        <label>חברת תעופה</label>
                        <select name="flight_airline">
                            <option value="">הכל</option>
                            <?php foreach ($airlines as $a): ?>
                            <option value="<?= htmlspecialchars($a) ?>" <?= $filters['flight_airline'] === $a ? 'selected' : '' ?>><?= htmlspecialchars($a) ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <?php if ($isAdmin): ?>
                    <div class="filter-group">
                        <label>נציג</label>
                        <select name="agent_id">
                            <option value="">הכל</option>
                            <?php foreach ($agents as $a): ?>
                            <option value="<?= $a['id'] ?>" <?= $filters['agent_id'] == $a['id'] ? 'selected' : '' ?>><?= htmlspecialchars($a['full_name']) ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <?php endif; ?>
                    <div class="filter-group">
                        <label>מתאריך הזמנה</label>
                        <input type="date" name="date_from" value="<?= htmlspecialchars($filters['date_from']) ?>">
                    </div>
                    <div class="filter-group">
                        <label>עד תאריך הזמנה</label>
                        <input type="date" name="date_to" value="<?= htmlspecialchars($filters['date_to']) ?>">
                    </div>
                    <div class="filter-group">
                        <label>מתאריך טיסה</label>
                        <input type="date" name="flight_from" value="<?= htmlspecialchars($filters['flight_from']) ?>">
                    </div>
                    <div class="filter-group">
                        <label>עד תאריך טיסה</label>
                        <input type="date" name="flight_to" value="<?= htmlspecialchars($filters['flight_to']) ?>">
                    </div>
                </div>
                <div class="filter-actions">
                    <a href="orders_panel.php" class="btn btn-secondary">נקה סינון</a>
                    <button type="submit" class="btn btn-primary">🔍 חפש</button>
                </div>
            </form>
        </div>
        
        <!-- טבלת הזמנות -->
        <div class="table-card">
            <div class="table-header">
                <span><?= $totalOrders ?> הזמנות נמצאו</span>
                <button class="columns-btn" onclick="openColumnsModal()">⚙️ התאמת עמודות</button>
            </div>
            <div class="table-wrapper">
                <table id="ordersTable">
                    <thead>
                        <tr>
                            <th data-col="id"><a href="?<?= http_build_query(array_merge($filters, ['sort' => 'id', 'order' => $filters['sort'] === 'id' && $filters['order'] === 'DESC' ? 'ASC' : 'DESC'])) ?>"># <?= $filters['sort'] === 'id' ? ($filters['order'] === 'DESC' ? '▼' : '▲') : '' ?></a></th>
                            <th data-col="date"><a href="?<?= http_build_query(array_merge($filters, ['sort' => 'created_at', 'order' => $filters['sort'] === 'created_at' && $filters['order'] === 'DESC' ? 'ASC' : 'DESC'])) ?>">תאריך <?= $filters['sort'] === 'created_at' ? ($filters['order'] === 'DESC' ? '▼' : '▲') : '' ?></a></th>
                            <?php if ($isAdmin): ?>
                            <th data-col="agent">נציג</th>
                            <th data-col="builder">בונה דיל</th>
                            <?php endif; ?>
                            <th data-col="customer">לקוח</th>
                            <th data-col="phone">טלפון</th>
                            <th data-col="email">אימייל</th>
                            <th data-col="order_type">סוג</th>
                            <th data-col="destination"><a href="?<?= http_build_query(array_merge($filters, ['sort' => 'destination', 'order' => $filters['sort'] === 'destination' && $filters['order'] === 'DESC' ? 'ASC' : 'DESC'])) ?>">יעד <?= $filters['sort'] === 'destination' ? ($filters['order'] === 'DESC' ? '▼' : '▲') : '' ?></a></th>
                            <th data-col="supplier_out">ספק הלוך</th>
                            <th data-col="code_out">קוד הלוך</th>
                            <th data-col="airline_out">תעופה הלוך</th>
                            <th data-col="supplier_ret">ספק חזור</th>
                            <th data-col="code_ret">קוד חזור</th>
                            <th data-col="airline_ret">תעופה חזור</th>
                            <th data-col="date_out"><a href="?<?= http_build_query(array_merge($filters, ['sort' => 'departure_date', 'order' => $filters['sort'] === 'departure_date' && $filters['order'] === 'DESC' ? 'ASC' : 'DESC'])) ?>">הלוך <?= $filters['sort'] === 'departure_date' ? ($filters['order'] === 'DESC' ? '▼' : '▲') : '' ?></a></th>
                            <th data-col="date_ret"><a href="?<?= http_build_query(array_merge($filters, ['sort' => 'return_date', 'order' => $filters['sort'] === 'return_date' && $filters['order'] === 'DESC' ? 'ASC' : 'DESC'])) ?>">חזור <?= $filters['sort'] === 'return_date' ? ($filters['order'] === 'DESC' ? '▼' : '▲') : '' ?></a></th>
                            <th data-col="actions">פעולות</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($orders as $o): 
                            // שם הנוסע הראשון
                            $passengers = json_decode($o['passengers'] ?? '[]', true);
                            $leadName = '-';
                            if (!empty($passengers) && isset($passengers[0])) {
                                $leadName = is_string($passengers[0]) ? $passengers[0] : ($passengers[0]['first_name'] ?? '') . ' ' . ($passengers[0]['last_name'] ?? '');
                                $leadName = trim($leadName) ?: '-';
                            }
                            
                            // יעד
                            $destination = $o['destination'] ?? '-';
                            if (strpos($destination, ',') !== false) {
                                $destination = trim(explode(',', $destination)[0]);
                            } elseif (strpos($destination, '-') !== false) {
                                $destination = trim(explode('-', $destination)[0]);
                            }
                            
                            // ספק וחברת תעופה
                            $outSupplier = strtoupper($o['flight_supplier'] ?? '');
                            $retSupplier = strtoupper($o['return_flight_supplier'] ?? '') ?: $outSupplier;
                            $outAirline = $o['airline_outbound'] ?? '';
                            $retAirline = $o['airline_return'] ?? $outAirline;
                            
                            // קודי הזמנה
                            $outCode = $o['booking_code_outbound'] ?? '';
                            $retCode = $o['booking_code_return'] ?? '';
                            
                            // הערות
                            $notes = trim($o['notes'] ?? '');
                            $hasNotes = !empty($notes);
                        ?>
                        <tr>
                            <td data-col="id" class="order-id"><?= $o['id'] ?></td>
                            <td data-col="date"><?= date('d/m/y', strtotime($o['created_at'])) ?></td>
                            <?php if ($isAdmin): ?>
                            <td data-col="agent"><?= htmlspecialchars($o['agent_name'] ?? '-') ?></td>
                            <td data-col="builder"><?= htmlspecialchars($o['deal_builder'] ?? '-') ?></td>
                            <?php endif; ?>
                            <td data-col="customer" class="customer-name">
                                <?php if ($hasNotes): ?>
                                <span class="notes-indicator">📝<span class="tooltip"><?= htmlspecialchars($notes) ?></span></span>
                                <?php endif; ?>
                                <span title="<?= htmlspecialchars($leadName) ?>"><?= htmlspecialchars($leadName) ?></span>
                            </td>
                            <td data-col="phone"><a href="tel:<?= htmlspecialchars($o['customer_phone'] ?? '') ?>" style="color:#333;"><?= htmlspecialchars($o['customer_phone'] ?? '-') ?></a></td>
                            <td data-col="email"><a href="mailto:<?= htmlspecialchars($o['customer_email'] ?? '') ?>" style="color:#667eea; font-size:0.8rem;"><?= htmlspecialchars($o['customer_email'] ?? '-') ?></a></td>
                            <td data-col="order_type"><?php 
                                $orderType = $o['order_type'] ?? '';
                                $typeLabels = [
                                    'package' => '📦 חבילה',
                                    'organized' => '🚌 מאורגן',
                                    'nofshon' => '🏨 נופשון'
                                ];
                                echo $typeLabels[$orderType] ?? ($orderType ?: '-');
                            ?></td>
                            <td data-col="destination" class="destination"><?= htmlspecialchars($destination) ?></td>
                            <td data-col="supplier_out"><?php if ($outSupplier): ?><span class="supplier"><?= htmlspecialchars($outSupplier) ?></span><?php else: ?>-<?php endif; ?></td>
                            <td data-col="code_out" class="booking-code"><?= htmlspecialchars($outCode ?: '-') ?></td>
                            <td data-col="airline_out"><?php if ($outAirline): ?><span class="airline"><?= htmlspecialchars($outAirline) ?></span><?php else: ?>-<?php endif; ?></td>
                            <td data-col="supplier_ret"><?php if ($retSupplier): ?><span class="supplier"><?= htmlspecialchars($retSupplier) ?></span><?php else: ?>-<?php endif; ?></td>
                            <td data-col="code_ret" class="booking-code"><?= htmlspecialchars($retCode ?: '-') ?></td>
                            <td data-col="airline_ret"><?php if ($retAirline): ?><span class="airline"><?= htmlspecialchars($retAirline) ?></span><?php else: ?>-<?php endif; ?></td>
                            <td data-col="date_out"><?= $o['departure_date'] ? date('d/m/y', strtotime($o['departure_date'])) : '-' ?></td>
                            <td data-col="date_ret"><?= $o['return_date'] ? date('d/m/y', strtotime($o['return_date'])) : '-' ?></td>
                            <td data-col="actions" class="actions">
                                <a href="view_order_new.php?id=<?= $o['id'] ?>" title="צפה">👁️</a>
                                <?php if ($isAdmin): ?>
                                <a href="edit_order.php?id=<?= $o['id'] ?>" title="ערוך">✏️</a>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                        
                        <?php if (empty($orders)): ?>
                        <tr>
                            <td colspan="20" style="text-align: center; padding: 40px; color: #666;">
                                לא נמצאו הזמנות לפי הסינון שנבחר
                            </td>
                        </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    
    <!-- מודאל התאמת עמודות -->
    <div class="modal-overlay" id="columnsModal">
        <div class="modal-content">
            <h3>
                ⚙️ התאמת עמודות
                <button class="modal-close" onclick="closeColumnsModal()">×</button>
            </h3>
            <div id="columnsList">
                <!-- יתמלא דינמית -->
            </div>
            <div class="modal-actions">
                <button class="btn btn-secondary" onclick="resetColumns()">איפוס לברירת מחדל</button>
                <button class="btn btn-primary" onclick="closeColumnsModal()">סגור</button>
            </div>
        </div>
    </div>
    
    <script>
    // הגדרת עמודות
    const allColumns = [
        { id: 'id', name: '# מספר', default: true },
        { id: 'date', name: 'תאריך הזמנה', default: true },
        <?php if ($isAdmin): ?>
        { id: 'agent', name: 'נציג', default: false },
        { id: 'builder', name: 'בונה דיל', default: false },
        <?php endif; ?>
        { id: 'customer', name: 'לקוח', default: true },
        { id: 'phone', name: 'טלפון', default: true },
        { id: 'email', name: 'אימייל', default: false },
        { id: 'order_type', name: 'סוג הזמנה', default: false },
        { id: 'destination', name: 'יעד', default: true },
        { id: 'supplier_out', name: 'ספק הלוך', default: true },
        { id: 'code_out', name: 'קוד הלוך', default: true },
        { id: 'airline_out', name: 'תעופה הלוך', default: true },
        { id: 'supplier_ret', name: 'ספק חזור', default: true },
        { id: 'code_ret', name: 'קוד חזור', default: true },
        { id: 'airline_ret', name: 'תעופה חזור', default: true },
        { id: 'date_out', name: 'תאריך הלוך', default: true },
        { id: 'date_ret', name: 'תאריך חזור', default: true },
        { id: 'actions', name: 'פעולות', default: true }
    ];
    
    // טעינת הגדרות שמורות
    function getVisibleColumns() {
        const saved = localStorage.getItem('ordersColumns');
        if (saved) {
            return JSON.parse(saved);
        }
        // ברירת מחדל
        return allColumns.filter(c => c.default).map(c => c.id);
    }
    
    // שמירת הגדרות
    function saveVisibleColumns(cols) {
        localStorage.setItem('ordersColumns', JSON.stringify(cols));
    }
    
    // החלת הגדרות על הטבלה
    function applyColumnVisibility() {
        const visible = getVisibleColumns();
        
        document.querySelectorAll('[data-col]').forEach(el => {
            const col = el.getAttribute('data-col');
            if (visible.includes(col)) {
                el.classList.remove('col-hidden');
            } else {
                el.classList.add('col-hidden');
            }
        });
    }
    
    // בניית רשימת עמודות במודאל
    function buildColumnsList() {
        const visible = getVisibleColumns();
        const container = document.getElementById('columnsList');
        
        container.innerHTML = allColumns.map(col => `
            <div class="column-option" onclick="toggleColumn('${col.id}')">
                <input type="checkbox" id="col_${col.id}" ${visible.includes(col.id) ? 'checked' : ''}>
                <label for="col_${col.id}">${col.name}</label>
            </div>
        `).join('');
    }
    
    // החלפת מצב עמודה
    function toggleColumn(colId) {
        let visible = getVisibleColumns();
        
        if (visible.includes(colId)) {
            visible = visible.filter(c => c !== colId);
        } else {
            visible.push(colId);
        }
        
        saveVisibleColumns(visible);
        applyColumnVisibility();
        buildColumnsList();
    }
    
    // איפוס לברירת מחדל
    function resetColumns() {
        localStorage.removeItem('ordersColumns');
        applyColumnVisibility();
        buildColumnsList();
    }
    
    // פתיחת מודאל
    function openColumnsModal() {
        buildColumnsList();
        document.getElementById('columnsModal').classList.add('show');
    }
    
    // סגירת מודאל
    function closeColumnsModal() {
        document.getElementById('columnsModal').classList.remove('show');
    }
    
    // סגירה בלחיצה מחוץ למודאל
    document.getElementById('columnsModal').addEventListener('click', function(e) {
        if (e.target === this) {
            closeColumnsModal();
        }
    });
    
    // גרירת עמודות
    document.addEventListener('DOMContentLoaded', function() {
        // החלת הגדרות עמודות
        applyColumnVisibility();
        
        // גרירת עמודות
        const table = document.querySelector('table');
        if (!table) return;
        
        const headers = table.querySelectorAll('th');
        let isResizing = false;
        let currentTh = null;
        let startX = 0;
        let startWidth = 0;
        
        headers.forEach(th => {
            const resizer = document.createElement('div');
            resizer.className = 'resizer';
            th.appendChild(resizer);
            
            resizer.addEventListener('mousedown', function(e) {
                isResizing = true;
                currentTh = th;
                startX = e.pageX;
                startWidth = th.offsetWidth;
                document.body.style.cursor = 'col-resize';
                document.body.style.userSelect = 'none';
                e.preventDefault();
            });
        });
        
        document.addEventListener('mousemove', function(e) {
            if (!isResizing) return;
            const diff = startX - e.pageX; // RTL
            const newWidth = Math.max(50, startWidth + diff);
            currentTh.style.width = newWidth + 'px';
            currentTh.style.minWidth = newWidth + 'px';
        });
        
        document.addEventListener('mouseup', function() {
            if (isResizing) {
                isResizing = false;
                currentTh = null;
                document.body.style.cursor = '';
                document.body.style.userSelect = '';
            }
        });
    });
    </script>
</body>
</html>
