<?php
// דוחות כספיים מקיפים
require_once 'includes/auth.php';
requireLogin();

// בדיקת הרשאות
$agent = getCurrentAgent();
$isAdmin = ($agent['role'] === 'admin');
$isAccountant = ($agent['role'] === 'accountant');

// רואה חשבון יכול לגשת רק לדוח accountant
if ($isAccountant) {
    if (!isset($_GET['report']) || $_GET['report'] !== 'accountant') {
        header('Location: reports.php?report=accountant');
        exit;
    }
}
// נציג רגיל לא יכול לגשת בכלל
elseif (!$isAdmin) {
    header('Location: complete_order_lite.php');
    exit;
}

$pdo = getDB();
$error = '';
$success = '';

// טיפול בסימון תשלום לספק
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['mark_supplier_paid'])) {
    $orderId = (int)($_POST['order_id'] ?? 0);
    $flightType = $_POST['flight_type'] ?? ''; // outbound, return, or hotel
    $bookingCode = $_POST['booking_code'] ?? '';
    
    if ($orderId && $flightType) {
        try {
            if ($flightType === 'outbound') {
                $stmt = $pdo->prepare("UPDATE orders SET flight_supplier_paid = 'paid' WHERE id = ?");
            } elseif ($flightType === 'return') {
                $stmt = $pdo->prepare("UPDATE orders SET return_flight_supplier_paid = 'paid' WHERE id = ?");
            } elseif ($flightType === 'hotel') {
                $stmt = $pdo->prepare("UPDATE orders SET hotel_supplier_paid = 'paid' WHERE id = ?");
            }
            $stmt->execute([$orderId]);
            
            if ($stmt->rowCount() > 0) {
                $success = "סומן כשולם בהצלחה";
            }
        } catch (Exception $e) {
            $error = "שגיאה בעדכון: " . $e->getMessage();
        }
    }
    
    // רענון העמוד
    header("Location: reports.php?report=suppliers&date_from=" . ($_GET['date_from'] ?? date('Y-m-01')) . "&date_to=" . ($_GET['date_to'] ?? date('Y-m-d')) . "&flight_from=" . ($_GET['flight_from'] ?? '') . "&flight_to=" . ($_GET['flight_to'] ?? '') . "&supplier_filter=" . urlencode($_GET['supplier_filter'] ?? '') . "&success=" . urlencode($success));
    exit;
}

// טיפול בסימון תוספת כשולמה
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['mark_addition_paid'])) {
    $additionId = (int)($_POST['addition_id'] ?? 0);
    
    if ($additionId) {
        try {
            $stmt = $pdo->prepare("UPDATE order_additions SET is_paid = 1 WHERE id = ?");
            $stmt->execute([$additionId]);
            
            if ($stmt->rowCount() > 0) {
                $success = "תוספת סומנה כשולמה בהצלחה";
            }
        } catch (Exception $e) {
            $error = "שגיאה בעדכון: " . $e->getMessage();
        }
    }
    
    // רענון העמוד
    header("Location: reports.php?report=suppliers&date_from=" . ($_GET['date_from'] ?? date('Y-m-01')) . "&date_to=" . ($_GET['date_to'] ?? date('Y-m-d')) . "&flight_from=" . ($_GET['flight_from'] ?? '') . "&flight_to=" . ($_GET['flight_to'] ?? '') . "&supplier_filter=" . urlencode($_GET['supplier_filter'] ?? '') . "&success=" . urlencode($success));
    exit;
}

// הודעת הצלחה מ-redirect
if (isset($_GET['success']) && $_GET['success']) {
    $success = $_GET['success'];
}

// פרמטרים
$reportType = $_GET['report'] ?? 'profitability';
$dateFrom = $_GET['date_from'] ?? date('Y-m-01');
$dateTo = $_GET['date_to'] ?? date('Y-m-d');
$flightFrom = $_GET['flight_from'] ?? '';
$flightTo = $_GET['flight_to'] ?? '';
$groupBy = $_GET['group_by'] ?? 'month';
$supplierFilter = $_GET['supplier_filter'] ?? '';
$sortBy = $_GET['sort_by'] ?? 'created_at';
$sortDir = $_GET['sort_dir'] ?? $_GET['sort_order'] ?? 'desc';

// דוח רווחיות
function getProfitabilityReport($pdo, $dateFrom, $dateTo, $groupBy) {
    $groupField = 'DATE_FORMAT(o.created_at, "%Y-%m")';
    $groupLabel = 'חודש';
    
    if ($groupBy === 'supplier') {
        $groupField = 'IFNULL(o.flight_supplier, "לא צוין")';
        $groupLabel = 'ספק';
    } elseif ($groupBy === 'agent') {
        $groupField = 'IFNULL(a.full_name, "לא צוין")';
        $groupLabel = 'נציג';
    } elseif ($groupBy === 'airline') {
        $groupField = 'IFNULL(o.airline_outbound, "לא צוין")';
        $groupLabel = 'חברת תעופה';
    } elseif ($groupBy === 'destination') {
        $groupField = 'IFNULL(o.destination, "לא צוין")';
        $groupLabel = 'יעד';
    }
    
    $sql = "SELECT 
                $groupField as group_name,
                COUNT(*) as order_count,
                SUM(IFNULL(o.total_price, 0)) as total_revenue,
                SUM(IFNULL(o.profit, 0)) as total_profit,
                AVG(IFNULL(o.profit, 0)) as avg_profit
            FROM orders o
            LEFT JOIN agents a ON o.agent_id = a.id
            WHERE DATE(o.created_at) BETWEEN ? AND ?
            GROUP BY $groupField
            ORDER BY total_profit DESC";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$dateFrom, $dateTo]);
    return ['data' => $stmt->fetchAll(PDO::FETCH_ASSOC), 'label' => $groupLabel];
}

// דוח תשלומים לספקים - סיכום
function getSupplierPaymentsReport($pdo, $dateFrom, $dateTo) {
    $sql = "SELECT 
                IFNULL(flight_supplier, 'לא צוין') as supplier_name,
                COUNT(*) as order_count,
                SUM(IFNULL(flight_supplier_cost, 0) + IFNULL(return_flight_supplier_cost, 0)) as total_cost_usd,
                SUM(CASE WHEN flight_supplier_paid = 'paid' THEN IFNULL(flight_supplier_cost, 0) ELSE 0 END + 
                    CASE WHEN return_flight_supplier_paid = 'paid' THEN IFNULL(return_flight_supplier_cost, 0) ELSE 0 END) as paid_usd,
                SUM(CASE WHEN flight_supplier_paid != 'paid' OR flight_supplier_paid IS NULL THEN IFNULL(flight_supplier_cost, 0) ELSE 0 END +
                    CASE WHEN return_flight_supplier_paid != 'paid' OR return_flight_supplier_paid IS NULL THEN IFNULL(return_flight_supplier_cost, 0) ELSE 0 END) as unpaid_usd
            FROM orders
            WHERE DATE(created_at) BETWEEN ? AND ?
            GROUP BY flight_supplier
            ORDER BY total_cost_usd DESC";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$dateFrom, $dateTo]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// דוח תשלומים לספקים - פירוט מלא לפי מספר הזמנה ספק
function getSupplierPaymentsDetailedReport($pdo, $dateFrom, $dateTo, $supplierFilter = '', $sortBy = 'created_at', $sortDir = 'desc', $flightFrom = '', $flightTo = '') {
    // שליפת כל ההזמנות עם פרטי ספקים
    $sql = "SELECT 
                id,
                flight_supplier,
                booking_code_outbound,
                flight_supplier_cost,
                flight_supplier_paid,
                return_flight_supplier,
                booking_code_return,
                return_flight_supplier_cost,
                return_flight_supplier_paid,
                hotel_supplier,
                hotel_supplier_booking,
                hotel_supplier_cost,
                hotel_supplier_paid,
                destination,
                departure_date,
                return_date,
                created_at
            FROM orders
            WHERE 1=1";
    
    $params = [];
    
    // אם יש פילטר תאריך טיסה - שולפים הזמנות שיש להן טיסה כלשהי בטווח
    if ($flightFrom || $flightTo) {
        $dateConditions = [];
        
        // תאריך הלוך בטווח
        if ($flightFrom && $flightTo) {
            $dateConditions[] = "(DATE(departure_date) BETWEEN ? AND ?)";
            $params[] = $flightFrom;
            $params[] = $flightTo;
        } elseif ($flightFrom) {
            $dateConditions[] = "(DATE(departure_date) >= ?)";
            $params[] = $flightFrom;
        } elseif ($flightTo) {
            $dateConditions[] = "(DATE(departure_date) <= ?)";
            $params[] = $flightTo;
        }
        
        // תאריך חזור בטווח - גם אם אותו ספק (בגלל תוספות לטיסת חזור)
        if ($flightFrom && $flightTo) {
            $dateConditions[] = "(DATE(return_date) BETWEEN ? AND ?)";
            $params[] = $flightFrom;
            $params[] = $flightTo;
        } elseif ($flightFrom) {
            $dateConditions[] = "(DATE(return_date) >= ?)";
            $params[] = $flightFrom;
        } elseif ($flightTo) {
            $dateConditions[] = "(DATE(return_date) <= ?)";
            $params[] = $flightTo;
        }
        
        $sql .= " AND (" . implode(" OR ", $dateConditions) . ")";
    } else {
        // פילטר לפי תאריך הזמנה (ברירת מחדל)
        $sql .= " AND DATE(created_at) BETWEEN ? AND ?";
        $params[] = $dateFrom;
        $params[] = $dateTo;
    }
    
    $sql .= " ORDER BY flight_supplier, created_at DESC";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $orders = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // שליפת תוספות לכל ההזמנות בתקופה
    $orderIds = array_column($orders, 'id');
    $additions = [];
    if (!empty($orderIds)) {
        $placeholders = implode(',', array_fill(0, count($orderIds), '?'));
        $addSql = "SELECT * FROM order_additions WHERE order_id IN ($placeholders) ORDER BY order_id, component";
        $addStmt = $pdo->prepare($addSql);
        $addStmt->execute($orderIds);
        $allAdditions = $addStmt->fetchAll(PDO::FETCH_ASSOC);
        
        // ארגון תוספות לפי order_id
        foreach ($allAdditions as $add) {
            $additions[$add['order_id']][] = $add;
        }
    }
    
    // פונקציה לבדיקה אם תאריך בטווח הפילטר
    $isInFlightRange = function($date) use ($flightFrom, $flightTo) {
        if (!$flightFrom && !$flightTo) return true; // אין פילטר טיסה
        if (!$date) return false;
        $d = date('Y-m-d', strtotime($date));
        if ($flightFrom && $d < $flightFrom) return false;
        if ($flightTo && $d > $flightTo) return false;
        return true;
    };
    
    // ארגון לפי ספקים
    $suppliers = [];
    
    foreach ($orders as $order) {
        $orderId = $order['id'];
        $orderAdditions = $additions[$orderId] ?? [];
        
        // בדיקה אם אותו ספק להלוך ולחזור
        $sameSupplier = (
            $order['flight_supplier'] && 
            (!$order['return_flight_supplier'] || 
             strtoupper($order['return_flight_supplier']) === strtoupper($order['flight_supplier']))
        );
        
        // טיסת הלוך - עלות בסיס
        if ($order['flight_supplier']) {
            $supplierKey = strtoupper($order['flight_supplier']);
            
            // אם אותו ספק - הלוך+חזור ביחד, סינון לפי departure_date
            // אם ספקים שונים - רק הלוך, סינון לפי departure_date
            $legDate = $order['departure_date'];
            
            // בדיקת פילטר תאריך טיסה
            if ($flightFrom || $flightTo) {
                if (!$isInFlightRange($legDate)) {
                    // תאריך ההלוך לא בטווח - לא מציגים את ה-LEG הזה
                    goto skipOutbound;
                }
            }
            
            if (!isset($suppliers[$supplierKey])) {
                $suppliers[$supplierKey] = [
                    'name' => $supplierKey,
                    'total_cost' => 0,
                    'total_paid' => 0,
                    'total_unpaid' => 0,
                    'items' => []
                ];
            }
            
            $cost = floatval($order['flight_supplier_cost'] ?? 0);
            $isPaid = ($order['flight_supplier_paid'] === 'paid');
            
            // עלות בסיס
            if ($cost > 0) {
                $suppliers[$supplierKey]['total_cost'] += $cost;
                if ($isPaid) {
                    $suppliers[$supplierKey]['total_paid'] += $cost;
                } else {
                    $suppliers[$supplierKey]['total_unpaid'] += $cost;
                }
                
                $suppliers[$supplierKey]['items'][] = [
                    'order_id' => $order['id'],
                    'booking_code' => $order['booking_code_outbound'] ?: '-',
                    'type' => $sameSupplier ? 'הלוך+חזור' : 'הלוך',
                    'description' => 'עלות בסיס',
                    'destination' => $order['destination'],
                    'flight_date' => $order['departure_date'],
                    'departure_date' => $order['departure_date'],
                    'return_date' => $order['return_date'],
                    'cost' => $cost,
                    'is_paid' => $isPaid,
                    'created_at' => $order['created_at'],
                    'is_addition' => false
                ];
            }
            
            // תוספות טיסת הלוך - לאותו ספק (בלי תלות בעלות בסיס)
            foreach ($orderAdditions as $add) {
                if ($add['component'] === 'flight_outbound') {
                    $addCost = floatval($add['amount'] ?? 0);
                    $addIsPaid = !empty($add['is_paid']);
                    
                    if ($addCost > 0) {
                        $suppliers[$supplierKey]['total_cost'] += $addCost;
                        if ($addIsPaid) {
                            $suppliers[$supplierKey]['total_paid'] += $addCost;
                        } else {
                            $suppliers[$supplierKey]['total_unpaid'] += $addCost;
                        }
                        
                        $suppliers[$supplierKey]['items'][] = [
                            'order_id' => $order['id'],
                            'booking_code' => $order['booking_code_outbound'] ?: '-',
                            'type' => $sameSupplier ? 'הלוך+חזור' : 'הלוך',
                            'description' => '+ ' . ($add['description'] ?? 'תוספת'),
                            'destination' => $order['destination'],
                            'flight_date' => $order['departure_date'],
                            'departure_date' => $order['departure_date'],
                            'return_date' => $order['return_date'],
                            'cost' => $addCost,
                            'is_paid' => $addIsPaid,
                            'created_at' => $add['created_at'],
                            'is_addition' => true,
                            'addition_id' => $add['id']
                        ];
                    }
                }
                
                // תוספות טיסת חזור כשאותו ספק - נבדקות אחרי skipOutbound
            }
        }
        
        skipOutbound:
        
        // תוספות טיסת חזור כשאותו ספק - בדיקה נפרדת לפי תאריך החזור
        $sameSupplier = (
            $order['flight_supplier'] && 
            (!$order['return_flight_supplier'] || 
             strtoupper($order['return_flight_supplier']) === strtoupper($order['flight_supplier']))
        );
        
        if ($order['flight_supplier'] && $sameSupplier) {
            $supplierKey = strtoupper($order['flight_supplier']);
            $returnDate = $order['return_date'];
            
            // בדיקת פילטר תאריך טיסה לתאריך החזור
            $returnInRange = true;
            if ($flightFrom || $flightTo) {
                $returnInRange = $isInFlightRange($returnDate);
            }
            
            if ($returnInRange) {
                foreach ($orderAdditions as $add) {
                    if ($add['component'] === 'flight_return') {
                        $addCost = floatval($add['amount'] ?? 0);
                        $addIsPaid = !empty($add['is_paid']);
                        
                        if ($addCost > 0) {
                            if (!isset($suppliers[$supplierKey])) {
                                $suppliers[$supplierKey] = [
                                    'name' => $supplierKey,
                                    'total_cost' => 0,
                                    'total_paid' => 0,
                                    'total_unpaid' => 0,
                                    'items' => []
                                ];
                            }
                            
                            $suppliers[$supplierKey]['total_cost'] += $addCost;
                            if ($addIsPaid) {
                                $suppliers[$supplierKey]['total_paid'] += $addCost;
                            } else {
                                $suppliers[$supplierKey]['total_unpaid'] += $addCost;
                            }
                            
                            $suppliers[$supplierKey]['items'][] = [
                                'order_id' => $order['id'],
                                'booking_code' => $order['booking_code_return'] ?: $order['booking_code_outbound'] ?: '-',
                                'type' => 'חזור',
                                'description' => '+ ' . ($add['description'] ?? 'תוספת'),
                                'destination' => $order['destination'],
                                'flight_date' => $order['return_date'],
                                'departure_date' => $order['departure_date'],
                                'return_date' => $order['return_date'],
                                'cost' => $addCost,
                                'is_paid' => $addIsPaid,
                                'created_at' => $add['created_at'],
                                'is_addition' => true,
                                'addition_id' => $add['id']
                            ];
                        }
                    }
                }
            }
        }
        
        // טיסת חזור - רק אם ספק שונה מההלוך!
        // (אם אותו ספק - כבר הוצג כ"הלוך+חזור" למעלה)
        $retSupplier = $order['return_flight_supplier'];
        $hasDifferentReturnSupplier = $retSupplier && strtoupper($retSupplier) !== strtoupper($order['flight_supplier'] ?? '');
        
        // בודקים אם יש תוספות לטיסת חזור
        $hasReturnAdditions = false;
        foreach ($orderAdditions as $add) {
            if ($add['component'] === 'flight_return' && floatval($add['amount'] ?? 0) > 0) {
                $hasReturnAdditions = true;
                break;
            }
        }
        
        $returnCost = floatval($order['return_flight_supplier_cost'] ?? 0);
        
        // מציגים אם: (ספק שונה) AND (יש עלות > 0 OR יש תוספות)
        if ($hasDifferentReturnSupplier && ($returnCost > 0 || $hasReturnAdditions)) {
            // סינון לפי תאריך החזור
            $returnDate = $order['return_date'];
            
            if ($flightFrom || $flightTo) {
                if (!$isInFlightRange($returnDate)) {
                    // תאריך החזור לא בטווח - לא מציגים את ה-LEG הזה
                    goto skipReturn;
                }
            }
            
            $supplierKey = strtoupper($retSupplier);
            if (!isset($suppliers[$supplierKey])) {
                $suppliers[$supplierKey] = [
                    'name' => $supplierKey,
                    'total_cost' => 0,
                    'total_paid' => 0,
                    'total_unpaid' => 0,
                    'items' => []
                ];
            }
            
            $cost = $returnCost;
            $isPaid = ($order['return_flight_supplier_paid'] === 'paid');
            
            // הוספת עלות בסיס רק אם יש עלות
            if ($cost > 0) {
                $suppliers[$supplierKey]['total_cost'] += $cost;
                if ($isPaid) {
                    $suppliers[$supplierKey]['total_paid'] += $cost;
                } else {
                    $suppliers[$supplierKey]['total_unpaid'] += $cost;
                }
                
                $suppliers[$supplierKey]['items'][] = [
                    'order_id' => $order['id'],
                    'booking_code' => $order['booking_code_return'] ?: '-',
                    'type' => 'חזור',
                    'description' => 'עלות בסיס',
                    'destination' => $order['destination'],
                    'flight_date' => $order['return_date'],
                    'departure_date' => $order['departure_date'],
                    'return_date' => $order['return_date'],
                    'cost' => $cost,
                    'is_paid' => $isPaid,
                    'created_at' => $order['created_at'],
                    'is_addition' => false
                ];
            }
            
            // תוספות טיסת חזור
            foreach ($orderAdditions as $add) {
                if ($add['component'] === 'flight_return') {
                    $addCost = floatval($add['amount'] ?? 0);
                    $addIsPaid = !empty($add['is_paid']);
                    
                    if ($addCost > 0) {
                        $suppliers[$supplierKey]['total_cost'] += $addCost;
                        if ($addIsPaid) {
                            $suppliers[$supplierKey]['total_paid'] += $addCost;
                        } else {
                            $suppliers[$supplierKey]['total_unpaid'] += $addCost;
                        }
                        
                        $suppliers[$supplierKey]['items'][] = [
                            'order_id' => $order['id'],
                            'booking_code' => $order['booking_code_return'] ?: '-',
                            'type' => 'חזור',
                            'description' => '+ ' . ($add['description'] ?? 'תוספת'),
                            'destination' => $order['destination'],
                            'flight_date' => $order['return_date'],
                            'departure_date' => $order['departure_date'],
                            'return_date' => $order['return_date'],
                            'cost' => $addCost,
                            'is_paid' => $addIsPaid,
                            'created_at' => $add['created_at'],
                            'is_addition' => true,
                            'addition_id' => $add['id']
                        ];
                    }
                }
            }
        }
        
        skipReturn:
        
        // מלון - עלות בסיס (סינון לפי תאריך הלוך = תחילת המלון)
        $hotelInRange = true;
        if ($flightFrom || $flightTo) {
            $hotelInRange = $isInFlightRange($order['departure_date']);
        }
        
        if ($hotelInRange && $order['hotel_supplier'] && floatval($order['hotel_supplier_cost'] ?? 0) > 0) {
            $supplierKey = strtoupper($order['hotel_supplier']);
            if (!isset($suppliers[$supplierKey])) {
                $suppliers[$supplierKey] = [
                    'name' => $supplierKey,
                    'total_cost' => 0,
                    'total_paid' => 0,
                    'total_unpaid' => 0,
                    'items' => []
                ];
            }
            
            $cost = floatval($order['hotel_supplier_cost'] ?? 0);
            $isPaid = ($order['hotel_supplier_paid'] === 'paid');
            
            $suppliers[$supplierKey]['total_cost'] += $cost;
            if ($isPaid) {
                $suppliers[$supplierKey]['total_paid'] += $cost;
            } else {
                $suppliers[$supplierKey]['total_unpaid'] += $cost;
            }
            
            $suppliers[$supplierKey]['items'][] = [
                'order_id' => $order['id'],
                'booking_code' => $order['hotel_supplier_booking'] ?: '-',
                'type' => 'מלון',
                'description' => 'עלות בסיס',
                'destination' => $order['destination'],
                'flight_date' => $order['departure_date'],
                'departure_date' => $order['departure_date'],
                'return_date' => $order['return_date'],
                'cost' => $cost,
                'is_paid' => $isPaid,
                'created_at' => $order['created_at'],
                'is_addition' => false
            ];
        }
        
        // תוספות מלון (גם לפי תאריך הלוך)
        if ($hotelInRange && $order['hotel_supplier']) {
            $supplierKey = strtoupper($order['hotel_supplier']);
            foreach ($orderAdditions as $add) {
                if ($add['component'] === 'hotel') {
                    if (!isset($suppliers[$supplierKey])) {
                        $suppliers[$supplierKey] = [
                            'name' => $supplierKey,
                            'total_cost' => 0,
                            'total_paid' => 0,
                            'total_unpaid' => 0,
                            'items' => []
                        ];
                    }
                    
                    $addCost = floatval($add['amount'] ?? 0);
                    $addIsPaid = !empty($add['is_paid']);
                    
                    if ($addCost > 0) {
                        $suppliers[$supplierKey]['total_cost'] += $addCost;
                        if ($addIsPaid) {
                            $suppliers[$supplierKey]['total_paid'] += $addCost;
                        } else {
                            $suppliers[$supplierKey]['total_unpaid'] += $addCost;
                        }
                        
                        $suppliers[$supplierKey]['items'][] = [
                            'order_id' => $order['id'],
                            'booking_code' => $order['hotel_supplier_booking'] ?: '-',
                            'type' => 'מלון',
                            'description' => '+ ' . ($add['description'] ?? 'תוספת'),
                            'destination' => $order['destination'],
                            'flight_date' => $order['departure_date'],
                            'departure_date' => $order['departure_date'],
                            'return_date' => $order['return_date'],
                            'cost' => $addCost,
                            'is_paid' => $addIsPaid,
                            'created_at' => $add['created_at'],
                            'is_addition' => true,
                            'addition_id' => $add['id']
                        ];
                    }
                }
            }
        }
    }
    
    // פילטור לפי ספק אם נבחר
    if ($supplierFilter) {
        $suppliers = array_filter($suppliers, function($s) use ($supplierFilter) {
            return $s['name'] === $supplierFilter;
        });
    }
    
    // מיון פריטים בתוך כל ספק
    foreach ($suppliers as &$supplier) {
        usort($supplier['items'], function($a, $b) use ($sortBy, $sortDir) {
            $fieldA = $a[$sortBy] ?? '';
            $fieldB = $b[$sortBy] ?? '';
            
            // מיון מספרי עבור cost ו-order_id
            if (in_array($sortBy, ['cost', 'order_id'])) {
                $fieldA = floatval($fieldA);
                $fieldB = floatval($fieldB);
                $result = $fieldA - $fieldB;
            }
            // מיון בוליאני עבור סטטוס תשלום
            elseif ($sortBy === 'is_paid') {
                $result = ($fieldA ? 1 : 0) - ($fieldB ? 1 : 0);
            }
            // מיון טקסטי/תאריכי
            else {
                $result = strcmp($fieldA, $fieldB);
            }
            
            // הפיכת הסדר אם יורד
            return $sortDir === 'asc' ? $result : -$result;
        });
    }
    
    // מיון לפי סה"כ חוב
    uasort($suppliers, function($a, $b) {
        return $b['total_unpaid'] - $a['total_unpaid'];
    });
    
    return $suppliers;
}

// דוח טיסות קרובות
function getUpcomingFlightsReport($pdo, $days = 14) {
    $sql = "SELECT o.*, a.full_name as agent_name
            FROM orders o
            LEFT JOIN agents a ON o.agent_id = a.id
            WHERE o.departure_date BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL ? DAY)
            ORDER BY o.departure_date ASC";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$days]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// דוח לרואה חשבון - מבוסס טרנזקציות
function getAccountantReport($pdo, $dateFrom, $dateTo) {
    // שאילתה משולבת: הזמנות ראשיות + תוספות + החזרים
    $sql = "
        -- הזמנות ראשיות
        SELECT 
            DATE(o.created_at) as transaction_date,
            o.created_at as full_date,
            o.id as order_id,
            NULL as addition_id,
            NULL as refund_id,
            'הזמנה ראשית' as transaction_type,
            o.destination as description,
            (IFNULL(o.total_price, 0) - IFNULL(o.profit, 0)) as cost_ils,
            IFNULL(o.total_price, 0) as income_ils,
            IFNULL(o.profit, 0) as profit_ils,
            o.payments as payments_json
        FROM orders o
        WHERE DATE(o.created_at) BETWEEN ? AND ?
        
        UNION ALL
        
        -- תוספות להזמנות
        SELECT 
            DATE(a.created_at) as transaction_date,
            a.created_at as full_date,
            a.order_id as order_id,
            a.id as addition_id,
            NULL as refund_id,
            CASE a.component 
                WHEN 'flight_outbound' THEN 'תוספת טיסת הלוך'
                WHEN 'flight_return' THEN 'תוספת טיסת חזור'
                WHEN 'hotel' THEN 'תוספת מלון'
                ELSE 'תוספת'
            END as transaction_type,
            a.description as description,
            IFNULL(a.amount_ils, IFNULL(a.amount, 0)) as cost_ils,
            IFNULL(a.customer_charge, 0) as income_ils,
            IFNULL(a.profit, 0) as profit_ils,
            NULL as payments_json
        FROM order_additions a
        WHERE DATE(a.created_at) BETWEEN ? AND ?
        
        UNION ALL
        
        -- החזרים
        SELECT 
            DATE(IFNULL(r.refund_date, r.created_at)) as transaction_date,
            r.created_at as full_date,
            r.order_id as order_id,
            NULL as addition_id,
            r.id as refund_id,
            'החזר' as transaction_type,
            CONCAT(r.description, 
                CASE 
                    WHEN r.supplier_name IS NOT NULL AND r.airline IS NOT NULL THEN CONCAT(' (', r.supplier_name, ' / ', r.airline, ')')
                    WHEN r.supplier_name IS NOT NULL THEN CONCAT(' (', r.supplier_name, ')')
                    WHEN r.airline IS NOT NULL THEN CONCAT(' (', r.airline, ')')
                    ELSE ''
                END
            ) as description,
            -IFNULL(r.received_from_supplier, 0) as cost_ils,
            -IFNULL(r.returned_to_customer, 0) as income_ils,
            (IFNULL(r.received_from_supplier, 0) - IFNULL(r.returned_to_customer, 0)) as profit_ils,
            NULL as payments_json
        FROM order_refunds r
        WHERE DATE(IFNULL(r.refund_date, r.created_at)) BETWEEN ? AND ?
        
        ORDER BY transaction_date ASC, full_date ASC
    ";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$dateFrom, $dateTo, $dateFrom, $dateTo, $dateFrom, $dateTo]);
    $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // עיבוד מספרי קבלות מה-JSON
    foreach ($results as &$row) {
        $receiptNumbers = [];
        if (!empty($row['payments_json'])) {
            $payments = json_decode($row['payments_json'], true);
            if (is_array($payments)) {
                foreach ($payments as $p) {
                    if (!empty($p['receipt_number'])) {
                        $receiptNumbers[] = $p['receipt_number'];
                    }
                }
            }
        }
        $row['receipt_numbers'] = implode(', ', $receiptNumbers);
        unset($row['payments_json']); // לא צריך להעביר את כל ה-JSON
    }
    
    return $results;
}

// שליפת נתונים לפי סוג דוח
$reportData = [];
$suppliersDetailed = [];
$reportLabel = '';
$summary = ['total_orders' => 0, 'total_revenue' => 0, 'total_profit' => 0, 'total_supplier_cost' => 0];

try {

switch ($reportType) {
    case 'profitability':
        $result = getProfitabilityReport($pdo, $dateFrom, $dateTo, $groupBy);
        $reportData = $result['data'];
        $reportLabel = $result['label'];
        break;
    case 'suppliers':
        $reportData = getSupplierPaymentsReport($pdo, $dateFrom, $dateTo);
        $suppliersDetailed = getSupplierPaymentsDetailedReport($pdo, $dateFrom, $dateTo, $supplierFilter, $sortBy, $sortDir, $flightFrom, $flightTo);
        break;
    case 'flights':
        $reportData = getUpcomingFlightsReport($pdo, 14);
        break;
    case 'accountant':
        $reportData = getAccountantReport($pdo, $dateFrom, $dateTo);
        break;
}

// סיכומים כלליים
$summaryStmt = $pdo->prepare("
    SELECT 
        COUNT(*) as total_orders,
        SUM(IFNULL(total_price, 0)) as total_revenue,
        SUM(IFNULL(profit, 0)) as total_profit,
        SUM(IFNULL(flight_supplier_cost, 0) + IFNULL(return_flight_supplier_cost, 0) + IFNULL(hotel_supplier_cost, 0)) as total_supplier_cost
    FROM orders 
    WHERE DATE(created_at) BETWEEN ? AND ?
");
$summaryStmt->execute([$dateFrom, $dateTo]);
$summary = $summaryStmt->fetch(PDO::FETCH_ASSOC);

} catch (Exception $e) {
    $error = $e->getMessage();
    $reportData = [];
    $summary = ['total_orders' => 0, 'total_revenue' => 0, 'total_profit' => 0, 'total_supplier_cost' => 0];
}
?>
<!DOCTYPE html>
<html lang="he" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>דוחות כספיים</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: 'Segoe UI', Tahoma, sans-serif; background: #f0f2f5; min-height: 100vh; }
        
        .header {
            background: linear-gradient(135deg, #2c3e50, #3498db);
            color: white;
            padding: 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .header h1 { font-size: 1.5rem; }
        .header a { color: white; text-decoration: none; background: rgba(255,255,255,0.2); padding: 10px 20px; border-radius: 8px; }
        
        .container { max-width: 1400px; margin: 0 auto; padding: 20px; }
        
        /* סיכום */
        .summary-row {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-bottom: 20px;
        }
        .summary-card {
            background: white;
            padding: 20px;
            border-radius: 12px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.08);
            text-align: center;
        }
        .summary-card .number { font-size: 1.6rem; font-weight: bold; direction: ltr; display: inline-block; }
        .summary-card .label { color: #666; font-size: 0.85rem; margin-top: 5px; }
        .summary-card.profit .number { color: #28a745; }
        
        /* טאבים */
        .tabs {
            display: flex;
            gap: 5px;
            margin-bottom: 20px;
            flex-wrap: wrap;
        }
        .tab {
            padding: 12px 20px;
            background: white;
            border-radius: 8px;
            text-decoration: none;
            color: #333;
            font-weight: 500;
            box-shadow: 0 2px 5px rgba(0,0,0,0.05);
        }
        .tab.active {
            background: #2c3e50;
            color: white;
        }
        
        /* כרטיס */
        .card {
            background: white;
            border-radius: 12px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.08);
            margin-bottom: 20px;
            overflow: hidden;
        }
        .card-header {
            padding: 15px 20px;
            background: #f8f9fa;
            border-bottom: 1px solid #eee;
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-wrap: wrap;
            gap: 10px;
        }
        .card-header h3 { font-size: 1.1rem; }
        .card-body { padding: 20px; }
        
        /* פילטרים */
        .filters {
            display: flex;
            gap: 15px;
            flex-wrap: wrap;
            align-items: flex-end;
        }
        .filter-group { display: flex; flex-direction: column; }
        .filter-group label { font-size: 0.8rem; color: #666; margin-bottom: 4px; }
        .filter-group input, .filter-group select {
            padding: 8px 12px;
            border: 1px solid #ddd;
            border-radius: 6px;
        }
        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 500;
        }
        .btn-primary { background: #2c3e50; color: white; }
        .btn-success { background: #28a745; color: white; }
        
        /* טבלה */
        table { width: 100%; border-collapse: collapse; }
        th, td { padding: 12px; text-align: right; border-bottom: 1px solid #eee; }
        th { background: #f8f9fa; font-weight: 600; font-size: 0.85rem; }
        tr:hover { background: #f8f9fa; }
        
        .positive { color: #28a745; font-weight: 600; }
        .negative { color: #dc3545; font-weight: 600; }
        .warning { color: #ffc107; font-weight: 600; }
        
        /* תיקון כיווניות מספרים - מינוס תמיד משמאל */
        .positive, .negative, td:nth-child(n+3) {
            direction: ltr;
            unicode-bidi: isolate;
        }
        
        .supplier-badge {
            background: #e3f2fd;
            padding: 3px 10px;
            border-radius: 4px;
            font-size: 0.8rem;
            font-weight: 500;
        }
        
        .progress-bar {
            background: #e9ecef;
            border-radius: 4px;
            height: 8px;
            overflow: hidden;
        }
        .progress-bar .fill {
            height: 100%;
            background: #28a745;
        }
        
        /* טיסות קרובות */
        .flight-card {
            background: #f8f9fa;
            border-radius: 8px;
            padding: 15px;
            margin-bottom: 10px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-wrap: wrap;
            gap: 10px;
        }
        .flight-date {
            font-size: 1.2rem;
            font-weight: bold;
            color: #2c3e50;
        }
        .flight-info { flex: 1; min-width: 200px; }
        .flight-customer { font-weight: 500; }
        .flight-destination { color: #28a745; }
        .days-left {
            padding: 8px 15px;
            border-radius: 20px;
            font-weight: 500;
            font-size: 0.85rem;
        }
        .days-urgent { background: #f8d7da; color: #721c24; }
        .days-soon { background: #fff3cd; color: #856404; }
        .days-normal { background: #d4edda; color: #155724; }
        
        @media print {
            .header, .tabs, .filters, .btn { display: none !important; }
            .card { box-shadow: none; border: 1px solid #ddd; }
        }
    </style>
</head>
<body>
    <div class="header">
        <h1>📊 <?= $isAccountant ? 'דוח לרואה חשבון' : 'דוחות כספיים' ?></h1>
        <?php if (!$isAccountant): ?>
        <a href="orders_panel.php">← חזרה לפאנל</a>
        <?php else: ?>
        <a href="logout.php">🚪 התנתק</a>
        <?php endif; ?>
    </div>
    
    <div class="container">
        <?php if (!empty($error)): ?>
        <div style="background: #f8d7da; color: #721c24; padding: 15px; border-radius: 8px; margin-bottom: 20px;">
            ⚠️ שגיאה: <?= htmlspecialchars($error) ?>
        </div>
        <?php endif; ?>
        
        <?php if (!empty($success)): ?>
        <div class="alert-success">
            ✅ <?= htmlspecialchars($success) ?>
        </div>
        <?php endif; ?>
        
        <?php if (!$isAccountant): ?>
        <!-- סיכום תקופה -->
        <div class="summary-row">
            <div class="summary-card">
                <div class="number"><?= number_format($summary['total_orders'] ?? 0) ?></div>
                <div class="label">הזמנות בתקופה</div>
            </div>
            <div class="summary-card">
                <div class="number">₪<?= number_format($summary['total_revenue'] ?? 0) ?></div>
                <div class="label">סה"כ הכנסות</div>
            </div>
            <div class="summary-card profit">
                <div class="number">₪<?= number_format($summary['total_profit'] ?? 0) ?></div>
                <div class="label">סה"כ רווח</div>
            </div>
            <div class="summary-card">
                <div class="number">$<?= number_format($summary['total_supplier_cost'] ?? 0) ?></div>
                <div class="label">עלות ספקים</div>
            </div>
        </div>
        <?php endif; ?>
        
        <?php if (!$isAccountant): ?>
        <!-- טאבים -->
        <div class="tabs">
            <a href="?report=profitability&date_from=<?= $dateFrom ?>&date_to=<?= $dateTo ?>" class="tab <?= $reportType === 'profitability' ? 'active' : '' ?>">📈 רווחיות</a>
            <a href="?report=suppliers&date_from=<?= $dateFrom ?>&date_to=<?= $dateTo ?>" class="tab <?= $reportType === 'suppliers' ? 'active' : '' ?>">🏢 תשלומים לספקים</a>
            <a href="?report=flights" class="tab <?= $reportType === 'flights' ? 'active' : '' ?>">✈️ טיסות קרובות</a>
            <a href="?report=accountant&date_from=<?= $dateFrom ?>&date_to=<?= $dateTo ?>" class="tab <?= $reportType === 'accountant' ? 'active' : '' ?>">📋 דוח לרו"ח</a>
            <a href="bank_transfers.php" class="tab">🏦 העברות בנקאיות</a>
        </div>
        <?php endif; ?>
        
        <!-- תוכן דוח -->
        <?php if ($reportType === 'profitability'): ?>
        <div class="card">
            <div class="card-header">
                <h3>📈 דוח רווחיות</h3>
                <form method="GET" class="filters">
                    <input type="hidden" name="report" value="profitability">
                    <div class="filter-group">
                        <label>מתאריך</label>
                        <input type="date" name="date_from" value="<?= $dateFrom ?>">
                    </div>
                    <div class="filter-group">
                        <label>עד תאריך</label>
                        <input type="date" name="date_to" value="<?= $dateTo ?>">
                    </div>
                    <div class="filter-group">
                        <label>קבץ לפי</label>
                        <select name="group_by">
                            <option value="month" <?= $groupBy === 'month' ? 'selected' : '' ?>>חודש</option>
                            <option value="supplier" <?= $groupBy === 'supplier' ? 'selected' : '' ?>>ספק</option>
                            <option value="agent" <?= $groupBy === 'agent' ? 'selected' : '' ?>>נציג</option>
                            <option value="airline" <?= $groupBy === 'airline' ? 'selected' : '' ?>>חברת תעופה</option>
                            <option value="destination" <?= $groupBy === 'destination' ? 'selected' : '' ?>>יעד</option>
                        </select>
                    </div>
                    <button type="submit" class="btn btn-primary">הצג</button>
                    <a href="export_report.php?report=profitability&date_from=<?= $dateFrom ?>&date_to=<?= $dateTo ?>&group_by=<?= $groupBy ?>" class="btn btn-success">📥 Excel</a>
                </form>
            </div>
            <div class="card-body">
                <table>
                    <thead>
                        <tr>
                            <th><?= $reportLabel ?></th>
                            <th>הזמנות</th>
                            <th>הכנסות</th>
                            <th>רווח</th>
                            <th>רווח ממוצע</th>
                            <th>% מהרווח</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php 
                        $totalProfit = array_sum(array_column($reportData, 'total_profit'));
                        foreach ($reportData as $row): 
                            $profitPercent = $totalProfit > 0 ? ($row['total_profit'] / $totalProfit * 100) : 0;
                        ?>
                        <tr>
                            <td><strong><?= htmlspecialchars($row['group_name']) ?></strong></td>
                            <td><?= number_format($row['order_count']) ?></td>
                            <td>₪<?= number_format($row['total_revenue']) ?></td>
                            <td class="positive">₪<?= number_format($row['total_profit']) ?></td>
                            <td>₪<?= number_format($row['avg_profit']) ?></td>
                            <td>
                                <div class="progress-bar" style="width: 100px;">
                                    <div class="fill" style="width: <?= min($profitPercent, 100) ?>%;"></div>
                                </div>
                                <?= number_format($profitPercent, 1) ?>%
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        
        <?php elseif ($reportType === 'suppliers'): ?>
        <?php 
        // שליפת רשימת ספקים לפילטר
        $allSupplierNames = array_keys($suppliersDetailed);
        sort($allSupplierNames);
        ?>
        <div class="card">
            <div class="card-header" style="display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; gap: 15px;">
                <h3>🏢 חובות לספקים - פירוט לפי מספר הזמנה ספק</h3>
                <div style="display: flex; gap: 10px; align-items: center;">
                    <a href="supplier_reconciliation.php" class="btn btn-success" style="display: inline-flex; align-items: center; gap: 5px;">
                        🔄 התאמות עם ספקים
                    </a>
                </div>
            </div>
            <div class="card-body">
                <!-- פילטרים -->
                <form method="GET" class="filters" style="margin-bottom: 20px; background: #f8f9fa; padding: 15px; border-radius: 10px;">
                    <input type="hidden" name="report" value="suppliers">
                    <div style="display: flex; gap: 20px; align-items: flex-end; flex-wrap: wrap;">
                        <div>
                            <label style="font-size: 0.8rem; display: block; margin-bottom: 5px;">📅 תאריך הזמנה מ-</label>
                            <input type="date" name="date_from" value="<?= $dateFrom ?>" style="padding: 8px 12px; border-radius: 5px; border: 1px solid #ddd;">
                        </div>
                        <div>
                            <label style="font-size: 0.8rem; display: block; margin-bottom: 5px;">עד</label>
                            <input type="date" name="date_to" value="<?= $dateTo ?>" style="padding: 8px 12px; border-radius: 5px; border: 1px solid #ddd;">
                        </div>
                        
                        <div style="border-right: 2px solid #ddd; height: 50px;"></div>
                        
                        <div>
                            <label style="font-size: 0.8rem; display: block; margin-bottom: 5px;">✈️ תאריך טיסה מ-</label>
                            <input type="date" name="flight_from" value="<?= $flightFrom ?>" style="padding: 8px 12px; border-radius: 5px; border: 1px solid #ddd;">
                        </div>
                        <div>
                            <label style="font-size: 0.8rem; display: block; margin-bottom: 5px;">עד</label>
                            <input type="date" name="flight_to" value="<?= $flightTo ?>" style="padding: 8px 12px; border-radius: 5px; border: 1px solid #ddd;">
                        </div>
                        
                        <div style="border-right: 2px solid #ddd; height: 50px;"></div>
                        
                        <div>
                            <label style="font-size: 0.8rem; display: block; margin-bottom: 5px;">🏢 ספק</label>
                            <select name="supplier_filter" style="padding: 8px 12px; border-radius: 5px; border: 1px solid #ddd; min-width: 150px;">
                                <option value="">כל הספקים</option>
                                <?php foreach ($allSupplierNames as $sName): ?>
                                <option value="<?= htmlspecialchars($sName) ?>" <?= $supplierFilter === $sName ? 'selected' : '' ?>><?= htmlspecialchars($sName) ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <button type="submit" class="btn btn-primary" style="padding: 8px 20px;">🔍 סנן</button>
                        <a href="?report=suppliers" class="btn btn-secondary" style="padding: 8px 15px;">נקה</a>
                    </div>
                    <p style="font-size: 0.75rem; color: #666; margin-top: 10px; margin-bottom: 0;">
                        💡 <strong>טיפ:</strong> אם תמלא תאריך טיסה - יוצגו רק הזמנות שהטיסה שלהן בטווח הזה (בלי קשר למתי הוזמנו)
                    </p>
                </form>
                
                <p style="font-size: 0.85rem; color: #666; margin-bottom: 15px;">💡 לחץ על כותרת עמודה למיון</p>
                
                <?php if (empty($suppliersDetailed)): ?>
                <p style="text-align: center; color: #666; padding: 40px;">אין נתונים לתקופה זו</p>
                <?php else: ?>
                
                <?php 
                // פונקציה ליצירת לינק מיון
                function sortLink($field, $label, $currentSort, $currentDir, $dateFrom, $dateTo, $supplierFilter, $flightFrom = '', $flightTo = '') {
                    $isActive = ($currentSort === $field);
                    $newDir = ($isActive && $currentDir === 'desc') ? 'asc' : 'desc';
                    $arrow = '';
                    if ($isActive) {
                        $arrow = $currentDir === 'desc' ? ' ▼' : ' ▲';
                    }
                    $url = "?report=suppliers&date_from=$dateFrom&date_to=$dateTo&supplier_filter=" . urlencode($supplierFilter) . "&flight_from=$flightFrom&flight_to=$flightTo&sort_by=$field&sort_dir=$newDir";
                    $class = $isActive ? 'sort-header active' : 'sort-header';
                    return "<a href=\"$url\" class=\"$class\">$label$arrow</a>";
                }
                ?>
                
                <?php foreach ($suppliersDetailed as $supplier): ?>
                <div class="supplier-section">
                    <div class="supplier-header" onclick="toggleSupplier(this)">
                        <div class="supplier-title">
                            <span class="toggle-icon">▼</span>
                            <span class="supplier-badge large"><?= htmlspecialchars($supplier['name']) ?></span>
                            <span class="supplier-count"><?= count($supplier['items']) ?> פריטים</span>
                        </div>
                        <div class="supplier-totals">
                            <span class="total-label">סה"כ: <strong>$<?= number_format($supplier['total_cost'], 2) ?></strong></span>
                            <span class="paid-label">שולם: <span class="positive">$<?= number_format($supplier['total_paid'], 2) ?></span></span>
                            <span class="unpaid-label">חוב: <span class="<?= $supplier['total_unpaid'] > 0 ? 'negative' : '' ?>">$<?= number_format($supplier['total_unpaid'], 2) ?></span></span>
                        </div>
                    </div>
                    <div class="supplier-details" style="display: block;">
                        <table class="details-table">
                            <thead>
                                <tr>
                                    <th><?= sortLink('booking_code', "מס' הזמנה ספק", $sortBy, $sortDir, $dateFrom, $dateTo, $supplierFilter, $flightFrom, $flightTo) ?></th>
                                    <th><?= sortLink('type', 'סוג', $sortBy, $sortDir, $dateFrom, $dateTo, $supplierFilter, $flightFrom, $flightTo) ?></th>
                                    <th><?= sortLink('description', 'תיאור', $sortBy, $sortDir, $dateFrom, $dateTo, $supplierFilter, $flightFrom, $flightTo) ?></th>
                                    <th><?= sortLink('destination', 'יעד', $sortBy, $sortDir, $dateFrom, $dateTo, $supplierFilter, $flightFrom, $flightTo) ?></th>
                                    <th><?= sortLink('created_at', 'תאריך הזמנה', $sortBy, $sortDir, $dateFrom, $dateTo, $supplierFilter, $flightFrom, $flightTo) ?></th>
                                    <th><?= sortLink('flight_date', 'תאריך טיסה', $sortBy, $sortDir, $dateFrom, $dateTo, $supplierFilter, $flightFrom, $flightTo) ?></th>
                                    <th><?= sortLink('cost', 'סכום', $sortBy, $sortDir, $dateFrom, $dateTo, $supplierFilter, $flightFrom, $flightTo) ?></th>
                                    <th><?= sortLink('is_paid', 'סטטוס', $sortBy, $sortDir, $dateFrom, $dateTo, $supplierFilter, $flightFrom, $flightTo) ?></th>
                                    <th><?= sortLink('order_id', 'הזמנה שלנו', $sortBy, $sortDir, $dateFrom, $dateTo, $supplierFilter, $flightFrom, $flightTo) ?></th>
                                    <th>פעולה</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($supplier['items'] as $item): ?>
                                <tr class="<?= $item['is_paid'] ? 'row-paid' : 'row-unpaid' ?> <?= !empty($item['is_addition']) ? 'row-addition' : '' ?>">
                                    <td><strong><?= htmlspecialchars($item['booking_code']) ?></strong></td>
                                    <td><span class="type-badge <?= $item['type'] === 'הלוך' ? 'outbound' : ($item['type'] === 'חזור' ? 'return' : 'hotel') ?>"><?= $item['type'] ?></span></td>
                                    <td class="<?= !empty($item['is_addition']) ? 'addition-desc' : '' ?>"><?= htmlspecialchars($item['description'] ?? 'עלות בסיס') ?></td>
                                    <td><?= htmlspecialchars($item['destination'] ?? '-') ?></td>
                                    <td><?= $item['created_at'] ? date('d/m/Y', strtotime($item['created_at'])) : '-' ?></td>
                                    <td><?= $item['flight_date'] ? date('d/m/Y', strtotime($item['flight_date'])) : '-' ?></td>
                                    <td><strong>$<?= number_format($item['cost'], 2) ?></strong></td>
                                    <td>
                                        <?php if ($item['is_paid']): ?>
                                        <span class="status-paid">✓ שולם</span>
                                        <?php else: ?>
                                        <span class="status-unpaid">✗ ממתין</span>
                                        <?php endif; ?>
                                    </td>
                                    <td><a href="view_order_new.php?id=<?= $item['order_id'] ?>" class="order-link">#<?= $item['order_id'] ?></a></td>
                                    <td>
                                        <?php if (!$item['is_paid']): ?>
                                            <?php if (!empty($item['is_addition'])): ?>
                                            <form method="POST" style="display:inline;" onsubmit="return confirm('לסמן תוספת זו כשולמה?');">
                                                <input type="hidden" name="mark_addition_paid" value="1">
                                                <input type="hidden" name="addition_id" value="<?= $item['addition_id'] ?>">
                                                <button type="submit" class="btn-mark-paid" title="סמן כשולם">✓</button>
                                            </form>
                                            <?php else: ?>
                                            <form method="POST" style="display:inline;" onsubmit="return confirm('לסמן קוד <?= $item['booking_code'] ?> כשולם?');">
                                                <input type="hidden" name="mark_supplier_paid" value="1">
                                                <input type="hidden" name="order_id" value="<?= $item['order_id'] ?>">
                                                <input type="hidden" name="flight_type" value="<?= $item['type'] === 'הלוך' ? 'outbound' : ($item['type'] === 'חזור' ? 'return' : 'hotel') ?>">
                                                <input type="hidden" name="booking_code" value="<?= htmlspecialchars($item['booking_code']) ?>">
                                                <button type="submit" class="btn-mark-paid" title="סמן כשולם">✓</button>
                                            </form>
                                            <?php endif; ?>
                                        <?php else: ?>
                                        <span style="color:#28a745;">✓</span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
                <?php endforeach; ?>
                
                <?php endif; ?>
            </div>
        </div>
        
        <style>
        .supplier-section {
            margin-bottom: 15px;
            border: 1px solid #e0e0e0;
            border-radius: 8px;
            overflow: hidden;
        }
        .supplier-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 15px 20px;
            background: linear-gradient(135deg, #f8f9fa, #e9ecef);
            cursor: pointer;
            transition: background 0.2s;
        }
        .supplier-header:hover {
            background: linear-gradient(135deg, #e9ecef, #dee2e6);
        }
        .supplier-title {
            display: flex;
            align-items: center;
            gap: 12px;
        }
        .toggle-icon {
            transition: transform 0.3s;
            font-size: 0.8rem;
            color: #666;
        }
        .supplier-header.collapsed .toggle-icon {
            transform: rotate(-90deg);
        }
        .supplier-badge.large {
            font-size: 1rem;
            padding: 6px 14px;
        }
        .supplier-count {
            color: #666;
            font-size: 0.85rem;
        }
        .supplier-totals {
            display: flex;
            gap: 20px;
            font-size: 0.9rem;
        }
        .supplier-totals span {
            white-space: nowrap;
        }
        .supplier-details {
            padding: 0;
            background: #fff;
        }
        .details-table {
            width: 100%;
            border-collapse: collapse;
            font-size: 0.85rem;
        }
        .details-table th {
            background: #f1f3f4;
            padding: 10px 12px;
            text-align: right;
            font-weight: 600;
            border-bottom: 2px solid #e0e0e0;
        }
        .details-table td {
            padding: 10px 12px;
            border-bottom: 1px solid #eee;
        }
        .details-table tr:hover {
            background: #f8f9fa;
        }
        
        /* סגנון כותרות מיון */
        .sort-header {
            color: #333;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 4px;
            cursor: pointer;
            transition: all 0.2s;
            padding: 2px 6px;
            border-radius: 4px;
        }
        .sort-header:hover {
            background: #e3e7eb;
            color: #1e3c72;
        }
        .sort-header.active {
            color: #1e3c72;
            font-weight: 700;
            background: #e8f0fe;
        }
        
        .row-paid {
            background: #f0fff0;
        }
        .row-unpaid {
            background: #fff5f5;
        }
        .row-addition {
            font-style: italic;
        }
        .row-addition td:first-child {
            padding-right: 25px;
        }
        .addition-desc {
            color: #6c757d;
            font-size: 0.8rem;
        }
        .type-badge {
            padding: 3px 8px;
            border-radius: 4px;
            font-size: 0.75rem;
            font-weight: 500;
        }
        .type-badge.outbound {
            background: #e3f2fd;
            color: #1565c0;
        }
        .type-badge.return {
            background: #fff3e0;
            color: #e65100;
        }
        .type-badge.hotel {
            background: #f3e5f5;
            color: #7b1fa2;
        }
        .status-paid {
            color: #28a745;
            font-weight: 500;
        }
        .status-unpaid {
            color: #dc3545;
            font-weight: 500;
        }
        .order-link {
            color: #1e3c72;
            text-decoration: none;
            font-weight: 500;
        }
        .order-link:hover {
            text-decoration: underline;
        }
        .positive { color: #28a745; }
        .negative { color: #dc3545; font-weight: bold; }
        .btn-mark-paid {
            background: #28a745;
            color: white;
            border: none;
            border-radius: 4px;
            padding: 4px 10px;
            cursor: pointer;
            font-size: 0.85rem;
            transition: all 0.2s;
        }
        .btn-mark-paid:hover {
            background: #218838;
            transform: scale(1.05);
        }
        .alert-success {
            background: #d4edda;
            color: #155724;
            padding: 15px 20px;
            border-radius: 8px;
            margin-bottom: 20px;
            border: 1px solid #c3e6cb;
        }
        </style>
        
        <script>
        function toggleSupplier(header) {
            const details = header.nextElementSibling;
            const isCollapsed = header.classList.toggle('collapsed');
            details.style.display = isCollapsed ? 'none' : 'block';
        }
        </script>
        
        <?php elseif ($reportType === 'flights'): ?>
        <div class="card">
            <div class="card-header">
                <h3>✈️ טיסות ב-14 הימים הקרובים</h3>
            </div>
            <div class="card-body">
                <?php if (empty($reportData)): ?>
                <p style="text-align: center; color: #666; padding: 40px;">אין טיסות קרובות</p>
                <?php else: ?>
                <?php foreach ($reportData as $flight): 
                    $flightDate = strtotime($flight['departure_date']);
                    $daysLeft = ceil(($flightDate - time()) / 86400);
                    $daysClass = $daysLeft <= 2 ? 'days-urgent' : ($daysLeft <= 5 ? 'days-soon' : 'days-normal');
                ?>
                <div class="flight-card">
                    <div class="flight-date">
                        <?= date('d/m', $flightDate) ?>
                        <br><small style="font-weight: normal; font-size: 0.8rem;"><?= date('l', $flightDate) ?></small>
                    </div>
                    <div class="flight-info">
                        <div class="flight-customer"><?= htmlspecialchars($flight['customer_name'] ?? '-') ?></div>
                        <div class="flight-destination">✈️ <?= htmlspecialchars($flight['destination']) ?></div>
                        <small style="color: #666;">
                            <?php if ($flight['flight_supplier']): ?><span class="supplier-badge"><?= strtoupper($flight['flight_supplier']) ?></span><?php endif; ?>
                            <?php if ($flight['airline_outbound']): ?> • <?= $flight['airline_outbound'] ?><?php endif; ?>
                            <?php if ($flight['agent_name']): ?> • <?= $flight['agent_name'] ?><?php endif; ?>
                        </small>
                    </div>
                    <div>
                        <span class="days-left <?= $daysClass ?>">
                            <?= $daysLeft == 0 ? 'היום!' : ($daysLeft == 1 ? 'מחר' : "עוד $daysLeft ימים") ?>
                        </span>
                    </div>
                    <a href="view_order_new.php?id=<?= $flight['id'] ?>" class="btn btn-primary" style="font-size: 0.85rem;">צפה</a>
                </div>
                <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
        
        <?php elseif ($reportType === 'accountant'): ?>
        <div class="card">
            <div class="card-header">
                <h3>📋 דוח לרואה חשבון - טרנזקציות</h3>
                <form method="GET" class="filters">
                    <input type="hidden" name="report" value="accountant">
                    <div class="filter-group">
                        <label>מתאריך</label>
                        <input type="date" name="date_from" value="<?= $dateFrom ?>">
                    </div>
                    <div class="filter-group">
                        <label>עד תאריך</label>
                        <input type="date" name="date_to" value="<?= $dateTo ?>">
                    </div>
                    <button type="submit" class="btn btn-primary">הצג</button>
                    <a href="export_report.php?report=accountant&date_from=<?= $dateFrom ?>&date_to=<?= $dateTo ?>" class="btn btn-success">📥 Excel</a>
                    <button type="button" onclick="window.print()" class="btn" style="background: #6c757d; color: white;">🖨️ הדפס</button>
                </form>
            </div>
            <div class="card-body">
                <style>
                    .transaction-type {
                        font-size: 0.75rem;
                        padding: 3px 8px;
                        border-radius: 4px;
                    }
                    .type-order { background: #e3f2fd; color: #1565c0; }
                    .type-addition { background: #fff3e0; color: #e65100; }
                    .type-refund { background: #e8f5e9; color: #2e7d32; }
                    
                    /* Sortable columns */
                    .sortable {
                        cursor: pointer;
                        user-select: none;
                        position: relative;
                        padding-left: 20px !important;
                    }
                    .sortable:hover {
                        background: #e3f2fd;
                    }
                    .sortable::before {
                        content: '⇅';
                        position: absolute;
                        left: 5px;
                        opacity: 0.4;
                        font-size: 12px;
                    }
                    .sortable.asc::before {
                        content: '↑';
                        opacity: 1;
                        color: #1976d2;
                    }
                    .sortable.desc::before {
                        content: '↓';
                        opacity: 1;
                        color: #1976d2;
                    }
                </style>
                <table id="accountantTable">
                    <thead>
                        <tr>
                            <th class="sortable" data-col="0" data-type="date">תאריך</th>
                            <th class="sortable" data-col="1" data-type="number">הזמנה</th>
                            <th class="sortable" data-col="2" data-type="string">סוג טרנזקציה</th>
                            <th class="sortable" data-col="3" data-type="string">תיאור</th>
                            <th class="sortable" data-col="4" data-type="string">קבלות</th>
                            <th class="sortable" data-col="5" data-type="number">עלות (₪)</th>
                            <th class="sortable" data-col="6" data-type="number">הכנסה (₪)</th>
                            <th class="sortable" data-col="7" data-type="number">רווח (₪)</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($reportData as $row): 
                            $typeClass = 'type-order';
                            if (!empty($row['addition_id'])) $typeClass = 'type-addition';
                            if (!empty($row['refund_id'])) $typeClass = 'type-refund';
                        ?>
                        <tr>
                            <td data-sort="<?= $row['transaction_date'] ?>"><?= date('d/m/Y', strtotime($row['transaction_date'])) ?></td>
                            <td data-sort="<?= $row['order_id'] ?>"><?php if ($isAccountant): ?>#<?= $row['order_id'] ?><?php else: ?><a href="view_order_new.php?id=<?= $row['order_id'] ?>">#<?= $row['order_id'] ?></a><?php endif; ?></td>
                            <td data-sort="<?= htmlspecialchars($row['transaction_type']) ?>">
                                <span class="transaction-type <?= $typeClass ?>">
                                    <?= htmlspecialchars($row['transaction_type']) ?>
                                </span>
                            </td>
                            <td data-sort="<?= htmlspecialchars($row['description'] ?? '') ?>"><?= htmlspecialchars($row['description'] ?? '-') ?></td>
                            <td data-sort="<?= htmlspecialchars($row['receipt_numbers'] ?? '') ?>" style="font-family: monospace; font-size: 0.85rem;"><?= htmlspecialchars($row['receipt_numbers'] ?? '-') ?></td>
                            <td data-sort="<?= $row['cost_ils'] ?? 0 ?>">₪<?= number_format($row['cost_ils'] ?? 0) ?></td>
                            <td data-sort="<?= $row['income_ils'] ?? 0 ?>">₪<?= number_format($row['income_ils'] ?? 0) ?></td>
                            <td data-sort="<?= $row['profit_ils'] ?? 0 ?>" class="<?= ($row['profit_ils'] ?? 0) >= 0 ? 'positive' : 'negative' ?>">₪<?= number_format($row['profit_ils'] ?? 0) ?></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                    <tfoot style="background: #f8f9fa; font-weight: bold;">
                        <tr>
                            <td colspan="5">סה"כ (<?= count($reportData) ?> טרנזקציות)</td>
                            <td>₪<?= number_format(array_sum(array_column($reportData, 'cost_ils'))) ?></td>
                            <td>₪<?= number_format(array_sum(array_column($reportData, 'income_ils'))) ?></td>
                            <td class="positive">₪<?= number_format(array_sum(array_column($reportData, 'profit_ils'))) ?></td>
                        </tr>
                    </tfoot>
                </table>
                
                <script>
                document.addEventListener('DOMContentLoaded', function() {
                    const table = document.getElementById('accountantTable');
                    if (!table) return;
                    
                    const headers = table.querySelectorAll('th.sortable');
                    const tbody = table.querySelector('tbody');
                    
                    headers.forEach(header => {
                        header.addEventListener('click', function() {
                            const col = parseInt(this.dataset.col);
                            const type = this.dataset.type;
                            const isAsc = this.classList.contains('asc');
                            
                            // Remove sort classes from all headers
                            headers.forEach(h => h.classList.remove('asc', 'desc'));
                            
                            // Set new sort direction
                            this.classList.add(isAsc ? 'desc' : 'asc');
                            const direction = isAsc ? -1 : 1;
                            
                            // Get rows and sort
                            const rows = Array.from(tbody.querySelectorAll('tr'));
                            
                            rows.sort((a, b) => {
                                const aCell = a.cells[col];
                                const bCell = b.cells[col];
                                const aVal = aCell.dataset.sort || aCell.textContent.trim();
                                const bVal = bCell.dataset.sort || bCell.textContent.trim();
                                
                                if (type === 'number') {
                                    return (parseFloat(aVal) - parseFloat(bVal)) * direction;
                                } else if (type === 'date') {
                                    return (new Date(aVal) - new Date(bVal)) * direction;
                                } else {
                                    return aVal.localeCompare(bVal, 'he') * direction;
                                }
                            });
                            
                            // Re-append rows in new order
                            rows.forEach(row => tbody.appendChild(row));
                        });
                    });
                });
                </script>
            </div>
        </div>
        <?php endif; ?>
    </div>
</body>
</html>
