<?php
// הגדרות מערכת - אדמין
require_once 'includes/auth.php';
requireLogin();

// רק מנהל יכול לגשת להגדרות
$agent = getCurrentAgent();
if ($agent['role'] !== 'admin') {
    header('Location: complete_order_lite.php');
    exit;
}

$pdo = getDB();
$pdo->exec("SET NAMES utf8mb4");

// יצירת טבלאות אם לא קיימות
$pdo->exec("
    CREATE TABLE IF NOT EXISTS settings_suppliers (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(100) NOT NULL UNIQUE,
        active TINYINT(1) DEFAULT 1,
        sort_order INT DEFAULT 0,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    ) DEFAULT CHARSET=utf8mb4
");

$pdo->exec("
    CREATE TABLE IF NOT EXISTS settings_airlines (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(100) NOT NULL UNIQUE,
        code VARCHAR(10),
        active TINYINT(1) DEFAULT 1,
        sort_order INT DEFAULT 0,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    ) DEFAULT CHARSET=utf8mb4
");

$pdo->exec("
    CREATE TABLE IF NOT EXISTS settings_destinations (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(100) NOT NULL,
        country VARCHAR(100),
        active TINYINT(1) DEFAULT 1,
        sort_order INT DEFAULT 0,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        UNIQUE KEY unique_dest (name, country)
    ) DEFAULT CHARSET=utf8mb4
");

$pdo->exec("
    CREATE TABLE IF NOT EXISTS settings_hotels (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(200) NOT NULL,
        destination VARCHAR(100),
        stars INT,
        active TINYINT(1) DEFAULT 1,
        times_used INT DEFAULT 0,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    ) DEFAULT CHARSET=utf8mb4
");

// הכנסת נתוני ברירת מחדל אם הטבלאות ריקות
$count = $pdo->query("SELECT COUNT(*) FROM settings_suppliers")->fetchColumn();
if ($count == 0) {
    $pdo->exec("INSERT INTO settings_suppliers (name, sort_order) VALUES ('KISHE', 1), ('ESHET', 2), ('OR', 3), ('KAVEI', 4)");
}

$count = $pdo->query("SELECT COUNT(*) FROM settings_airlines")->fetchColumn();
if ($count == 0) {
    $pdo->exec("INSERT INTO settings_airlines (name, code, sort_order) VALUES ('Wizz Air', 'W6', 1), ('Ryanair', 'FR', 2), ('El Al', 'LY', 3), ('Aegean', 'A3', 4), ('Austrian', 'OS', 5), ('Lufthansa', 'LH', 6), ('Turkish Airlines', 'TK', 7), ('Israir', '6H', 8), ('Arkia', 'IZ', 9), ('EasyJet', 'U2', 10), ('Vueling', 'VY', 11), ('LOT', 'LO', 12)");
}

$count = $pdo->query("SELECT COUNT(*) FROM settings_destinations")->fetchColumn();
if ($count == 0) {
    $pdo->exec("INSERT INTO settings_destinations (name, country, sort_order) VALUES ('Budapest', 'Hungary', 1), ('Prague', 'Czech Republic', 2), ('Barcelona', 'Spain', 3), ('Rome', 'Italy', 4), ('Vienna', 'Austria', 5), ('Athens', 'Greece', 6), ('Krakow', 'Poland', 7), ('Warsaw', 'Poland', 8), ('Amsterdam', 'Netherlands', 9), ('Paris', 'France', 10), ('London', 'UK', 11), ('Lisbon', 'Portugal', 12), ('Madrid', 'Spain', 13), ('Milan', 'Italy', 14), ('Venice', 'Italy', 15), ('Dubai', 'UAE', 16), ('Larnaca', 'Cyprus', 17), ('Paphos', 'Cyprus', 18), ('Rhodes', 'Greece', 19), ('Santorini', 'Greece', 20), ('Istanbul', 'Turkey', 21), ('Antalya', 'Turkey', 22), ('Bangkok', 'Thailand', 23), ('Tbilisi', 'Georgia', 24), ('Batumi', 'Georgia', 25)");
}

// טיפול בפעולות POST
$message = '';
$messageType = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $table = $_POST['table'] ?? '';
    
    try {
        switch ($action) {
            case 'add_supplier':
                $name = trim($_POST['name']);
                if ($name) {
                    $stmt = $pdo->prepare("INSERT INTO settings_suppliers (name) VALUES (?)");
                    $stmt->execute([$name]);
                    $message = "ספק נוסף בהצלחה";
                    $messageType = 'success';
                }
                break;
                
            case 'add_airline':
                $name = trim($_POST['name']);
                $code = trim($_POST['code'] ?? '');
                if ($name) {
                    $stmt = $pdo->prepare("INSERT INTO settings_airlines (name, code) VALUES (?, ?)");
                    $stmt->execute([$name, $code]);
                    $message = "חברת תעופה נוספה בהצלחה";
                    $messageType = 'success';
                }
                break;
                
            case 'add_destination':
                $name = trim($_POST['name']);
                $country = trim($_POST['country'] ?? '');
                $nameHe = trim($_POST['name_he'] ?? '');
                if ($name) {
                    $stmt = $pdo->prepare("INSERT INTO settings_destinations (name, country, name_he) VALUES (?, ?, ?)");
                    $stmt->execute([$name, $country, $nameHe]);
                    $message = "יעד נוסף בהצלחה";
                    $messageType = 'success';
                }
                break;
                
            case 'add_hotel':
                $name = trim($_POST['name']);
                $destination = trim($_POST['destination'] ?? '');
                $stars = (int)($_POST['stars'] ?? 0);
                if ($name) {
                    $stmt = $pdo->prepare("INSERT INTO settings_hotels (name, destination, stars) VALUES (?, ?, ?)");
                    $stmt->execute([$name, $destination, $stars ?: null]);
                    $message = "מלון נוסף בהצלחה";
                    $messageType = 'success';
                }
                break;
                
            case 'edit_supplier':
                $id = (int)$_POST['id'];
                $name = trim($_POST['name']);
                if ($name && $id) {
                    $stmt = $pdo->prepare("UPDATE settings_suppliers SET name = ? WHERE id = ?");
                    $stmt->execute([$name, $id]);
                    $message = "ספק עודכן בהצלחה";
                    $messageType = 'success';
                }
                break;
                
            case 'edit_airline':
                $id = (int)$_POST['id'];
                $name = trim($_POST['name']);
                $code = trim($_POST['code'] ?? '');
                if ($name && $id) {
                    $stmt = $pdo->prepare("UPDATE settings_airlines SET name = ?, code = ? WHERE id = ?");
                    $stmt->execute([$name, $code, $id]);
                    $message = "חברת תעופה עודכנה בהצלחה";
                    $messageType = 'success';
                }
                break;
                
            case 'edit_destination':
                $id = (int)$_POST['id'];
                $name = trim($_POST['name']);
                $country = trim($_POST['country'] ?? '');
                $nameHe = trim($_POST['name_he'] ?? '');
                if ($name && $id) {
                    $stmt = $pdo->prepare("UPDATE settings_destinations SET name = ?, country = ?, name_he = ? WHERE id = ?");
                    $stmt->execute([$name, $country, $nameHe, $id]);
                    $message = "יעד עודכן בהצלחה";
                    $messageType = 'success';
                }
                break;
                
            case 'edit_hotel':
                $id = (int)$_POST['id'];
                $name = trim($_POST['name']);
                $destination = trim($_POST['destination'] ?? '');
                $stars = (int)($_POST['stars'] ?? 0);
                if ($name && $id) {
                    $stmt = $pdo->prepare("UPDATE settings_hotels SET name = ?, destination = ?, stars = ? WHERE id = ?");
                    $stmt->execute([$name, $destination, $stars ?: null, $id]);
                    $message = "מלון עודכן בהצלחה";
                    $messageType = 'success';
                }
                break;
                
            case 'delete':
                $id = (int)$_POST['id'];
                $tables = ['suppliers' => 'settings_suppliers', 'airlines' => 'settings_airlines', 'destinations' => 'settings_destinations', 'hotels' => 'settings_hotels'];
                if (isset($tables[$table])) {
                    $stmt = $pdo->prepare("DELETE FROM {$tables[$table]} WHERE id = ?");
                    $stmt->execute([$id]);
                    $message = "הפריט נמחק בהצלחה";
                    $messageType = 'success';
                }
                break;
                
            case 'toggle':
                $id = (int)$_POST['id'];
                $tables = ['suppliers' => 'settings_suppliers', 'airlines' => 'settings_airlines', 'destinations' => 'settings_destinations', 'hotels' => 'settings_hotels'];
                if (isset($tables[$table])) {
                    $stmt = $pdo->prepare("UPDATE {$tables[$table]} SET active = NOT active WHERE id = ?");
                    $stmt->execute([$id]);
                    $message = "הסטטוס עודכן";
                    $messageType = 'success';
                }
                break;
        }
    } catch (PDOException $e) {
        if (strpos($e->getMessage(), 'Duplicate') !== false) {
            $message = "פריט זה כבר קיים במערכת";
        } else {
            $message = "שגיאה: " . $e->getMessage();
        }
        $messageType = 'error';
    }
}

// שליפת הנתונים
$suppliers = $pdo->query("SELECT * FROM settings_suppliers ORDER BY sort_order, name")->fetchAll();
$airlines = $pdo->query("SELECT * FROM settings_airlines ORDER BY sort_order, name")->fetchAll();
$destinations = $pdo->query("SELECT * FROM settings_destinations ORDER BY sort_order, name")->fetchAll();
$hotels = $pdo->query("SELECT * FROM settings_hotels ORDER BY destination, name")->fetchAll();

// קיבוץ מלונות לפי יעד
$hotelsByDest = [];
foreach ($hotels as $h) {
    $dest = $h['destination'] ?: 'ללא יעד';
    $hotelsByDest[$dest][] = $h;
}

$activeTab = $_GET['tab'] ?? 'suppliers';
?>
<!DOCTYPE html>
<html lang="he" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>הגדרות מערכת | CRM</title>
    <style>
        * { box-sizing: border-box; margin: 0; padding: 0; }
        body { font-family: 'Segoe UI', Tahoma, Arial, sans-serif; background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%); min-height: 100vh; padding: 20px; }
        .container { max-width: 1200px; margin: 0 auto; }
        .header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px; }
        .header h1 { color: white; }
        .back-btn { background: #6c757d; color: white; padding: 10px 20px; text-decoration: none; border-radius: 8px; }
        .message { padding: 15px 20px; border-radius: 8px; margin-bottom: 20px; font-weight: 500; }
        .message.success { background: #d4edda; color: #155724; }
        .message.error { background: #f8d7da; color: #721c24; }
        .tabs { display: flex; gap: 5px; margin-bottom: 20px; flex-wrap: wrap; }
        .tab { padding: 12px 25px; background: rgba(255,255,255,0.1); color: white; text-decoration: none; border-radius: 8px 8px 0 0; font-weight: 500; }
        .tab:hover { background: rgba(255,255,255,0.2); }
        .tab.active { background: white; color: #333; }
        .content { background: white; border-radius: 0 10px 10px 10px; padding: 25px; }
        .section-title { font-size: 1.3rem; margin-bottom: 20px; color: #333; }
        .add-form { background: #f8f9fa; padding: 20px; border-radius: 10px; margin-bottom: 25px; display: flex; gap: 15px; flex-wrap: wrap; align-items: flex-end; }
        .form-group { display: flex; flex-direction: column; gap: 5px; }
        .form-group label { font-size: 0.85rem; color: #666; font-weight: 500; }
        .form-group input, .form-group select { padding: 10px 15px; border: 2px solid #e0e0e0; border-radius: 8px; font-size: 0.95rem; min-width: 180px; }
        .form-group input:focus, .form-group select:focus { border-color: #667eea; outline: none; }
        .btn-add { padding: 10px 25px; background: #28a745; color: white; border: none; border-radius: 8px; cursor: pointer; font-size: 0.95rem; font-weight: 500; }
        .btn-add:hover { background: #218838; }
        .items-grid { display: grid; grid-template-columns: repeat(auto-fill, minmax(300px, 1fr)); gap: 10px; }
        .item { display: flex; justify-content: space-between; align-items: center; padding: 15px; background: #f8f9fa; border-radius: 8px; border: 1px solid #e0e0e0; }
        .item:hover { box-shadow: 0 2px 8px rgba(0,0,0,0.1); }
        .item.inactive { opacity: 0.5; background: #fff; }
        .item-info { flex: 1; }
        .item-name { font-weight: 600; color: #333; font-size: 1.05rem; }
        .item-sub { font-size: 0.8rem; color: #666; margin-top: 3px; }
        .item-actions { display: flex; gap: 5px; align-items: center; }
        .btn-icon { width: 32px; height: 32px; border: none; border-radius: 6px; cursor: pointer; display: flex; align-items: center; justify-content: center; font-size: 0.9rem; text-decoration: none; }
        .btn-stats { background: #e8f4fd; color: #1976d2; }
        .btn-stats:hover { background: #1976d2; color: white; }
        .btn-edit { background: #fff3cd; color: #856404; }
        .btn-edit:hover { background: #ffc107; color: #333; }
        .btn-toggle { background: #d4edda; color: #155724; }
        .btn-toggle.off { background: #e9ecef; color: #6c757d; }
        .btn-delete { background: #f8d7da; color: #721c24; }
        .btn-delete:hover { background: #dc3545; color: white; }
        .dest-group { margin-bottom: 20px; }
        .dest-title { font-weight: 600; color: #667eea; margin-bottom: 10px; padding-bottom: 5px; border-bottom: 2px solid #667eea; }
        .stats { display: flex; gap: 20px; margin-bottom: 20px; flex-wrap: wrap; }
        .stat { background: #e8f4fd; padding: 15px 25px; border-radius: 10px; text-align: center; }
        .stat-value { font-size: 1.5rem; font-weight: bold; color: #1976d2; }
        .stat-label { font-size: 0.85rem; color: #666; }
        .empty-state { text-align: center; padding: 30px; color: #666; }
        .modal-overlay { display: none; position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.5); z-index: 1000; align-items: center; justify-content: center; }
        .modal-overlay.show { display: flex; }
        .modal { background: white; padding: 25px; border-radius: 15px; width: 90%; max-width: 450px; box-shadow: 0 10px 40px rgba(0,0,0,0.3); }
        .modal h3 { margin-bottom: 20px; color: #333; }
        .modal .form-group { margin-bottom: 15px; }
        .modal .form-group input, .modal .form-group select { width: 100%; }
        .modal-actions { display: flex; gap: 10px; justify-content: flex-end; margin-top: 20px; }
        .modal-actions button { padding: 10px 25px; border: none; border-radius: 8px; cursor: pointer; font-size: 0.95rem; }
        .btn-save { background: #28a745; color: white; }
        .btn-cancel { background: #6c757d; color: white; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>⚙️ הגדרות מערכת</h1>
            <a href="orders_new.php" class="back-btn">← חזרה</a>
        </div>

        <?php if ($message): ?>
            <div class="message <?= $messageType ?>"><?= htmlspecialchars($message) ?></div>
        <?php endif; ?>

        <div class="tabs">
            <a href="?tab=suppliers" class="tab <?= $activeTab === 'suppliers' ? 'active' : '' ?>">📦 ספקים</a>
            <a href="?tab=airlines" class="tab <?= $activeTab === 'airlines' ? 'active' : '' ?>">✈️ חברות תעופה</a>
            <a href="?tab=destinations" class="tab <?= $activeTab === 'destinations' ? 'active' : '' ?>">🌍 יעדים</a>
            <a href="?tab=hotels" class="tab <?= $activeTab === 'hotels' ? 'active' : '' ?>">🏨 בתי מלון</a>
            <a href="?tab=agents" class="tab <?= $activeTab === 'agents' ? 'active' : '' ?>">👥 נציגים</a>
            <a href="?tab=cardcom" class="tab <?= $activeTab === 'cardcom' ? 'active' : '' ?>">💳 Cardcom</a>
            <a href="?tab=system" class="tab <?= $activeTab === 'system' ? 'active' : '' ?>">🔧 מערכת</a>
        </div>

        <div class="content">
            <?php if ($activeTab === 'suppliers'): ?>
                <h2 class="section-title">📦 ניהול ספקים</h2>
                <div class="stats">
                    <div class="stat"><div class="stat-value"><?= count($suppliers) ?></div><div class="stat-label">סה"כ ספקים</div></div>
                    <div class="stat"><div class="stat-value"><?= count(array_filter($suppliers, function($s) { return $s['active']; })) ?></div><div class="stat-label">פעילים</div></div>
                </div>
                <form method="POST" class="add-form">
                    <input type="hidden" name="action" value="add_supplier">
                    <div class="form-group"><label>שם ספק חדש:</label><input type="text" name="name" required placeholder="למשל: GAYA"></div>
                    <button type="submit" class="btn-add">+ הוסף ספק</button>
                </form>
                <div class="items-grid">
                    <?php foreach ($suppliers as $s): ?>
                        <div class="item <?= $s['active'] ? '' : 'inactive' ?>">
                            <div class="item-info"><div class="item-name"><?= htmlspecialchars($s['name']) ?></div></div>
                            <div class="item-actions">
                                <a href="entity_details.php?type=supplier&name=<?= urlencode($s['name']) ?>" class="btn-icon btn-stats" title="סטטיסטיקות">📊</a>
                                <button type="button" class="btn-icon btn-edit" onclick="openEditModal('supplier', <?= $s['id'] ?>, '<?= htmlspecialchars($s['name'], ENT_QUOTES) ?>')" title="עריכה">✏️</button>
                                <form method="POST" style="display:inline;"><input type="hidden" name="action" value="toggle"><input type="hidden" name="table" value="suppliers"><input type="hidden" name="id" value="<?= $s['id'] ?>"><button type="submit" class="btn-icon btn-toggle <?= $s['active'] ? '' : 'off' ?>" title="<?= $s['active'] ? 'פעיל' : 'מושבת' ?>"><?= $s['active'] ? '✓' : '✗' ?></button></form>
                                <form method="POST" style="display:inline;" onsubmit="return confirm('למחוק?')"><input type="hidden" name="action" value="delete"><input type="hidden" name="table" value="suppliers"><input type="hidden" name="id" value="<?= $s['id'] ?>"><button type="submit" class="btn-icon btn-delete" title="מחיקה">🗑️</button></form>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>

            <?php elseif ($activeTab === 'airlines'): ?>
                <h2 class="section-title">✈️ ניהול חברות תעופה</h2>
                <div class="stats">
                    <div class="stat"><div class="stat-value"><?= count($airlines) ?></div><div class="stat-label">סה"כ חברות</div></div>
                    <div class="stat"><div class="stat-value"><?= count(array_filter($airlines, function($a) { return $a['active']; })) ?></div><div class="stat-label">פעילות</div></div>
                </div>
                <form method="POST" class="add-form">
                    <input type="hidden" name="action" value="add_airline">
                    <div class="form-group"><label>שם חברת תעופה:</label><input type="text" name="name" required placeholder="למשל: Swiss Air"></div>
                    <div class="form-group"><label>קוד:</label><input type="text" name="code" placeholder="LX" style="min-width:80px;"></div>
                    <button type="submit" class="btn-add">+ הוסף חברה</button>
                </form>
                <div class="items-grid">
                    <?php foreach ($airlines as $a): ?>
                        <div class="item <?= $a['active'] ? '' : 'inactive' ?>">
                            <div class="item-info"><div class="item-name"><?= htmlspecialchars($a['name']) ?></div><?php if ($a['code']): ?><div class="item-sub">קוד: <?= htmlspecialchars($a['code']) ?></div><?php endif; ?></div>
                            <div class="item-actions">
                                <a href="entity_details.php?type=airline&name=<?= urlencode($a['name']) ?>" class="btn-icon btn-stats" title="סטטיסטיקות">📊</a>
                                <button type="button" class="btn-icon btn-edit" onclick="openEditModal('airline', <?= $a['id'] ?>, '<?= htmlspecialchars($a['name'], ENT_QUOTES) ?>', '<?= htmlspecialchars($a['code'] ?? '', ENT_QUOTES) ?>')" title="עריכה">✏️</button>
                                <form method="POST" style="display:inline;"><input type="hidden" name="action" value="toggle"><input type="hidden" name="table" value="airlines"><input type="hidden" name="id" value="<?= $a['id'] ?>"><button type="submit" class="btn-icon btn-toggle <?= $a['active'] ? '' : 'off' ?>"><?= $a['active'] ? '✓' : '✗' ?></button></form>
                                <form method="POST" style="display:inline;" onsubmit="return confirm('למחוק?')"><input type="hidden" name="action" value="delete"><input type="hidden" name="table" value="airlines"><input type="hidden" name="id" value="<?= $a['id'] ?>"><button type="submit" class="btn-icon btn-delete">🗑️</button></form>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>

            <?php elseif ($activeTab === 'destinations'): ?>
                <h2 class="section-title">🌍 ניהול יעדים</h2>
                <div class="stats"><div class="stat"><div class="stat-value"><?= count($destinations) ?></div><div class="stat-label">סה"כ יעדים</div></div></div>
                <form method="POST" class="add-form">
                    <input type="hidden" name="action" value="add_destination">
                    <div class="form-group"><label>שם העיר (אנגלית):</label><input type="text" name="name" required placeholder="למשל: Berlin"></div>
                    <div class="form-group"><label>מדינה (אנגלית):</label><input type="text" name="country" placeholder="Germany"></div>
                    <div class="form-group"><label>שם בעברית:</label><input type="text" name="name_he" placeholder="ברלין"></div>
                    <button type="submit" class="btn-add">+ הוסף יעד</button>
                </form>
                <div class="items-grid">
                    <?php foreach ($destinations as $d): ?>
                        <div class="item <?= $d['active'] ? '' : 'inactive' ?>">
                            <div class="item-info"><div class="item-name"><?= htmlspecialchars($d['name']) ?><?php if (!empty($d['name_he'])): ?> <span style="color:#666;">(<?= htmlspecialchars($d['name_he']) ?>)</span><?php endif; ?></div><?php if ($d['country']): ?><div class="item-sub"><?= htmlspecialchars($d['country']) ?></div><?php endif; ?></div>
                            <div class="item-actions">
                                <a href="entity_details.php?type=destination&name=<?= urlencode($d['name']) ?>" class="btn-icon btn-stats" title="סטטיסטיקות">📊</a>
                                <button type="button" class="btn-icon btn-edit" onclick="openEditModal('destination', <?= $d['id'] ?>, '<?= htmlspecialchars($d['name'], ENT_QUOTES) ?>', '<?= htmlspecialchars($d['country'] ?? '', ENT_QUOTES) ?>', '<?= htmlspecialchars($d['name_he'] ?? '', ENT_QUOTES) ?>')" title="עריכה">✏️</button>
                                <form method="POST" style="display:inline;"><input type="hidden" name="action" value="toggle"><input type="hidden" name="table" value="destinations"><input type="hidden" name="id" value="<?= $d['id'] ?>"><button type="submit" class="btn-icon btn-toggle <?= $d['active'] ? '' : 'off' ?>"><?= $d['active'] ? '✓' : '✗' ?></button></form>
                                <form method="POST" style="display:inline;" onsubmit="return confirm('למחוק?')"><input type="hidden" name="action" value="delete"><input type="hidden" name="table" value="destinations"><input type="hidden" name="id" value="<?= $d['id'] ?>"><button type="submit" class="btn-icon btn-delete">🗑️</button></form>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>

            <?php elseif ($activeTab === 'hotels'): ?>
                <h2 class="section-title">🏨 ניהול בתי מלון</h2>
                <div class="stats">
                    <div class="stat"><div class="stat-value"><?= count($hotels) ?></div><div class="stat-label">סה"כ מלונות</div></div>
                    <div class="stat"><div class="stat-value"><?= count($hotelsByDest) ?></div><div class="stat-label">יעדים</div></div>
                </div>
                <form method="POST" class="add-form">
                    <input type="hidden" name="action" value="add_hotel">
                    <div class="form-group"><label>שם המלון:</label><input type="text" name="name" required placeholder="Hilton Budapest"></div>
                    <div class="form-group"><label>יעד:</label><select name="destination"><option value="">בחר יעד</option><?php foreach ($destinations as $d): ?><option value="<?= htmlspecialchars($d['name']) ?>"><?= htmlspecialchars($d['name']) ?></option><?php endforeach; ?></select></div>
                    <div class="form-group"><label>כוכבים:</label><select name="stars" style="min-width:80px;"><option value="">-</option><option value="3">3⭐</option><option value="4">4⭐</option><option value="5">5⭐</option></select></div>
                    <button type="submit" class="btn-add">+ הוסף מלון</button>
                </form>
                <?php if (empty($hotelsByDest)): ?>
                    <div class="empty-state"><p>עדיין אין מלונות. הוסף את המלון הראשון!</p></div>
                <?php else: ?>
                    <?php foreach ($hotelsByDest as $dest => $destHotels): ?>
                        <div class="dest-group">
                            <div class="dest-title">📍 <?= htmlspecialchars($dest) ?> (<?= count($destHotels) ?>)</div>
                            <div class="items-grid">
                                <?php foreach ($destHotels as $h): ?>
                                    <div class="item <?= $h['active'] ? '' : 'inactive' ?>">
                                        <div class="item-info"><div class="item-name"><?= htmlspecialchars($h['name']) ?></div><div class="item-sub"><?php if ($h['stars']): echo str_repeat('⭐', $h['stars']); endif; ?></div></div>
                                        <div class="item-actions">
                                            <a href="entity_details.php?type=hotel&name=<?= urlencode($h['name']) ?>" class="btn-icon btn-stats" title="סטטיסטיקות">📊</a>
                                            <button type="button" class="btn-icon btn-edit" onclick="openEditModal('hotel', <?= $h['id'] ?>, '<?= htmlspecialchars($h['name'], ENT_QUOTES) ?>', '<?= htmlspecialchars($h['destination'] ?? '', ENT_QUOTES) ?>', <?= $h['stars'] ?? 0 ?>)" title="עריכה">✏️</button>
                                            <form method="POST" style="display:inline;" onsubmit="return confirm('למחוק?')"><input type="hidden" name="action" value="delete"><input type="hidden" name="table" value="hotels"><input type="hidden" name="id" value="<?= $h['id'] ?>"><button type="submit" class="btn-icon btn-delete">🗑️</button></form>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>

            <?php elseif ($activeTab === 'agents'): ?>
                <h2 class="section-title">👥 ניהול נציגים</h2>
                <?php $agentsList = $pdo->query("SELECT * FROM agents ORDER BY full_name")->fetchAll(); ?>
                <div class="stats"><div class="stat"><div class="stat-value"><?= count($agentsList) ?></div><div class="stat-label">סה"כ נציגים</div></div></div>
                <div class="items-grid">
                    <?php foreach ($agentsList as $ag): ?>
                        <div class="item">
                            <div class="item-info"><div class="item-name"><?= htmlspecialchars($ag['full_name']) ?></div><div class="item-sub"><?= htmlspecialchars($ag['username']) ?><?php if ($ag['role'] === 'admin'): ?> · <span style="color:#667eea;">מנהל</span><?php endif; ?></div></div>
                        </div>
                    <?php endforeach; ?>
                </div>
                <p style="margin-top:20px; color:#666;">💡 להוספת נציג חדש, צור קשר עם מנהל המערכת</p>
            
            <?php elseif ($activeTab === 'cardcom'): ?>
                <h2 class="section-title">💳 הגדרות Cardcom</h2>
                <?php
                // וודא שהטבלה קיימת
                $pdo->exec("CREATE TABLE IF NOT EXISTS cardcom_settings (
                    id INT PRIMARY KEY DEFAULT 1,
                    terminal_number VARCHAR(50),
                    api_name VARCHAR(100),
                    api_password VARCHAR(200),
                    success_url VARCHAR(500),
                    failure_url VARCHAR(500),
                    notify_url VARCHAR(500),
                    is_test_mode TINYINT(1) DEFAULT 1,
                    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
                ) DEFAULT CHARSET=utf8mb4");
                $pdo->exec("INSERT IGNORE INTO cardcom_settings (id) VALUES (1)");
                
                // שמירת הגדרות
                if (isset($_POST['action']) && $_POST['action'] === 'save_cardcom') {
                    $stmt = $pdo->prepare("UPDATE cardcom_settings SET 
                        terminal_number = ?,
                        api_name = ?,
                        api_password = ?,
                        is_test_mode = ?
                        WHERE id = 1");
                    $stmt->execute([
                        $_POST['terminal_number'],
                        $_POST['api_name'],
                        $_POST['api_password'],
                        isset($_POST['is_test_mode']) ? 1 : 0
                    ]);
                    echo '<div class="message success">✅ ההגדרות נשמרו בהצלחה!</div>';
                }
                
                // שליפת ההגדרות
                $cardcom = $pdo->query("SELECT * FROM cardcom_settings WHERE id = 1")->fetch();
                ?>
                
                <div style="background:#e3f2fd; padding:20px; border-radius:10px; margin-bottom:25px;">
                    <h4 style="margin:0 0 10px 0; color:#1976d2;">ℹ️ מידע על Cardcom</h4>
                    <p style="margin:0; color:#555;">Cardcom היא מערכת סליקה ישראלית לקבלת תשלומי אשראי באינטרנט.</p>
                    <p style="margin:10px 0 0 0; color:#555;">לקבלת פרטי API, צור קשר עם Cardcom: <a href="https://www.cardcom.solutions" target="_blank">www.cardcom.solutions</a></p>
                </div>
                
                <form method="POST" style="max-width:500px;">
                    <input type="hidden" name="action" value="save_cardcom">
                    
                    <div class="form-group" style="margin-bottom:20px;">
                        <label style="display:block; margin-bottom:8px; font-weight:600;">Terminal Number</label>
                        <input type="text" name="terminal_number" value="<?= htmlspecialchars($cardcom['terminal_number'] ?? '') ?>" 
                               style="width:100%; padding:12px; border:2px solid #e0e0e0; border-radius:8px; font-size:1rem;"
                               placeholder="מספר הטרמינל שקיבלת מ-Cardcom">
                    </div>
                    
                    <div class="form-group" style="margin-bottom:20px;">
                        <label style="display:block; margin-bottom:8px; font-weight:600;">API Name (Username)</label>
                        <input type="text" name="api_name" value="<?= htmlspecialchars($cardcom['api_name'] ?? '') ?>" 
                               style="width:100%; padding:12px; border:2px solid #e0e0e0; border-radius:8px; font-size:1rem;"
                               placeholder="שם המשתמש ל-API">
                    </div>
                    
                    <div class="form-group" style="margin-bottom:20px;">
                        <label style="display:block; margin-bottom:8px; font-weight:600;">API Password</label>
                        <input type="password" name="api_password" value="<?= htmlspecialchars($cardcom['api_password'] ?? '') ?>" 
                               style="width:100%; padding:12px; border:2px solid #e0e0e0; border-radius:8px; font-size:1rem;"
                               placeholder="הסיסמה ל-API">
                    </div>
                    
                    <div class="form-group" style="margin-bottom:25px;">
                        <label style="display:flex; align-items:center; gap:10px; cursor:pointer;">
                            <input type="checkbox" name="is_test_mode" <?= ($cardcom['is_test_mode'] ?? 1) ? 'checked' : '' ?> 
                                   style="width:20px; height:20px;">
                            <span style="font-weight:600;">מצב בדיקות (Test Mode)</span>
                        </label>
                        <p style="margin:8px 0 0 30px; color:#666; font-size:0.9rem;">סמן בזמן פיתוח. בטל את הסימון כשעוברים לייצור.</p>
                    </div>
                    
                    <button type="submit" style="background:linear-gradient(135deg, #667eea, #764ba2); color:white; border:none; padding:15px 40px; border-radius:10px; font-size:1.1rem; cursor:pointer; font-weight:600;">
                        💾 שמור הגדרות
                    </button>
                </form>
                
                <div style="margin-top:30px; padding:20px; background:#f8f9fa; border-radius:10px;">
                    <h4 style="margin:0 0 15px 0;">🔗 קישורים למערכת</h4>
                    <p style="margin:5px 0;"><strong>דף טיוטות:</strong> <a href="drafts_list.php">drafts_list.php</a></p>
                    <p style="margin:5px 0;"><strong>יצירת טיוטה:</strong> <a href="draft.php">draft.php</a></p>
                </div>
                
            <?php elseif ($activeTab === 'system'): ?>
                <h2 class="section-title">🔧 הגדרות מערכת</h2>
                
                <?php
                // סטטיסטיקות
                $ordersCount = $pdo->query("SELECT COUNT(*) FROM orders")->fetchColumn();
                $draftsCount = $pdo->query("SELECT COUNT(*) FROM drafts")->fetchColumn();
                
                // טיפול במחיקת נתונים
                if (isset($_POST['action']) && $_POST['action'] === 'delete_all_data') {
                    $confirmText = $_POST['confirm_text'] ?? '';
                    
                    if ($confirmText === 'מחק הכל') {
                        try {
                            // מחיקת כל טבלאות הנתונים (לא הגדרות!)
                            // משתמשים ב-try נפרד לכל טבלה כדי שלא תכשל אם טבלה לא קיימת
                            $tablesToDelete = ['orders', 'drafts', 'draft_payments', 'pending_payments', 'bank_transfers', 'order_additions'];
                            
                            foreach ($tablesToDelete as $table) {
                                try {
                                    $pdo->exec("DELETE FROM $table");
                                    $pdo->exec("ALTER TABLE $table AUTO_INCREMENT = 1");
                                } catch (Exception $e) {
                                    // טבלה לא קיימת - ממשיכים
                                }
                            }
                            
                            // מחיקת קבצים - וואצ'רים
                            $uploadDir = 'uploads/vouchers/';
                            if (is_dir($uploadDir)) {
                                $files = new RecursiveIteratorIterator(
                                    new RecursiveDirectoryIterator($uploadDir, RecursiveDirectoryIterator::SKIP_DOTS),
                                    RecursiveIteratorIterator::CHILD_FIRST
                                );
                                foreach ($files as $file) {
                                    if ($file->isDir()) {
                                        @rmdir($file->getRealPath());
                                    } else {
                                        @unlink($file->getRealPath());
                                    }
                                }
                            }
                            
                            // מחיקת קבצים - צילומי מסך העברות בנקאיות
                            $bankUploadsDir = 'uploads/bank_transfers/';
                            if (is_dir($bankUploadsDir)) {
                                $files = new RecursiveIteratorIterator(
                                    new RecursiveDirectoryIterator($bankUploadsDir, RecursiveDirectoryIterator::SKIP_DOTS),
                                    RecursiveIteratorIterator::CHILD_FIRST
                                );
                                foreach ($files as $file) {
                                    if ($file->isDir()) {
                                        @rmdir($file->getRealPath());
                                    } else {
                                        @unlink($file->getRealPath());
                                    }
                                }
                            }
                            
                            echo '<div class="message success">✅ כל הנתונים נמחקו בהצלחה!</div>';
                            $ordersCount = 0;
                            $draftsCount = 0;
                        } catch (Exception $e) {
                            echo '<div class="message error">❌ שגיאה: ' . $e->getMessage() . '</div>';
                        }
                    } else {
                        echo '<div class="message error">❌ טקסט האישור לא נכון. הקלד "מחק הכל" בדיוק.</div>';
                    }
                }
                ?>
                
                <div class="stats">
                    <div class="stat"><div class="stat-value"><?= $ordersCount ?></div><div class="stat-label">הזמנות</div></div>
                    <div class="stat"><div class="stat-value"><?= $draftsCount ?></div><div class="stat-label">טיוטות</div></div>
                </div>
                
                <div style="background:#fff5f5; border:2px solid #dc3545; border-radius:10px; padding:25px; margin-top:30px;">
                    <div style="text-align:center; font-size:3rem; margin-bottom:15px;">⚠️</div>
                    <h3 style="color:#dc3545; margin-bottom:15px; text-align:center;">אזור מסוכן - מחיקת כל הנתונים</h3>
                    <p style="color:#666; margin-bottom:20px; text-align:center;">
                        פעולה זו תמחק את כל הנתונים העסקיים:<br>
                        הזמנות, טיוטות, תשלומים, העברות בנקאיות, תוספות וקבצים.<br>
                        <strong style="color:#dc3545;">הגדרות (ספקים, חברות תעופה, יעדים) לא יימחקו.</strong><br>
                        <strong>אי אפשר לשחזר נתונים שנמחקו!</strong>
                    </p>
                    
                    <form method="POST" onsubmit="return confirmDelete()" style="max-width:400px; margin:0 auto;">
                        <input type="hidden" name="action" value="delete_all_data">
                        <input 
                            type="text" 
                            name="confirm_text" 
                            id="confirmInput"
                            placeholder='הקלד "מחק הכל" לאישור'
                            autocomplete="off"
                            style="width:100%; padding:12px; border:2px solid #ddd; border-radius:8px; font-size:1rem; margin-bottom:15px; text-align:center;"
                        >
                        <button 
                            type="submit" 
                            id="deleteBtn"
                            disabled
                            style="width:100%; padding:15px; background:#dc3545; color:white; border:none; border-radius:8px; font-size:1rem; cursor:pointer; font-weight:bold; opacity:0.5;"
                        >
                            🗑️ מחק את כל הנתונים
                        </button>
                    </form>
                </div>
                
                <script>
                    const confirmInput = document.getElementById('confirmInput');
                    const deleteBtn = document.getElementById('deleteBtn');
                    
                    if (confirmInput && deleteBtn) {
                        confirmInput.addEventListener('input', function() {
                            const isValid = this.value === 'מחק הכל';
                            deleteBtn.disabled = !isValid;
                            deleteBtn.style.opacity = isValid ? '1' : '0.5';
                        });
                    }
                    
                    function confirmDelete() {
                        return confirm('האם אתה בטוח לחלוטין?\n\nכל הנתונים יימחקו לצמיתות!');
                    }
                </script>
                
            <?php endif; ?>
        </div>
    </div>

    <!-- Edit Modal -->
    <div class="modal-overlay" id="editModal">
        <div class="modal">
            <h3 id="modalTitle">עריכה</h3>
            <form method="POST" id="editForm">
                <input type="hidden" name="action" id="editAction">
                <input type="hidden" name="id" id="editId">
                <div class="form-group"><label>שם:</label><input type="text" name="name" id="editName" required></div>
                <div class="form-group" id="editCodeGroup" style="display:none;"><label>קוד:</label><input type="text" name="code" id="editCode"></div>
                <div class="form-group" id="editCountryGroup" style="display:none;"><label>מדינה:</label><input type="text" name="country" id="editCountry"></div>
                <div class="form-group" id="editNameHeGroup" style="display:none;"><label>שם בעברית:</label><input type="text" name="name_he" id="editNameHe"></div>
                <div class="form-group" id="editDestinationGroup" style="display:none;"><label>יעד:</label><select name="destination" id="editDestination"><option value="">בחר יעד</option><?php foreach ($destinations as $d): ?><option value="<?= htmlspecialchars($d['name']) ?>"><?= htmlspecialchars($d['name']) ?></option><?php endforeach; ?></select></div>
                <div class="form-group" id="editStarsGroup" style="display:none;"><label>כוכבים:</label><select name="stars" id="editStars"><option value="">-</option><option value="3">3⭐</option><option value="4">4⭐</option><option value="5">5⭐</option></select></div>
                <div class="modal-actions"><button type="button" class="btn-cancel" onclick="closeEditModal()">ביטול</button><button type="submit" class="btn-save">💾 שמור</button></div>
            </form>
        </div>
    </div>

    <script>
    function openEditModal(type, id, name, extra1, extra2) {
        document.getElementById('editCodeGroup').style.display = 'none';
        document.getElementById('editCountryGroup').style.display = 'none';
        document.getElementById('editNameHeGroup').style.display = 'none';
        document.getElementById('editDestinationGroup').style.display = 'none';
        document.getElementById('editStarsGroup').style.display = 'none';
        
        document.getElementById('editId').value = id;
        document.getElementById('editName').value = name;
        
        if (type === 'supplier') {
            document.getElementById('modalTitle').textContent = '✏️ עריכת ספק';
            document.getElementById('editAction').value = 'edit_supplier';
        } else if (type === 'airline') {
            document.getElementById('modalTitle').textContent = '✏️ עריכת חברת תעופה';
            document.getElementById('editAction').value = 'edit_airline';
            document.getElementById('editCodeGroup').style.display = 'block';
            document.getElementById('editCode').value = extra1 || '';
        } else if (type === 'destination') {
            document.getElementById('modalTitle').textContent = '✏️ עריכת יעד';
            document.getElementById('editAction').value = 'edit_destination';
            document.getElementById('editCountryGroup').style.display = 'block';
            document.getElementById('editNameHeGroup').style.display = 'block';
            document.getElementById('editCountry').value = extra1 || '';
            document.getElementById('editNameHe').value = extra2 || '';
        } else if (type === 'hotel') {
            document.getElementById('modalTitle').textContent = '✏️ עריכת מלון';
            document.getElementById('editAction').value = 'edit_hotel';
            document.getElementById('editDestinationGroup').style.display = 'block';
            document.getElementById('editStarsGroup').style.display = 'block';
            document.getElementById('editDestination').value = extra1 || '';
            document.getElementById('editStars').value = extra2 || '';
        }
        
        document.getElementById('editModal').classList.add('show');
    }
    
    function closeEditModal() { document.getElementById('editModal').classList.remove('show'); }
    document.getElementById('editModal').addEventListener('click', function(e) { if (e.target === this) closeEditModal(); });
    document.addEventListener('keydown', function(e) { if (e.key === 'Escape') closeEditModal(); });
    </script>
</body>
</html>
