<?php
// התאמות ספקים - Supplier Reconciliation
require_once 'includes/auth.php';
requireLogin();

$pdo = getDB();
$error = '';
$success = '';
$matches = [];
$mismatches = [];
$notFound = [];
$supplierData = [];
$selectedSupplier = $_POST['supplier'] ?? $_GET['supplier'] ?? '';

// שליפת ספקים
$suppliers = $pdo->query("
    SELECT DISTINCT supplier FROM (
        SELECT flight_supplier as supplier FROM orders WHERE flight_supplier IS NOT NULL AND flight_supplier != ''
        UNION
        SELECT return_flight_supplier as supplier FROM orders WHERE return_flight_supplier IS NOT NULL AND return_flight_supplier != ''
    ) as all_suppliers ORDER BY supplier
")->fetchAll(PDO::FETCH_COLUMN);

// עיבוד טקסט מה-PDF
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['pdf_text'])) {
    $pdfText = $_POST['pdf_text'];
    $selectedSupplier = strtoupper($_POST['supplier'] ?? '');
    
    if (empty($selectedSupplier)) {
        $error = 'יש לבחור ספק';
    } elseif (empty($pdfText)) {
        $error = 'יש להדביק את תוכן הדוח';
    } else {
        // פרסור הטקסט - מחפש שורות עם מספרי הזמנה
        $lines = explode("\n", $pdfText);
        $bookings = [];
        
        foreach ($lines as $line) {
            // מחפש מספר הזמנה (5-10 ספרות) וסכום בכל מטבע (USD, EUR, ILS, או סימן מטבע)
            if (preg_match('/(\d{5,10}).*?(USD|EUR|ILS|\$|€|₪)\s*([\d,]+\.?\d*)\s+([\d,]+\.?\d*)\s+([\d,]+\.?\d*)/', $line, $matches_line)) {
                $bookingCode = $matches_line[1];
                $currencyRaw = $matches_line[2];
                $due = floatval(str_replace(',', '', $matches_line[3]));
                $paid = floatval(str_replace(',', '', $matches_line[4]));
                $balance = floatval(str_replace(',', '', $matches_line[5]));
                
                // המרת סימן מטבע לקוד
                $currencyMap = ['$' => 'USD', '€' => 'EUR', '₪' => 'ILS'];
                $currency = $currencyMap[$currencyRaw] ?? $currencyRaw;
                
                // מחפש שם נוסע
                $passengerName = '';
                if (preg_match('/([A-Z]{2,}\s+[A-Z]{2,}(?:\s+[A-Z]{2,})?)/', $line, $nameMatch)) {
                    $passengerName = $nameMatch[1];
                }
                
                $bookings[] = [
                    'booking_code' => $bookingCode,
                    'passenger_name' => $passengerName,
                    'currency' => $currency,
                    'due' => $due,
                    'paid' => $paid,
                    'balance' => $balance
                ];
            }
        }
        
        if (empty($bookings)) {
            $error = 'לא נמצאו הזמנות בטקסט. וודא שהעתקת את כל התוכן מה-PDF';
        } else {
            // השוואה למסד הנתונים
            foreach ($bookings as $booking) {
                $code = $booking['booking_code'];
                
                // חיפוש בטיסות הלוך
                $stmt = $pdo->prepare("
                    SELECT id, booking_code_outbound, flight_supplier_cost, flight_supplier_paid, 
                           destination, departure_date, passengers, 'outbound' as flight_type
                    FROM orders 
                    WHERE booking_code_outbound = ? AND UPPER(flight_supplier) = ?
                ");
                $stmt->execute([$code, $selectedSupplier]);
                $orderOut = $stmt->fetch(PDO::FETCH_ASSOC);
                
                // חיפוש בטיסות חזור
                $stmt = $pdo->prepare("
                    SELECT id, booking_code_return, return_flight_supplier_cost, return_flight_supplier_paid,
                           destination, departure_date, passengers, 'return' as flight_type
                    FROM orders 
                    WHERE booking_code_return = ? AND (UPPER(return_flight_supplier) = ? OR (return_flight_supplier IS NULL AND UPPER(flight_supplier) = ?))
                ");
                $stmt->execute([$code, $selectedSupplier, $selectedSupplier]);
                $orderRet = $stmt->fetch(PDO::FETCH_ASSOC);
                
                $order = $orderOut ?: $orderRet;
                
                if ($order) {
                    $ourCost = $order['flight_type'] === 'outbound' 
                        ? floatval($order['flight_supplier_cost']) 
                        : floatval($order['return_flight_supplier_cost']);
                    $isPaid = $order['flight_type'] === 'outbound'
                        ? ($order['flight_supplier_paid'] === 'paid')
                        : ($order['return_flight_supplier_paid'] === 'paid');
                    
                    $supplierBalance = $booking['balance'];
                    $diff = abs($ourCost - $supplierBalance);
                    
                    $item = [
                        'booking_code' => $code,
                        'order_id' => $order['id'],
                        'flight_type' => $order['flight_type'],
                        'passenger_name' => $booking['passenger_name'],
                        'destination' => $order['destination'],
                        'flight_date' => $order['departure_date'],
                        'currency' => $booking['currency'] ?? 'USD',
                        'supplier_due' => $booking['due'],
                        'supplier_paid' => $booking['paid'],
                        'supplier_balance' => $supplierBalance,
                        'our_cost' => $ourCost,
                        'difference' => $diff,
                        'is_paid' => $isPaid
                    ];
                    
                    if ($diff < 1) { // הפרש קטן מ-$1 = התאמה
                        $matches[] = $item;
                    } else {
                        $mismatches[] = $item;
                    }
                } else {
                    $notFound[] = $booking;
                }
            }
            
            $supplierData = $bookings;
        }
    }
}

// סימון כשולם
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['mark_paid'])) {
    $bookingCodes = $_POST['booking_codes'] ?? [];
    $supplier = strtoupper($_POST['supplier_name'] ?? '');
    
    if (!empty($bookingCodes) && !empty($supplier)) {
        $updatedCount = 0;
        
        foreach ($bookingCodes as $code) {
            // עדכון טיסת הלוך
            $stmt = $pdo->prepare("
                UPDATE orders 
                SET flight_supplier_paid = 'paid' 
                WHERE booking_code_outbound = ? AND UPPER(flight_supplier) = ?
            ");
            $stmt->execute([$code, $supplier]);
            $updatedCount += $stmt->rowCount();
            
            // עדכון טיסת חזור
            $stmt = $pdo->prepare("
                UPDATE orders 
                SET return_flight_supplier_paid = 'paid' 
                WHERE booking_code_return = ? AND (UPPER(return_flight_supplier) = ? OR (return_flight_supplier IS NULL AND UPPER(flight_supplier) = ?))
            ");
            $stmt->execute([$code, $supplier, $supplier]);
            $updatedCount += $stmt->rowCount();
        }
        
        $success = "עודכנו $updatedCount הזמנות כשולמו לספק";
    }
}
?>
<!DOCTYPE html>
<html lang="he" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>התאמות ספקים</title>
    <style>
        * { box-sizing: border-box; margin: 0; padding: 0; }
        body { font-family: 'Segoe UI', Tahoma, sans-serif; background: #f5f7fa; min-height: 100vh; }
        
        .header {
            background: linear-gradient(135deg, #1e3c72, #2a5298);
            color: white;
            padding: 20px 30px;
        }
        .header h1 { font-size: 1.5rem; }
        .header a { color: white; text-decoration: none; opacity: 0.8; }
        .header a:hover { opacity: 1; }
        
        .container { max-width: 1400px; margin: 0 auto; padding: 20px; }
        
        .alert {
            padding: 15px 20px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        .alert-error { background: #f8d7da; color: #721c24; }
        .alert-success { background: #d4edda; color: #155724; }
        
        .card {
            background: white;
            border-radius: 12px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.08);
            margin-bottom: 20px;
            overflow: hidden;
        }
        .card-header {
            padding: 20px;
            background: #f8f9fa;
            border-bottom: 1px solid #eee;
        }
        .card-header h2 { font-size: 1.2rem; color: #333; }
        .card-body { padding: 20px; }
        
        .form-row {
            display: flex;
            gap: 15px;
            margin-bottom: 15px;
            align-items: flex-end;
        }
        .form-group { flex: 1; }
        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: 500;
            color: #333;
        }
        .form-group select, .form-group textarea {
            width: 100%;
            padding: 10px 12px;
            border: 1px solid #ddd;
            border-radius: 6px;
            font-size: 0.95rem;
        }
        .form-group textarea {
            min-height: 200px;
            font-family: monospace;
            font-size: 0.85rem;
        }
        
        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 0.95rem;
            font-weight: 500;
        }
        .btn-primary { background: #1e3c72; color: white; }
        .btn-success { background: #28a745; color: white; }
        .btn-danger { background: #dc3545; color: white; }
        .btn:hover { opacity: 0.9; }
        
        .stats-row {
            display: grid;
            grid-template-columns: repeat(4, 1fr);
            gap: 15px;
            margin-bottom: 20px;
        }
        .stat-card {
            background: white;
            padding: 20px;
            border-radius: 10px;
            text-align: center;
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        }
        .stat-card .number {
            font-size: 2rem;
            font-weight: bold;
        }
        .stat-card .label {
            color: #666;
            margin-top: 5px;
        }
        .stat-match { border-right: 4px solid #28a745; }
        .stat-match .number { color: #28a745; }
        .stat-mismatch { border-right: 4px solid #ffc107; }
        .stat-mismatch .number { color: #ffc107; }
        .stat-notfound { border-right: 4px solid #dc3545; }
        .stat-notfound .number { color: #dc3545; }
        .stat-total { border-right: 4px solid #1e3c72; }
        .stat-total .number { color: #1e3c72; }
        
        table {
            width: 100%;
            border-collapse: collapse;
            font-size: 0.85rem;
        }
        th, td {
            padding: 12px 10px;
            text-align: right;
            border-bottom: 1px solid #eee;
        }
        th {
            background: #f8f9fa;
            font-weight: 600;
        }
        tr:hover { background: #f8f9fa; }
        
        .match-row { background: #f0fff0; }
        .mismatch-row { background: #fffbf0; }
        .notfound-row { background: #fff5f5; }
        
        .badge {
            padding: 4px 10px;
            border-radius: 4px;
            font-size: 0.75rem;
            font-weight: 500;
        }
        .badge-match { background: #d4edda; color: #155724; }
        .badge-mismatch { background: #fff3cd; color: #856404; }
        .badge-notfound { background: #f8d7da; color: #721c24; }
        .badge-paid { background: #28a745; color: white; }
        .badge-unpaid { background: #dc3545; color: white; }
        
        .section-title {
            display: flex;
            align-items: center;
            gap: 10px;
            margin: 25px 0 15px;
            padding-bottom: 10px;
            border-bottom: 2px solid #eee;
        }
        .section-title h3 { font-size: 1.1rem; }
        .section-count {
            background: #e9ecef;
            padding: 3px 10px;
            border-radius: 12px;
            font-size: 0.85rem;
        }
        
        .diff-positive { color: #28a745; }
        .diff-negative { color: #dc3545; }
        
        .checkbox-cell { width: 40px; text-align: center; }
        .checkbox-cell input { width: 18px; height: 18px; cursor: pointer; }
        
        .actions-bar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 15px 20px;
            background: #e9f7ef;
            border-radius: 8px;
            margin-bottom: 15px;
        }
        
        .instructions {
            background: #e7f3ff;
            padding: 15px 20px;
            border-radius: 8px;
            margin-bottom: 20px;
            border-right: 4px solid #1e3c72;
        }
        .instructions h4 { margin-bottom: 10px; color: #1e3c72; }
        .instructions ol { margin-right: 20px; }
        .instructions li { margin-bottom: 5px; }
    </style>
</head>
<body>
    <div class="header">
        <a href="reports.php?report=suppliers">← חזרה לדוח ספקים</a>
        <h1>🔄 התאמות ספקים (Reconciliation)</h1>
    </div>
    
    <div class="container">
        <?php if ($error): ?>
        <div class="alert alert-error">⚠️ <?= htmlspecialchars($error) ?></div>
        <?php endif; ?>
        
        <?php if ($success): ?>
        <div class="alert alert-success">✅ <?= htmlspecialchars($success) ?></div>
        <?php endif; ?>
        
        <div class="instructions">
            <h4>📋 הוראות שימוש</h4>
            <ol>
                <li>בחר את הספק שממנו קיבלת את הדוח</li>
                <li>פתח את קובץ ה-PDF שקיבלת מהספק</li>
                <li>סמן הכל (Ctrl+A) והעתק (Ctrl+C)</li>
                <li>הדבק (Ctrl+V) בתיבת הטקסט למטה</li>
                <li>לחץ "השווה נתונים" לראות התאמות</li>
                <li>סמן את ההזמנות שרוצים לעדכן כשולמו</li>
            </ol>
        </div>
        
        <div class="card">
            <div class="card-header">
                <h2>📄 העלאת דוח ספק</h2>
            </div>
            <div class="card-body">
                <form method="POST">
                    <div class="form-row">
                        <div class="form-group" style="flex: 0 0 200px;">
                            <label>ספק</label>
                            <select name="supplier" required>
                                <option value="">-- בחר ספק --</option>
                                <?php foreach ($suppliers as $s): ?>
                                <option value="<?= htmlspecialchars($s) ?>" <?= strtoupper($selectedSupplier) === strtoupper($s) ? 'selected' : '' ?>><?= htmlspecialchars(strtoupper($s)) ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                    <div class="form-group">
                        <label>תוכן הדוח (העתק מה-PDF)</label>
                        <textarea name="pdf_text" placeholder="הדבק כאן את התוכן שהעתקת מה-PDF..."><?= htmlspecialchars($_POST['pdf_text'] ?? '') ?></textarea>
                    </div>
                    <button type="submit" class="btn btn-primary">🔍 השווה נתונים</button>
                </form>
            </div>
        </div>
        
        <?php if (!empty($matches) || !empty($mismatches) || !empty($notFound)): ?>
        
        <!-- סטטיסטיקות -->
        <div class="stats-row">
            <div class="stat-card stat-total">
                <div class="number"><?= count($supplierData) ?></div>
                <div class="label">סה"כ בדוח הספק</div>
            </div>
            <div class="stat-card stat-match">
                <div class="number"><?= count($matches) ?></div>
                <div class="label">התאמות מלאות ✓</div>
            </div>
            <div class="stat-card stat-mismatch">
                <div class="number"><?= count($mismatches) ?></div>
                <div class="label">הפרשי סכומים ⚠️</div>
            </div>
            <div class="stat-card stat-notfound">
                <div class="number"><?= count($notFound) ?></div>
                <div class="label">לא נמצאו במערכת ❌</div>
            </div>
        </div>
        
        <form method="POST" id="markPaidForm">
            <input type="hidden" name="mark_paid" value="1">
            <input type="hidden" name="supplier_name" value="<?= htmlspecialchars($selectedSupplier) ?>">
            
            <!-- התאמות מלאות -->
            <?php if (!empty($matches)): ?>
            <div class="card">
                <div class="card-header">
                    <div class="section-title" style="margin: 0; border: none; padding: 0;">
                        <h3>✅ התאמות מלאות</h3>
                        <span class="section-count"><?= count($matches) ?> הזמנות</span>
                    </div>
                </div>
                <div class="card-body">
                    <div class="actions-bar">
                        <div>
                            <label><input type="checkbox" id="selectAllMatches" onchange="toggleAll('matches')"> בחר הכל</label>
                        </div>
                        <button type="submit" class="btn btn-success">✓ סמן נבחרים כשולמו</button>
                    </div>
                    <table>
                        <thead>
                            <tr>
                                <th class="checkbox-cell"></th>
                                <th>מס' הזמנה ספק</th>
                                <th>סוג</th>
                                <th>נוסע</th>
                                <th>יעד</th>
                                <th>סכום ספק</th>
                                <th>סכום שלנו</th>
                                <th>הפרש</th>
                                <th>סטטוס</th>
                                <th>הזמנה</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($matches as $item): ?>
                            <tr class="match-row">
                                <td class="checkbox-cell">
                                    <?php if (!$item['is_paid']): ?>
                                    <input type="checkbox" name="booking_codes[]" value="<?= htmlspecialchars($item['booking_code']) ?>" class="match-checkbox">
                                    <?php endif; ?>
                                </td>
                                <td><strong><?= htmlspecialchars($item['booking_code']) ?></strong></td>
                                <td><?= $item['flight_type'] === 'outbound' ? 'הלוך' : 'חזור' ?></td>
                                <td><?= htmlspecialchars($item['passenger_name']) ?></td>
                                <td><?= htmlspecialchars($item['destination']) ?></td>
                                <td><?= ($item['currency'] === 'EUR' ? '€' : ($item['currency'] === 'ILS' ? '₪' : '$')) ?><?= number_format($item['supplier_balance'], 2) ?></td>
                                <td><?= ($item['currency'] === 'EUR' ? '€' : ($item['currency'] === 'ILS' ? '₪' : '$')) ?><?= number_format($item['our_cost'], 2) ?></td>
                                <td class="diff-positive">$<?= number_format($item['difference'], 2) ?></td>
                                <td>
                                    <?php if ($item['is_paid']): ?>
                                    <span class="badge badge-paid">שולם</span>
                                    <?php else: ?>
                                    <span class="badge badge-unpaid">ממתין</span>
                                    <?php endif; ?>
                                </td>
                                <td><a href="view_order_new.php?id=<?= $item['order_id'] ?>">#<?= $item['order_id'] ?></a></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            <?php endif; ?>
            
            <!-- הפרשי סכומים -->
            <?php if (!empty($mismatches)): ?>
            <div class="card">
                <div class="card-header">
                    <div class="section-title" style="margin: 0; border: none; padding: 0;">
                        <h3>⚠️ הפרשי סכומים</h3>
                        <span class="section-count"><?= count($mismatches) ?> הזמנות</span>
                    </div>
                </div>
                <div class="card-body">
                    <p style="color: #856404; margin-bottom: 15px;">
                        ⚠️ הזמנות אלו נמצאו במערכת אך יש הפרש בין הסכום שלנו לסכום הספק. יש לבדוק ידנית.
                    </p>
                    <table>
                        <thead>
                            <tr>
                                <th class="checkbox-cell"></th>
                                <th>מס' הזמנה ספק</th>
                                <th>סוג</th>
                                <th>נוסע</th>
                                <th>יעד</th>
                                <th>סכום ספק</th>
                                <th>סכום שלנו</th>
                                <th>הפרש</th>
                                <th>סטטוס</th>
                                <th>הזמנה</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($mismatches as $item): ?>
                            <tr class="mismatch-row">
                                <td class="checkbox-cell">
                                    <?php if (!$item['is_paid']): ?>
                                    <input type="checkbox" name="booking_codes[]" value="<?= htmlspecialchars($item['booking_code']) ?>" class="mismatch-checkbox">
                                    <?php endif; ?>
                                </td>
                                <td><strong><?= htmlspecialchars($item['booking_code']) ?></strong></td>
                                <td><?= $item['flight_type'] === 'outbound' ? 'הלוך' : 'חזור' ?></td>
                                <td><?= htmlspecialchars($item['passenger_name']) ?></td>
                                <td><?= htmlspecialchars($item['destination']) ?></td>
                                <td><?= ($item['currency'] === 'EUR' ? '€' : ($item['currency'] === 'ILS' ? '₪' : '$')) ?><?= number_format($item['supplier_balance'], 2) ?></td>
                                <td><?= ($item['currency'] === 'EUR' ? '€' : ($item['currency'] === 'ILS' ? '₪' : '$')) ?><?= number_format($item['our_cost'], 2) ?></td>
                                <td class="diff-negative"><strong>$<?= number_format($item['difference'], 2) ?></strong></td>
                                <td>
                                    <?php if ($item['is_paid']): ?>
                                    <span class="badge badge-paid">שולם</span>
                                    <?php else: ?>
                                    <span class="badge badge-unpaid">ממתין</span>
                                    <?php endif; ?>
                                </td>
                                <td><a href="view_order_new.php?id=<?= $item['order_id'] ?>">#<?= $item['order_id'] ?></a></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            <?php endif; ?>
        </form>
        
        <!-- לא נמצאו -->
        <?php if (!empty($notFound)): ?>
        <div class="card">
            <div class="card-header">
                <div class="section-title" style="margin: 0; border: none; padding: 0;">
                    <h3>❌ לא נמצאו במערכת</h3>
                    <span class="section-count"><?= count($notFound) ?> הזמנות</span>
                </div>
            </div>
            <div class="card-body">
                <p style="color: #721c24; margin-bottom: 15px;">
                    ❌ הזמנות אלו מופיעות בדוח הספק אך לא נמצאו במערכת שלנו. יתכן שהוזנו עם מספר הזמנה שונה או שחסרות.
                </p>
                <table>
                    <thead>
                        <tr>
                            <th>מס' הזמנה ספק</th>
                            <th>נוסע</th>
                            <th>סכום לחיוב</th>
                            <th>שולם</th>
                            <th>יתרה</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($notFound as $item): ?>
                        <tr class="notfound-row">
                            <td><strong><?= htmlspecialchars($item['booking_code']) ?></strong></td>
                            <td><?= htmlspecialchars($item['passenger_name']) ?></td>
                            <td>$<?= number_format($item['due'], 2) ?></td>
                            <td>$<?= number_format($item['paid'], 2) ?></td>
                            <td><strong>$<?= number_format($item['balance'], 2) ?></strong></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php endif; ?>
        
        <?php endif; ?>
    </div>
    
    <script>
    function toggleAll(type) {
        const checkboxes = document.querySelectorAll('.' + type + '-checkbox');
        const selectAll = document.getElementById('selectAll' + type.charAt(0).toUpperCase() + type.slice(1));
        checkboxes.forEach(cb => cb.checked = selectAll.checked);
    }
    </script>
</body>
</html>