<?php
// לוג מיידי - לפני כל דבר אחר
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

ob_start(); // תפיסת כל output

$logDir = __DIR__ . '/../logs';
if (!is_dir($logDir)) @mkdir($logDir, 0755, true);
$logFile = $logDir . '/update_order_' . date('Y-m-d') . '.log';
file_put_contents($logFile, "\n" . date('Y-m-d H:i:s') . " - SCRIPT STARTED\n", FILE_APPEND);

// Custom error handler
set_error_handler(function($errno, $errstr, $errfile, $errline) use ($logFile) {
    file_put_contents($logFile, "PHP Error [$errno]: $errstr in $errfile:$errline\n", FILE_APPEND);
    return false;
});

register_shutdown_function(function() use ($logFile) {
    $error = error_get_last();
    if ($error && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
        file_put_contents($logFile, "FATAL: " . print_r($error, true) . "\n", FILE_APPEND);
        ob_end_clean();
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode(['success' => false, 'error' => 'שגיאת שרת: ' . $error['message']]);
    }
});

// עדכון הזמנה קיימת - גרסה חכמה שמתאימה לעמודות קיימות
require_once '../includes/auth.php';
requireLogin();

header('Content-Type: application/json; charset=utf-8');

file_put_contents($logFile, date('Y-m-d H:i:s') . " - After auth\n", FILE_APPEND);

$agent = getCurrentAgent();
$pdo = getDB();
$pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

try {
    $orderId = $_POST['order_id'] ?? null;
    
    if (!$orderId) {
        throw new Exception('חסר מזהה הזמנה');
    }
    
    // וידוא שההזמנה קיימת
    $stmt = $pdo->prepare("SELECT * FROM orders WHERE id = ?");
    $stmt->execute([$orderId]);
    $existingOrder = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$existingOrder) {
        throw new Exception('הזמנה לא נמצאה');
    }
    
    // בדיקה אילו עמודות קיימות בטבלה
    $tableColumns = [];
    $columnsStmt = $pdo->query("SHOW COLUMNS FROM orders");
    while ($col = $columnsStmt->fetch()) {
        $tableColumns[] = $col['Field'];
    }
    file_put_contents($logFile, "Existing columns: " . implode(', ', $tableColumns) . "\n", FILE_APPEND);
    
    // פונקציית סניטציה
    if (!function_exists('sanitize_input')) {
        function sanitize_input($str) {
            return htmlspecialchars(trim($str ?? ''), ENT_QUOTES, 'UTF-8');
        }
    }
    
    // מיפוי כל השדות האפשריים
    $allFields = [
        'order_date' => $_POST['order_date'] ?: date('Y-m-d'),
        'customer_phone' => sanitize_input($_POST['customer_phone'] ?? ''),
        'customer_email' => sanitize_input($_POST['customer_email'] ?? ''),
        'passengers' => $_POST['passengers'] ?? '[]',
        'destination' => sanitize_input($_POST['destination'] ?? ''),
        'departure_date' => $_POST['departure_date'] ?: null,
        'return_date' => $_POST['return_date'] ?: null,
        'airline_outbound' => sanitize_input($_POST['airline_outbound'] ?? ''),
        'booking_code_outbound' => sanitize_input($_POST['booking_code_outbound'] ?? ''),
        'airline_return' => sanitize_input($_POST['airline_return'] ?? ''),
        'booking_code_return' => sanitize_input($_POST['booking_code_return'] ?? ''),
        'flight_supplier' => sanitize_input($_POST['flight_supplier'] ?? ''),
        'flight_supplier_cost' => (float)($_POST['flight_supplier_cost'] ?? 0),
        'flight_supplier_currency' => $_POST['flight_supplier_currency'] ?? 'ILS',
        'flight_supplier_paid' => $_POST['flight_supplier_paid'] ?? 'unpaid',
        'return_flight_supplier' => sanitize_input($_POST['return_flight_supplier'] ?? $_POST['flight_supplier'] ?? ''),
        'return_flight_supplier_cost' => (float)($_POST['return_flight_supplier_cost'] ?? 0),
        'return_flight_supplier_currency' => $_POST['return_flight_supplier_currency'] ?? 'ILS',
        'return_flight_supplier_paid' => $_POST['return_flight_supplier_paid'] ?? 'unpaid',
        'hotel_supplier' => sanitize_input($_POST['hotel_supplier'] ?? ''),
        'hotel_supplier_booking' => sanitize_input($_POST['hotel_booking_code'] ?? ''),
        'hotel_supplier_cost' => (float)($_POST['hotel_supplier_cost'] ?? 0),
        'hotel_supplier_currency' => $_POST['hotel_supplier_currency'] ?? 'ILS',
        'hotel_supplier_paid' => $_POST['hotel_supplier_paid'] ?? 'unpaid',
        'luggage_details' => sanitize_input($_POST['luggage_details'] ?? ''),
        'total_price' => (float)($_POST['total_price'] ?? 0),
        'profit' => (float)($_POST['profit'] ?? 0),
        'payments' => $_POST['payments'] ?? '[]',
        'extras' => $_POST['extras'] ?? '[]',
        'notes' => sanitize_input($_POST['notes'] ?? '')
    ];
    
    // סינון רק לשדות שקיימים בטבלה
    $updateFields = [];
    $params = [];
    foreach ($allFields as $field => $value) {
        if (in_array($field, $tableColumns)) {
            $updateFields[] = "$field = :$field";
            $params[$field] = $value;
        }
    }
    
    if (empty($updateFields)) {
        throw new Exception('אין שדות לעדכון');
    }
    
    $params['order_id'] = $orderId;
    
    $sql = "UPDATE orders SET " . implode(', ', $updateFields) . ", updated_at = NOW() WHERE id = :order_id";
    
    file_put_contents($logFile, "SQL: $sql\n", FILE_APPEND);
    file_put_contents($logFile, "Params: " . print_r($params, true) . "\n", FILE_APPEND);
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    
    file_put_contents($logFile, date('Y-m-d H:i:s') . " - SUCCESS order_id: $orderId\n", FILE_APPEND);
    
    ob_end_clean(); // ניקוי buffer
    echo json_encode([
        'success' => true,
        'message' => 'ההזמנה עודכנה בהצלחה',
        'order_id' => $orderId
    ]);
    
} catch (Exception $e) {
    file_put_contents($logFile, date('Y-m-d H:i:s') . " - ERROR: " . $e->getMessage() . "\n", FILE_APPEND);
    file_put_contents($logFile, "Trace: " . $e->getTraceAsString() . "\n", FILE_APPEND);
    ob_end_clean(); // ניקוי buffer
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
