<?php
// צפייה בהזמנה
require_once 'includes/auth.php';
requireLogin();

$agent = getCurrentAgent();
$order_id = $_GET['id'] ?? null;

if (!$order_id) {
    header('Location: orders.php');
    exit;
}

$pdo = getDB();
$stmt = $pdo->prepare("SELECT o.*, a.full_name as agent_name FROM orders o JOIN agents a ON o.agent_id = a.id WHERE o.id = ?");
$stmt->execute([$order_id]);
$order = $stmt->fetch();

if (!$order) {
    header('Location: orders.php');
    exit;
}

$passengers = json_decode($order['passengers'] ?? '[]', true);
$payments = json_decode($order['payments'] ?? '[]', true);

// זיהוי ספק הטיסה לפי סוג ההזמנה
$flightSupplier = '';
$flightSupplierBooking = '';
$flightSupplierCurrency = 'USD';

$orderType = $order['order_type'] ?? 'separate';
if ($orderType === 'separate') {
    // טיסה + מלון בנפרד - ספק טיסה
    $flightSupplier = $order['flight_supplier'] ?? '';
    $flightSupplierBooking = $order['flight_supplier_booking'] ?? '';
    $flightSupplierCurrency = $order['flight_supplier_currency'] ?? 'USD';
} else {
    // חבילה או מאורגן - ספק החבילה
    $flightSupplier = $order['package_supplier'] ?? '';
    $flightSupplierBooking = $order['package_supplier_booking'] ?? '';
    $flightSupplierCurrency = $order['package_supplier_currency'] ?? 'USD';
}

// שליפת תוספות להזמנה זו
$stmt = $pdo->prepare("SELECT * FROM drafts WHERE parent_order_id = ? ORDER BY created_at DESC");
$stmt->execute([$order_id]);
$addons = $stmt->fetchAll();

// שליפת חובות לספקים
$supplierDebts = [];
try {
    $stmt = $pdo->prepare("SELECT * FROM supplier_debts WHERE order_id = ? ORDER BY created_at ASC");
    $stmt->execute([$order_id]);
    $supplierDebts = $stmt->fetchAll();
} catch (Exception $e) {
    // טבלה לא קיימת עדיין
}

// חישוב סיכום חובות
$totalDebt = 0;
$totalUnpaidDebt = 0;
$debtByCurrency = [];
foreach ($supplierDebts as $debt) {
    $currency = $debt['currency'] ?? 'USD';
    if (!isset($debtByCurrency[$currency])) {
        $debtByCurrency[$currency] = ['total' => 0, 'unpaid' => 0];
    }
    $debtByCurrency[$currency]['total'] += floatval($debt['amount']);
    if ($debt['status'] === 'unpaid') {
        $debtByCurrency[$currency]['unpaid'] += floatval($debt['amount']);
    }
}

// חישוב סיכום כולל
$totalPaidAddons = 0;
$totalProfitAddons = 0;
foreach ($addons as $addon) {
    if ($addon['status'] === 'paid') {
        $totalPaidAddons += floatval($addon['amount_paid'] ?? $addon['total_price']);
    }
}

$statusNames = [
    'draft' => 'טיוטה',
    'sent' => 'נשלח',
    'paid' => 'שולם',
    'confirmed' => 'אושר',
    'cancelled' => 'בוטל'
];
?>
<!DOCTYPE html>
<html lang="he" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>הזמנה #<?= $order['id'] ?> - <?= SITE_NAME ?></title>
    <link rel="stylesheet" href="css/style.css">
    <style>
        .order-details {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
        }
        .detail-card {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 10px;
        }
        .detail-card h3 {
            color: #e94560;
            margin-bottom: 15px;
            font-size: 1.1rem;
        }
        .detail-row {
            display: flex;
            justify-content: space-between;
            padding: 8px 0;
            border-bottom: 1px solid #eee;
        }
        .detail-row:last-child { border-bottom: none; }
        .detail-label { color: #666; }
        .detail-value { font-weight: 600; }
        .bank-warning {
            background: #fff3cd;
            border: 2px solid #ffc107;
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        .status-badge {
            display: inline-block;
            padding: 8px 20px;
            border-radius: 20px;
            color: white;
            font-weight: 600;
        }
        .status-confirmed { background: #007bff; }
        .status-paid { background: #28a745; }
        .status-sent { background: #17a2b8; }
        .status-draft { background: #6c757d; }
        .status-cancelled { background: #dc3545; }
        .back-link {
            display: inline-block;
            margin-bottom: 20px;
            color: #666;
            text-decoration: none;
        }
        .back-link:hover { color: #333; }
        
        /* כפתור הוספת תוספת */
        .btn-addon {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 600;
            margin-left: 10px;
        }
        .btn-addon:hover {
            transform: scale(1.02);
        }
        
        /* מודאל */
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.5);
            z-index: 1000;
            justify-content: center;
            align-items: center;
        }
        .modal.active { display: flex; }
        .modal-content {
            background: white;
            padding: 30px;
            border-radius: 15px;
            max-width: 500px;
            width: 90%;
            max-height: 90vh;
            overflow-y: auto;
        }
        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }
        .modal-header h2 { margin: 0; }
        .modal-close {
            background: none;
            border: none;
            font-size: 24px;
            cursor: pointer;
            color: #666;
        }
        .form-group {
            margin-bottom: 15px;
        }
        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: 600;
        }
        .form-group input, .form-group select, .form-group textarea {
            width: 100%;
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 8px;
            font-size: 14px;
        }
        .btn-submit {
            background: #28a745;
            color: white;
            border: none;
            padding: 12px 30px;
            border-radius: 8px;
            cursor: pointer;
            font-size: 16px;
            font-weight: 600;
            width: 100%;
        }
        .btn-submit:hover { background: #218838; }
        
        /* תוספות */
        .addons-section {
            background: #e8f5e9;
            border: 2px solid #4caf50;
            padding: 20px;
            border-radius: 10px;
            margin-top: 20px;
        }
        .addon-item {
            background: white;
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 10px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .addon-status {
            padding: 4px 12px;
            border-radius: 15px;
            font-size: 12px;
            font-weight: 600;
        }
        .addon-status.paid { background: #d4edda; color: #155724; }
        .addon-status.pending { background: #fff3cd; color: #856404; }
        
        /* סיכום כולל */
        .total-summary {
            background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
            color: white;
            padding: 20px;
            border-radius: 10px;
            margin-top: 20px;
        }
        .total-row {
            display: flex;
            justify-content: space-between;
            padding: 8px 0;
            border-bottom: 1px solid rgba(255,255,255,0.1);
        }
        .total-row:last-child { border-bottom: none; }
        .total-row.grand-total {
            font-size: 1.3rem;
            font-weight: bold;
            padding-top: 15px;
            margin-top: 10px;
            border-top: 2px solid rgba(255,255,255,0.3);
        }
    </style>
</head>
<body>
    <header class="header">
        <div class="logo">✈️ <span>חופשה</span> במבצע</div>
        <div class="user-info">
            <span class="user-name">שלום, <?= sanitize($agent['full_name']) ?></span>
            <a href="orders_new.php" style="margin-left:15px; color:#666;">רשימת הזמנות</a>
            <a href="logout.php" class="btn-logout">התנתק</a>
        </div>
    </header>

    <div class="container" style="max-width:1000px;">
        <a href="orders_new.php" class="back-link">← חזרה לרשימת הזמנות</a>
        
        <div class="card">
            <div style="display:flex; justify-content:space-between; align-items:center; margin-bottom:20px;">
                <h1 class="card-title" style="margin:0;">📋 הזמנה #<?= $order['id'] ?></h1>
                <div>
                    <button onclick="openAddonModal()" class="btn-addon">➕ הוסף פריט ללקוח</button>
                    <a href="voucher.php?token=<?= $order['draft_token'] ?>" target="_blank" class="action-btn action-btn-view" style="margin-left:10px;">📄 וואצ'ר</a>
                    <span class="status-badge status-<?= $order['status'] ?>">
                        <?= $statusNames[$order['status']] ?? $order['status'] ?>
                    </span>
                </div>
            </div>

            <?php 
            // בדיקת תשלום בהעברה בנקאית
            $hasBankTransfer = false;
            if (is_array($payments)) {
                foreach ($payments as $p) {
                    if (isset($p['method']) && $p['method'] === 'העברה בנקאית') {
                        $hasBankTransfer = true;
                        break;
                    }
                }
            }
            if ($hasBankTransfer): 
            ?>
            <div class="bank-warning">
                ⚠️ <strong>שים לב:</strong> בהזמנה זו יש תשלום בהעברה בנקאית - יש לוודא קבלת הכסף בחשבון הבנק!
            </div>
            <?php endif; ?>

            <div class="order-details">
                <!-- פרטי עסקה -->
                <div class="detail-card">
                    <h3>📅 פרטי עסקה</h3>
                    <div class="detail-row">
                        <span class="detail-label">תאריך הזמנה</span>
                        <span class="detail-value"><?= $order['order_date'] ? date('d/m/Y', strtotime($order['order_date'])) : date('d/m/Y', strtotime($order['created_at'])) ?></span>
                    </div>
                    <div class="detail-row">
                        <span class="detail-label">נציג</span>
                        <span class="detail-value"><?= sanitize($order['agent_name']) ?></span>
                    </div>
                    <?php if ($order['deal_builder']): ?>
                    <div class="detail-row">
                        <span class="detail-label">בונה הדיל</span>
                        <span class="detail-value"><?= sanitize($order['deal_builder']) ?></span>
                    </div>
                    <?php endif; ?>
                </div>

                <!-- פרטי לקוח -->
                <div class="detail-card">
                    <h3>👤 פרטי לקוח</h3>
                    <div class="detail-row">
                        <span class="detail-label">טלפון</span>
                        <span class="detail-value"><?= sanitize($order['customer_phone']) ?></span>
                    </div>
                    <?php if ($order['customer_email']): ?>
                    <div class="detail-row">
                        <span class="detail-label">אימייל</span>
                        <span class="detail-value"><?= sanitize($order['customer_email']) ?></span>
                    </div>
                    <?php endif; ?>
                    <?php if (!empty($passengers)): ?>
                    <div style="margin-top:10px;">
                        <strong>נוסעים:</strong>
                        <ul style="margin:5px 0 0 20px;">
                            <?php foreach ($passengers as $p): ?>
                                <li><?= sanitize($p) ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                    <?php endif; ?>
                </div>

                <!-- פרטי טיסה -->
                <div class="detail-card">
                    <h3>✈️ פרטי טיסה</h3>
                    <div class="detail-row">
                        <span class="detail-label">יעד</span>
                        <span class="detail-value"><?= sanitize($order['destination']) ?></span>
                    </div>
                    <div class="detail-row">
                        <span class="detail-label">תאריכים</span>
                        <span class="detail-value"><?= date('d/m/Y', strtotime($order['departure_date'])) ?> - <?= date('d/m/Y', strtotime($order['return_date'])) ?></span>
                    </div>
                    <?php if ($order['airline_outbound']): ?>
                    <div class="detail-row">
                        <span class="detail-label">חברת תעופה</span>
                        <span class="detail-value"><?= sanitize($order['airline_outbound']) ?></span>
                    </div>
                    <?php endif; ?>
                    <?php if ($order['booking_code_outbound']): ?>
                    <div class="detail-row">
                        <span class="detail-label">קוד הזמנה</span>
                        <span class="detail-value"><?= sanitize($order['booking_code_outbound']) ?></span>
                    </div>
                    <?php endif; ?>
                    <?php if ($order['airline_return']): ?>
                    <div class="detail-row">
                        <span class="detail-label">חברה חזור</span>
                        <span class="detail-value"><?= sanitize($order['airline_return']) ?> (<?= sanitize($order['booking_code_return']) ?>)</span>
                    </div>
                    <?php endif; ?>
                </div>

                <!-- פרטי מלון -->
                <div class="detail-card">
                    <h3>🏨 פרטי מלון</h3>
                    <?php if ($order['hotel_name']): ?>
                    <div class="detail-row">
                        <span class="detail-label">מלון</span>
                        <span class="detail-value"><?= sanitize($order['hotel_name']) ?></span>
                    </div>
                    <?php endif; ?>
                    <?php if ($order['hotel_checkin']): ?>
                    <div class="detail-row">
                        <span class="detail-label">צ'ק אין</span>
                        <span class="detail-value"><?= date('d/m/Y', strtotime($order['hotel_checkin'])) ?></span>
                    </div>
                    <?php endif; ?>
                    <?php if ($order['hotel_checkout']): ?>
                    <div class="detail-row">
                        <span class="detail-label">צ'ק אאוט</span>
                        <span class="detail-value"><?= date('d/m/Y', strtotime($order['hotel_checkout'])) ?></span>
                    </div>
                    <?php endif; ?>
                    <?php if ($order['room_type']): ?>
                    <div class="detail-row">
                        <span class="detail-label">חדר</span>
                        <span class="detail-value"><?= sanitize($order['room_type']) ?></span>
                    </div>
                    <?php endif; ?>
                    <?php if ($order['meal_basis']): ?>
                    <div class="detail-row">
                        <span class="detail-label">בסיס אירוח</span>
                        <span class="detail-value"><?= sanitize($order['meal_basis']) ?></span>
                    </div>
                    <?php endif; ?>
                    <?php if ($order['hotel_confirmation']): ?>
                    <div class="detail-row">
                        <span class="detail-label">מספר אישור</span>
                        <span class="detail-value"><?= sanitize($order['hotel_confirmation']) ?></span>
                    </div>
                    <?php endif; ?>
                </div>

                <!-- כספים -->
                <div class="detail-card">
                    <h3>💰 כספים</h3>
                    <div class="detail-row">
                        <span class="detail-label">מחיר ללקוח</span>
                        <span class="detail-value" style="font-size:1.2rem; color:#28a745;">₪<?= number_format($order['total_price']) ?></span>
                    </div>
                    <?php if ($order['profit']): ?>
                    <div class="detail-row">
                        <span class="detail-label">רווח</span>
                        <span class="detail-value" style="color:#e94560;">₪<?= number_format($order['profit']) ?></span>
                    </div>
                    <?php endif; ?>
                    
                    <?php if (!empty($payments)): ?>
                    <div style="margin-top:15px;">
                        <strong>פירוט תשלומים:</strong>
                        <?php foreach ($payments as $p): ?>
                            <div style="background:<?= ($p['method'] ?? '') === 'העברה בנקאית' ? '#fff3cd' : '#fff' ?>; padding:8px; margin:5px 0; border-radius:5px; border:1px solid #eee;">
                                <?= sanitize($p['name'] ?? 'לא צוין') ?> - 
                                ₪<?= number_format($p['amount'] ?? 0) ?> 
                                (<?= sanitize($p['method'] ?? 'אשראי') ?>)
                                <?php if (($p['method'] ?? '') === 'העברה בנקאית'): ?>
                                    <strong>⚠️</strong>
                                <?php endif; ?>
                            </div>
                        <?php endforeach; ?>
                    </div>
                    <?php endif; ?>
                </div>

                <!-- הערות -->
                <?php if ($order['notes'] || $order['luggage_details']): ?>
                <div class="detail-card">
                    <h3>📝 הערות</h3>
                    <?php if ($order['luggage_details']): ?>
                    <div style="margin-bottom:10px;">
                        <strong>כבודה:</strong><br>
                        <?php
                        $luggageData = json_decode(html_entity_decode($order['luggage_details']), true);
                        if (is_array($luggageData)) {
                            foreach ($luggageData as $passenger) {
                                $name = $passenger['name'] ?? 'נוסע';
                                
                                // בדיקה אם זה פורמט חדש (הלוך/חזור נפרד)
                                if (isset($passenger['same_both_ways']) && !$passenger['same_both_ways']) {
                                    // פורמט חדש - נפרד
                                    $outbound = $passenger['outbound'] ?? [];
                                    $return = $passenger['return'] ?? [];
                                    
                                    echo "<div style='margin-bottom:8px;'><strong>👤 {$name}</strong></div>";
                                    
                                    // הלוך
                                    $outParts = [];
                                    if (!empty($outbound['trolley'])) $outParts[] = "Trolley {$outbound['trolley']}kg";
                                    if (!empty($outbound['suitcase'])) $outParts[] = "Suitcase {$outbound['suitcase']}kg";
                                    if (!empty($outbound['backpack'])) $outParts[] = "Backpack";
                                    $outText = !empty($outParts) ? implode(', ', $outParts) : 'ללא כבודה';
                                    echo "<div style='padding-right:15px;'>✈️ הלוך: {$outText}</div>";
                                    
                                    // חזור
                                    $retParts = [];
                                    if (!empty($return['trolley'])) $retParts[] = "Trolley {$return['trolley']}kg";
                                    if (!empty($return['suitcase'])) $retParts[] = "Suitcase {$return['suitcase']}kg";
                                    if (!empty($return['backpack'])) $retParts[] = "Backpack";
                                    $retText = !empty($retParts) ? implode(', ', $retParts) : 'ללא כבודה';
                                    echo "<div style='padding-right:15px;'>✈️ חזור: {$retText}</div>";
                                    
                                } else {
                                    // פורמט ישן או משותף
                                    $trolley = $passenger['trolley'] ?? 0;
                                    $suitcase = $passenger['suitcase'] ?? 0;
                                    $backpack = !empty($passenger['backpack']) ? ' + Backpack' : '';
                                    echo "<div>🧳 <strong>{$name}</strong>: Trolley {$trolley}kg, Suitcase {$suitcase}kg{$backpack}</div>";
                                }
                            }
                        } else {
                            echo nl2br(sanitize($order['luggage_details']));
                        }
                        ?>
                    </div>
                    <?php endif; ?>
                    <?php if ($order['notes']): ?>
                    <div>
                        <strong>הערות:</strong><br>
                        <?= nl2br(sanitize($order['notes'])) ?>
                    </div>
                    <?php endif; ?>
                </div>
                <?php endif; ?>
            </div>
            
            <!-- תוספות להזמנה -->
            <?php if (!empty($addons)): ?>
            <div class="addons-section">
                <h3>📦 תוספות להזמנה</h3>
                <?php foreach ($addons as $addon): ?>
                <div class="addon-item">
                    <div>
                        <strong><?= sanitize($addon['addon_description'] ?: $addon['destination']) ?></strong>
                        <br>
                        <small style="color:#666;">
                            <?= date('d/m/Y H:i', strtotime($addon['created_at'])) ?>
                            | ₪<?= number_format($addon['total_price']) ?>
                        </small>
                    </div>
                    <div>
                        <?php if ($addon['status'] === 'paid'): ?>
                            <span class="addon-status paid">✅ שולם</span>
                        <?php else: ?>
                            <span class="addon-status pending">⏳ ממתין</span>
                            <a href="pay.php?token=<?= $addon['token'] ?>" target="_blank" style="margin-right:10px;">🔗 לינק</a>
                        <?php endif; ?>
                        <?php if (!$addon['order_id']): ?>
                            <button onclick="deleteDraft(<?= $addon['id'] ?>, '<?= htmlspecialchars($addon['addon_description'] ?: 'תוספת') ?>')" 
                                    style="background:#dc3545; color:white; border:none; padding:5px 10px; border-radius:5px; cursor:pointer; margin-right:5px;" 
                                    title="מחק תוספת">🗑️</button>
                        <?php endif; ?>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
            <?php endif; ?>
            
            <!-- סיכום כולל -->
            <?php if (!empty($addons)): ?>
            <div class="total-summary">
                <h3 style="margin-top:0;">📊 סיכום כולל</h3>
                <div class="total-row">
                    <span>הזמנה מקורית:</span>
                    <span>₪<?= number_format($order['total_price']) ?></span>
                </div>
                <?php foreach ($addons as $addon): ?>
                    <?php if ($addon['status'] === 'paid'): ?>
                    <div class="total-row">
                        <span><?= sanitize($addon['addon_description'] ?: 'תוספת') ?>:</span>
                        <span>₪<?= number_format($addon['amount_paid'] ?? $addon['total_price']) ?></span>
                    </div>
                    <?php endif; ?>
                <?php endforeach; ?>
                <div class="total-row grand-total">
                    <span>סה"כ שולם:</span>
                    <span>₪<?= number_format($order['total_price'] + $totalPaidAddons) ?></span>
                </div>
            </div>
            <?php endif; ?>
            
            <!-- חובות לספקים -->
            <?php if (!empty($supplierDebts)): ?>
            <div style="background: linear-gradient(135deg, #ff6b6b 0%, #ee5a5a 100%); color: white; padding: 20px; border-radius: 10px; margin-top: 20px;">
                <h3 style="margin:0 0 15px 0;">💳 פירוט חוב לספק: <?= htmlspecialchars($flightSupplier) ?></h3>
                <?php if (!empty($flightSupplierBooking)): ?>
                <p style="margin:0 0 15px 0; opacity:0.9;">מספר הזמנה: <?= htmlspecialchars($flightSupplierBooking) ?></p>
                <?php endif; ?>
                
                <table style="width:100%; border-collapse:collapse;">
                    <thead>
                        <tr style="border-bottom:2px solid rgba(255,255,255,0.3);">
                            <th style="text-align:right; padding:10px 5px;">תאריך</th>
                            <th style="text-align:right; padding:10px 5px;">פירוט</th>
                            <th style="text-align:right; padding:10px 5px;">סכום</th>
                            <th style="text-align:center; padding:10px 5px;">סטטוס</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($supplierDebts as $debt): ?>
                        <tr style="border-bottom:1px solid rgba(255,255,255,0.1);">
                            <td style="padding:10px 5px;"><?= date('d/m/y', strtotime($debt['created_at'])) ?></td>
                            <td style="padding:10px 5px;"><?= htmlspecialchars($debt['description']) ?></td>
                            <td style="padding:10px 5px;">
                                <?php 
                                $symbol = ['USD' => '$', 'EUR' => '€', 'ILS' => '₪'][$debt['currency']] ?? $debt['currency'];
                                echo $symbol . number_format($debt['amount'], 2);
                                ?>
                            </td>
                            <td style="text-align:center; padding:10px 5px;">
                                <?php if ($debt['status'] === 'paid'): ?>
                                    <span style="background:rgba(255,255,255,0.2); padding:3px 10px; border-radius:10px;">✅ שולם</span>
                                <?php else: ?>
                                    <span style="background:rgba(255,255,255,0.3); padding:3px 10px; border-radius:10px;">⏳ פתוח</span>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
                
                <!-- סיכום חוב -->
                <div style="margin-top:15px; padding-top:15px; border-top:2px solid rgba(255,255,255,0.3);">
                    <?php foreach ($debtByCurrency as $currency => $amounts): ?>
                    <?php $symbol = ['USD' => '$', 'EUR' => '€', 'ILS' => '₪'][$currency] ?? $currency; ?>
                    <div style="display:flex; justify-content:space-between; font-size:1.1rem; margin-bottom:5px;">
                        <span>סה"כ <?= $currency ?>:</span>
                        <span><?= $symbol ?><?= number_format($amounts['total'], 2) ?></span>
                    </div>
                    <?php if ($amounts['unpaid'] > 0): ?>
                    <div style="display:flex; justify-content:space-between; font-size:1.3rem; font-weight:bold;">
                        <span>🔴 חוב פתוח:</span>
                        <span><?= $symbol ?><?= number_format($amounts['unpaid'], 2) ?></span>
                    </div>
                    <?php endif; ?>
                    <?php endforeach; ?>
                </div>
            </div>
            <?php endif; ?>
        </div>
    </div>
    
    <!-- מודאל הוספת תוספת רגילה -->
    <div id="addonModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2>➕ הוספת פריט ללקוח</h2>
                <button class="modal-close" onclick="closeAddonModal()">&times;</button>
            </div>
            <form id="addonForm" onsubmit="submitAddon(event)">
                <input type="hidden" name="parent_order_id" value="<?= $order['id'] ?>">
                <input type="hidden" name="customer_name" value="<?= htmlspecialchars($passengers[0] ?? $order['customer_phone']) ?>">
                <input type="hidden" name="customer_phone" value="<?= htmlspecialchars($order['customer_phone']) ?>">
                
                <div class="form-group">
                    <label>תיאור התוספת *</label>
                    <select name="addon_type" id="addon_type" onchange="handleAddonTypeChange()">
                        <option value="">-- בחר סוג --</option>
                        <option value="כבודה נוספת">🧳 כבודה נוספת</option>
                        <option value="שינוי שם">שינוי שם</option>
                        <option value="שדרוג מקום">שדרוג מקום</option>
                        <option value="ביטוח נסיעות">ביטוח נסיעות</option>
                        <option value="העברות">העברות משדה התעופה</option>
                        <option value="סיור">סיור מאורגן</option>
                        <option value="אחר">אחר</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label>פרטים נוספים</label>
                    <textarea name="addon_description" id="addon_description" rows="3" placeholder="פרט את התוספת..."></textarea>
                </div>
                
                <div class="form-group">
                    <label>מחיר ללקוח (₪) *</label>
                    <input type="number" name="total_price" id="addon_price" required min="1" step="1" placeholder="0">
                </div>
                
                <button type="submit" class="btn-submit">📤 צור טיוטה ושלח ללקוח</button>
            </form>
        </div>
    </div>
    
    <!-- מודאל כבודה נוספת -->
    <div id="luggageModal" class="modal">
        <div class="modal-content" style="max-width:700px;">
            <div class="modal-header">
                <h2>🧳 הוספת כבודה נוספת</h2>
                <button class="modal-close" onclick="closeLuggageModal()">&times;</button>
            </div>
            
            <div style="background:#e8f5e9; padding:15px; border-radius:8px; margin-bottom:20px;">
                <strong>📦 כבודה קיימת בהזמנה:</strong>
            </div>
            
            <!-- בחירת כיוון -->
            <div style="background:#fff3cd; padding:15px; border-radius:8px; margin-bottom:20px;">
                <label style="font-weight:600; display:block; margin-bottom:8px;">✈️ לאיזה כיוון להוסיף?</label>
                <select id="luggage_direction" name="luggage_direction" style="padding:10px; border-radius:8px; border:1px solid #ddd; width:100%; font-size:15px;">
                    <option value="both">🔄 הלוך + חזור (שניהם)</option>
                    <option value="outbound">🛫 הלוך בלבד</option>
                    <option value="return">🛬 חזור בלבד</option>
                </select>
            </div>
            
            <form id="luggageForm" onsubmit="submitLuggageAddon(event)">
                <input type="hidden" name="parent_order_id" value="<?= $order['id'] ?>">
                <input type="hidden" name="customer_name" value="<?= htmlspecialchars($passengers[0] ?? $order['customer_phone']) ?>">
                <input type="hidden" name="customer_phone" value="<?= htmlspecialchars($order['customer_phone']) ?>">
                <input type="hidden" name="addon_type" value="כבודה נוספת">
                
                <?php 
                $currentLuggage = json_decode(html_entity_decode($order['luggage_details'] ?? '[]'), true) ?: [];
                $passengerIndex = 0;
                foreach ($passengers as $passengerName): 
                    $existingLuggage = $currentLuggage[$passengerIndex] ?? ['trolley' => 0, 'suitcase' => 0, 'backpack' => false];
                    
                    // בדיקה אם יש כבודה נפרדת
                    $hasSplitLuggage = isset($existingLuggage['same_both_ways']) && !$existingLuggage['same_both_ways'];
                ?>
                <div class="luggage-passenger-row" style="background:#f8f9fa; padding:15px; border-radius:10px; margin-bottom:15px;">
                    <h4 style="margin:0 0 15px 0;">👤 <?= htmlspecialchars($passengerName) ?></h4>
                    <input type="hidden" name="luggage_passenger_name[]" value="<?= htmlspecialchars($passengerName) ?>">
                    
                    <!-- כבודה קיימת -->
                    <div style="background:#fff; padding:10px; border-radius:8px; margin-bottom:10px; border:1px solid #ddd;">
                        <small style="color:#666;">כבודה קיימת:</small><br>
                        <?php if ($hasSplitLuggage): ?>
                            <?php 
                            $out = $existingLuggage['outbound'] ?? [];
                            $ret = $existingLuggage['return'] ?? [];
                            ?>
                            <span style="color:#28a745; display:block;">
                                ✈️ הלוך: 
                                <?= !empty($out['trolley']) ? "Trolley {$out['trolley']}kg" : '' ?>
                                <?= !empty($out['suitcase']) ? "| Suitcase {$out['suitcase']}kg" : '' ?>
                                <?= !empty($out['backpack']) ? '| Backpack ✓' : '' ?>
                            </span>
                            <span style="color:#ff9800; display:block;">
                                ✈️ חזור: 
                                <?= !empty($ret['trolley']) ? "Trolley {$ret['trolley']}kg" : '' ?>
                                <?= !empty($ret['suitcase']) ? "| Suitcase {$ret['suitcase']}kg" : '' ?>
                                <?= !empty($ret['backpack']) ? '| Backpack ✓' : '' ?>
                            </span>
                            <input type="hidden" name="existing_trolley[]" value="<?= $out['trolley'] ?? 0 ?>">
                            <input type="hidden" name="existing_suitcase[]" value="<?= $out['suitcase'] ?? 0 ?>">
                            <input type="hidden" name="existing_backpack[]" value="<?= !empty($out['backpack']) ? '1' : '0' ?>">
                            <input type="hidden" name="existing_trolley_return[]" value="<?= $ret['trolley'] ?? 0 ?>">
                            <input type="hidden" name="existing_suitcase_return[]" value="<?= $ret['suitcase'] ?? 0 ?>">
                            <input type="hidden" name="existing_backpack_return[]" value="<?= !empty($ret['backpack']) ? '1' : '0' ?>">
                        <?php else: ?>
                            <span style="color:#28a745;">
                                ✅ Trolley: <?= $existingLuggage['trolley'] ?? 0 ?>kg | 
                                Suitcase: <?= $existingLuggage['suitcase'] ?? 0 ?>kg
                                <?= !empty($existingLuggage['backpack']) ? '| Backpack ✓' : '' ?>
                            </span>
                            <input type="hidden" name="existing_trolley[]" value="<?= $existingLuggage['trolley'] ?? 0 ?>">
                            <input type="hidden" name="existing_suitcase[]" value="<?= $existingLuggage['suitcase'] ?? 0 ?>">
                            <input type="hidden" name="existing_backpack[]" value="<?= !empty($existingLuggage['backpack']) ? '1' : '0' ?>">
                            <input type="hidden" name="existing_trolley_return[]" value="<?= $existingLuggage['trolley'] ?? 0 ?>">
                            <input type="hidden" name="existing_suitcase_return[]" value="<?= $existingLuggage['suitcase'] ?? 0 ?>">
                            <input type="hidden" name="existing_backpack_return[]" value="<?= !empty($existingLuggage['backpack']) ? '1' : '0' ?>">
                        <?php endif; ?>
                    </div>
                    
                    <!-- הוספת כבודה -->
                    <div style="display:flex; gap:15px; flex-wrap:wrap; align-items:center;">
                        <div>
                            <label style="font-size:12px; color:#666;">➕ Trolley נוסף</label>
                            <select name="add_trolley[]" class="luggage-select">
                                <option value="0">ללא</option>
                                <option value="7">+7kg</option>
                                <option value="8">+8kg</option>
                                <option value="10">+10kg</option>
                            </select>
                        </div>
                        <div>
                            <label style="font-size:12px; color:#666;">➕ Suitcase נוספת</label>
                            <select name="add_suitcase[]" class="luggage-select">
                                <option value="0">ללא</option>
                                <option value="15">+15kg</option>
                                <option value="20">+20kg</option>
                                <option value="23">+23kg</option>
                                <option value="25">+25kg</option>
                                <option value="30">+30kg</option>
                            </select>
                        </div>
                        <div>
                            <label style="font-size:12px; color:#666;">➕ Backpack</label>
                            <select name="add_backpack[]" class="luggage-select">
                                <option value="0">ללא</option>
                                <option value="1" <?= !empty($existingLuggage['backpack']) ? 'disabled' : '' ?>>הוסף</option>
                            </select>
                        </div>
                    </div>
                </div>
                <?php 
                $passengerIndex++;
                endforeach; 
                ?>
                
                <!-- פרטי ספק -->
                <div style="background:#e3f2fd; padding:15px; border-radius:8px; margin:20px 0; border:2px solid #2196f3;">
                    <h4 style="margin:0 0 15px 0;">📦 פרטי ספק</h4>
                    <div style="display:grid; grid-template-columns: 1fr 1fr; gap:15px;">
                        <div>
                            <label style="font-size:12px; color:#666; display:block; margin-bottom:5px;">ספק</label>
                            <input type="text" name="supplier_name" value="<?= htmlspecialchars($flightSupplier) ?>" 
                                   readonly style="padding:10px; border:1px solid #ddd; border-radius:8px; width:100%; background:#f5f5f5;">
                        </div>
                        <div>
                            <label style="font-size:12px; color:#666; display:block; margin-bottom:5px;">מספר הזמנה ספק</label>
                            <input type="text" name="supplier_booking" value="<?= htmlspecialchars($flightSupplierBooking) ?>" 
                                   readonly style="padding:10px; border:1px solid #ddd; border-radius:8px; width:100%; background:#f5f5f5;">
                        </div>
                    </div>
                    
                    <div style="display:grid; grid-template-columns: 1fr 1fr 1fr; gap:15px; margin-top:15px;">
                        <div>
                            <label style="font-size:12px; color:#666; display:block; margin-bottom:5px;">עלות לספק *</label>
                            <input type="number" name="supplier_cost" id="supplier_cost" required min="0" step="0.01" 
                                   style="padding:10px; border:1px solid #ddd; border-radius:8px; width:100%;" placeholder="0">
                        </div>
                        <div>
                            <label style="font-size:12px; color:#666; display:block; margin-bottom:5px;">מטבע</label>
                            <select name="supplier_currency" style="padding:10px; border:1px solid #ddd; border-radius:8px; width:100%;">
                                <option value="USD" <?= $flightSupplierCurrency === 'USD' ? 'selected' : '' ?>>$ USD</option>
                                <option value="EUR" <?= $flightSupplierCurrency === 'EUR' ? 'selected' : '' ?>>€ EUR</option>
                                <option value="ILS" <?= $flightSupplierCurrency === 'ILS' ? 'selected' : '' ?>>₪ ILS</option>
                            </select>
                        </div>
                        <div>
                            <label style="font-size:12px; color:#666; display:block; margin-bottom:5px;">סטטוס תשלום</label>
                            <select name="supplier_paid" style="padding:10px; border:1px solid #ddd; border-radius:8px; width:100%;">
                                <option value="unpaid">⏳ טרם שולם</option>
                                <option value="paid">✅ שולם</option>
                            </select>
                        </div>
                    </div>
                </div>
                
                <!-- מחיר ללקוח -->
                <div style="background:#fff3cd; padding:15px; border-radius:8px; margin:20px 0;">
                    <div style="display:flex; justify-content:space-between; align-items:center;">
                        <span><strong>💰 מחיר ללקוח:</strong></span>
                        <div>
                            <input type="number" name="total_price" id="luggage_price" required min="1" step="1" 
                                   style="width:100px; padding:8px; font-size:16px; font-weight:bold;" placeholder="0">
                            <span>₪</span>
                        </div>
                    </div>
                    <small style="color:#666;">הזן את המחיר הכולל לתוספת הכבודה</small>
                </div>
                
                <button type="submit" class="btn-submit">📤 צור טיוטה ושלח ללקוח</button>
            </form>
        </div>
    </div>
    
    <script>
        function openAddonModal() {
            document.getElementById('addonModal').classList.add('active');
        }
        
        function closeAddonModal() {
            document.getElementById('addonModal').classList.remove('active');
            document.getElementById('addon_type').value = '';
        }
        
        function openLuggageModal() {
            closeAddonModal();
            document.getElementById('luggageModal').classList.add('active');
        }
        
        function closeLuggageModal() {
            document.getElementById('luggageModal').classList.remove('active');
        }
        
        function handleAddonTypeChange() {
            const type = document.getElementById('addon_type').value;
            if (type === 'כבודה נוספת') {
                openLuggageModal();
            } else {
                const desc = document.getElementById('addon_description');
                if (type && type !== 'אחר') {
                    desc.placeholder = 'פרט: ' + type;
                }
            }
        }
        
        async function submitAddon(e) {
            e.preventDefault();
            
            const form = document.getElementById('addonForm');
            const formData = new FormData(form);
            
            const type = formData.get('addon_type');
            const details = formData.get('addon_description');
            const fullDescription = type + (details ? ': ' + details : '');
            formData.set('addon_description', fullDescription);
            formData.set('destination', fullDescription);
            
            try {
                const response = await fetch('create_addon_draft.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    alert('✅ הטיוטה נוצרה!\n\nתוכל לשלוח אותה ללקוח מרשימת הטיוטות.');
                    location.reload();
                } else {
                    alert('❌ שגיאה: ' + result.error);
                }
            } catch (error) {
                alert('❌ שגיאה: ' + error.message);
            }
        }
        
        async function submitLuggageAddon(e) {
            e.preventDefault();
            
            const form = document.getElementById('luggageForm');
            const formData = new FormData(form);
            const direction = document.getElementById('luggage_direction').value;
            
            // בניית מערך הכבודה הנוספת
            const addTrolleys = form.querySelectorAll('select[name="add_trolley[]"]');
            const addSuitcases = form.querySelectorAll('select[name="add_suitcase[]"]');
            const addBackpacks = form.querySelectorAll('select[name="add_backpack[]"]');
            const passengerNames = form.querySelectorAll('input[name="luggage_passenger_name[]"]');
            const existingTrolleys = form.querySelectorAll('input[name="existing_trolley[]"]');
            const existingSuitcases = form.querySelectorAll('input[name="existing_suitcase[]"]');
            const existingBackpacks = form.querySelectorAll('input[name="existing_backpack[]"]');
            const existingTrolleysReturn = form.querySelectorAll('input[name="existing_trolley_return[]"]');
            const existingSuitcasesReturn = form.querySelectorAll('input[name="existing_suitcase_return[]"]');
            const existingBackpacksReturn = form.querySelectorAll('input[name="existing_backpack_return[]"]');
            
            let luggageData = [];
            let description = [];
            
            const directionText = {
                'both': 'הלוך+חזור',
                'outbound': 'הלוך בלבד',
                'return': 'חזור בלבד'
            };
            
            for (let i = 0; i < passengerNames.length; i++) {
                const name = passengerNames[i].value;
                const addTrolley = parseInt(addTrolleys[i].value) || 0;
                const addSuitcase = parseInt(addSuitcases[i].value) || 0;
                const addBackpack = parseInt(addBackpacks[i].value) || 0;
                const existingTrolley = parseInt(existingTrolleys[i].value) || 0;
                const existingSuitcase = parseInt(existingSuitcases[i].value) || 0;
                const existingBackpack = parseInt(existingBackpacks[i].value) || 0;
                const existingTrolleyReturn = existingTrolleysReturn[i] ? parseInt(existingTrolleysReturn[i].value) || 0 : existingTrolley;
                const existingSuitcaseReturn = existingSuitcasesReturn[i] ? parseInt(existingSuitcasesReturn[i].value) || 0 : existingSuitcase;
                const existingBackpackReturn = existingBackpacksReturn[i] ? parseInt(existingBackpacksReturn[i].value) || 0 : existingBackpack;
                
                // שמירת הכבודה הנוספת
                if (addTrolley > 0 || addSuitcase > 0 || addBackpack > 0) {
                    luggageData.push({
                        name: name,
                        direction: direction,
                        add_trolley: addTrolley,
                        add_suitcase: addSuitcase,
                        add_backpack: addBackpack,
                        existing_trolley: existingTrolley,
                        existing_suitcase: existingSuitcase,
                        existing_backpack: existingBackpack,
                        existing_trolley_return: existingTrolleyReturn,
                        existing_suitcase_return: existingSuitcaseReturn,
                        existing_backpack_return: existingBackpackReturn
                    });
                    
                    let parts = [];
                    if (addTrolley > 0) parts.push('Trolley +' + addTrolley + 'kg');
                    if (addSuitcase > 0) parts.push('Suitcase +' + addSuitcase + 'kg');
                    if (addBackpack > 0) parts.push('Backpack');
                    description.push(name + ': ' + parts.join(', '));
                }
            }
            
            if (luggageData.length === 0) {
                alert('❌ לא נבחרה כבודה להוספה');
                return;
            }
            
            const fullDescription = 'כבודה נוספת (' + directionText[direction] + '): ' + description.join(' | ');
            formData.set('addon_description', fullDescription);
            formData.set('destination', 'כבודה נוספת');
            formData.set('luggage_data', JSON.stringify(luggageData));
            formData.set('luggage_direction', direction);
            
            try {
                const response = await fetch('create_addon_draft.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    alert('✅ טיוטת הכבודה נוצרה!\n\nתוכל לשלוח אותה ללקוח מרשימת הטיוטות.');
                    location.reload();
                } else {
                    alert('❌ שגיאה: ' + result.error);
                }
            } catch (error) {
                alert('❌ שגיאה: ' + error.message);
            }
        }
        
        async function deleteDraft(draftId, description) {
            if (!confirm('למחוק את התוספת "' + description + '"?\n\nפעולה זו לא ניתנת לביטול!')) {
                return;
            }
            
            try {
                const response = await fetch('delete_draft.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: 'draft_id=' + draftId
                });
                
                const result = await response.json();
                
                if (result.success) {
                    alert('✅ התוספת נמחקה בהצלחה!');
                    location.reload();
                } else {
                    alert('❌ שגיאה: ' + result.error);
                }
            } catch (error) {
                alert('❌ שגיאה: ' + error.message);
            }
        }
        
        // סגירה בלחיצה מחוץ למודאל
        document.getElementById('addonModal').addEventListener('click', function(e) {
            if (e.target === this) closeAddonModal();
        });
        document.getElementById('luggageModal').addEventListener('click', function(e) {
            if (e.target === this) closeLuggageModal();
        });
    </script>
</body>
</html>
