<?php
// צפייה בהזמנה - גרסה חדשה
require_once 'includes/auth.php';
requireLogin();
blockAccountant();

$agent = getCurrentAgent();
$isAdmin = ($agent && $agent['role'] === 'admin');
$order_id = $_GET['id'] ?? null;

if (!$order_id) {
    header('Location: orders_panel.php');
    exit;
}

$pdo = getDB();
$stmt = $pdo->prepare("SELECT o.*, a.full_name as agent_name FROM orders o JOIN agents a ON o.agent_id = a.id WHERE o.id = ?");
$stmt->execute([$order_id]);
$order = $stmt->fetch();

// טעינת רשימת נציגים לעריכה
$agentsStmt = $pdo->query("SELECT id, full_name FROM agents WHERE role != 'accountant' ORDER BY full_name");
$agents = $agentsStmt->fetchAll(PDO::FETCH_ASSOC);

if (!$order) {
    header('Location: orders_panel.php');
    exit;
}

// נציג יכול לראות רק הזמנות שלו
if (!$isAdmin && $order['agent_id'] != $agent['id']) {
    header('Location: orders_panel.php');
    exit;
}

$passengers = json_decode($order['passengers'] ?? '[]', true);
$payments = json_decode($order['payments'] ?? '[]', true);
$extras = json_decode($order['extras'] ?? '[]', true);

// שליפת תשלומים מתוספות (דרך טיוטות)
$additionPayments = [];
try {
    $stmt = $pdo->prepare("
        SELECT dp.*, d.addon_description, d.payment_method as draft_payment_method
        FROM draft_payments dp
        JOIN drafts d ON dp.draft_id = d.id
        WHERE d.parent_order_id = ?
        ORDER BY dp.payment_date
    ");
    $stmt->execute([$order_id]);
    $additionPayments = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    // טבלה לא קיימת
}

// שליפת סטטוסי העברות בנקאיות
$bankTransfers = [];
try {
    $stmt = $pdo->prepare("
        SELECT bt.*, 
               d.order_id as draft_order_id,
               d.parent_order_id
        FROM bank_transfers bt
        LEFT JOIN drafts d ON bt.draft_id = d.id
        WHERE bt.order_id = ? OR d.order_id = ? OR d.parent_order_id = ?
        ORDER BY bt.created_at DESC
    ");
    $stmt->execute([$order_id, $order_id, $order_id]);
    $bankTransfers = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    // טבלה לא קיימת
}

// שליפת תוספות לרכיבים
$stmt = $pdo->prepare("SELECT * FROM order_additions WHERE order_id = ? ORDER BY component, created_at");
$stmt->execute([$order_id]);
$allAdditions = $stmt->fetchAll(PDO::FETCH_ASSOC);

// מיון תוספות לפי רכיב
$additions = [
    'flight_outbound' => [],
    'flight_return' => [],
    'hotel' => []
];
$totalAdditionsProfit = 0;
$totalAdditionsCustomerCharge = 0;
$totalAdditionsCostIls = 0;

foreach ($allAdditions as $add) {
    if (isset($additions[$add['component']])) {
        $additions[$add['component']][] = $add;
    }
    $totalAdditionsProfit += floatval($add['profit'] ?? 0);
    $totalAdditionsCustomerCharge += floatval($add['customer_charge'] ?? 0);
    $totalAdditionsCostIls += floatval($add['amount_ils'] ?? 0);
}

// שליפת שערי מטבע עדכניים
$currentRates = ['USD' => 3.65, 'EUR' => 3.95, 'ILS' => 1];
$ratesResult = $pdo->query("SELECT currency, rate FROM exchange_rates");
while ($row = $ratesResult->fetch()) {
    $currentRates[$row['currency']] = (float)$row['rate'];
}

// שליפת מלונות מהטבלה החדשה
$orderHotels = [];
try {
    $stmt = $pdo->prepare("SELECT * FROM order_hotels WHERE order_id = ? ORDER BY sort_order");
    $stmt->execute([$order_id]);
    $orderHotels = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    // טבלה לא קיימת - נשתמש בנתונים מטבלת orders
}

// חישוב סה"כ תשלומים מלקוחות
$totalPayments = 0;
foreach ($payments as $p) {
    $totalPayments += floatval($p['amount'] ?? 0);
}

// חישוב תשלומי תוספות
$totalAdditionPayments = 0;
foreach ($additionPayments as $ap) {
    $totalAdditionPayments += floatval($ap['amount'] ?? 0);
}

// שליפת החזרים
$stmt = $pdo->prepare("SELECT * FROM order_refunds WHERE order_id = ? ORDER BY refund_date DESC, created_at DESC");
$stmt->execute([$order_id]);
$refunds = $stmt->fetchAll(PDO::FETCH_ASSOC);

// סיכום החזרים
$totalRefundReceived = 0;
$totalRefundReturned = 0;
foreach ($refunds as $refund) {
    $totalRefundReceived += floatval($refund['received_from_supplier'] ?? 0);
    $totalRefundReturned += floatval($refund['returned_to_customer'] ?? 0);
}
$totalRefundProfit = $totalRefundReceived - $totalRefundReturned;

// חישוב עלויות ספקים
$flightCost = floatval($order['flight_supplier_cost'] ?? 0);
$returnFlightCost = floatval($order['return_flight_supplier_cost'] ?? 0);

// חישוב עלות מלונות - מהטבלה החדשה או מהישנה
$hotelCost = 0;
$hotelCostIls = 0;
if (!empty($orderHotels)) {
    // יש מלונות בטבלה החדשה
    foreach ($orderHotels as $hotel) {
        $cost = floatval($hotel['supplier_cost'] ?? 0);
        $currency = $hotel['supplier_currency'] ?? 'USD';
        $hotelCost += $cost; // לתצוגה (המטבע הראשון)
        // המרה לשקלים
        if ($currency === 'USD') {
            $hotelCostIls += $cost * $currentRates['USD'];
        } elseif ($currency === 'EUR') {
            $hotelCostIls += $cost * $currentRates['EUR'];
        } else {
            $hotelCostIls += $cost;
        }
    }
} else {
    // fallback לנתונים מטבלת orders
    $hotelCost = floatval($order['hotel_supplier_cost'] ?? 0);
    $hotelCurrency = $order['hotel_supplier_currency'] ?? 'USD';
    if ($hotelCurrency === 'USD') {
        $hotelCostIls = $hotelCost * $currentRates['USD'];
    } elseif ($hotelCurrency === 'EUR') {
        $hotelCostIls = $hotelCost * $currentRates['EUR'];
    } else {
        $hotelCostIls = $hotelCost;
    }
}

// חישוב עלות שונות (extras) בשקלים
$extrasCostIls = 0;
if (!empty($extras)) {
    $usdRate = floatval($order['usd_rate'] ?? 3.65);
    $eurRate = floatval($order['eur_rate'] ?? 3.95);
    foreach ($extras as $extra) {
        $cost = floatval($extra['cost'] ?? 0);
        $currency = $extra['currency'] ?? 'ILS';
        if ($currency === 'USD') {
            $extrasCostIls += $cost * $usdRate;
        } elseif ($currency === 'EUR') {
            $extrasCostIls += $cost * $eurRate;
        } else {
            $extrasCostIls += $cost;
        }
    }
}

// סמלי מטבע
$currencySymbols = ['ILS' => '₪', 'USD' => '$', 'EUR' => '€'];

$statusNames = [
    'draft' => 'טיוטה',
    'pending' => 'ממתין',
    'paid' => 'שולם',
    'confirmed' => 'מאושר',
    'cancelled' => 'בוטל'
];

$statusColors = [
    'draft' => '#6c757d',
    'pending' => '#ffc107',
    'paid' => '#28a745',
    'confirmed' => '#17a2b8',
    'cancelled' => '#dc3545'
];
?>
<!DOCTYPE html>
<html lang="he" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>הזמנה #<?= $order['id'] ?> | <?= SITE_NAME ?></title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: 'Segoe UI', Tahoma, sans-serif;
            background: #f0f2f5;
            min-height: 100vh;
            color: #333;
        }
        
        /* Header */
        .header {
            background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
            color: white;
            padding: 20px 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-wrap: wrap;
            gap: 15px;
        }
        .header-right {
            display: flex;
            align-items: center;
            gap: 20px;
        }
        .order-number {
            font-size: 1.8rem;
            font-weight: bold;
        }
        .order-status {
            padding: 8px 20px;
            border-radius: 25px;
            font-weight: 600;
            font-size: 0.95rem;
        }
        .header-actions {
            display: flex;
            gap: 10px;
        }
        .btn {
            padding: 10px 20px;
            border-radius: 8px;
            text-decoration: none;
            font-weight: 500;
            cursor: pointer;
            border: none;
            font-size: 0.95rem;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }
        .btn-light { background: rgba(255,255,255,0.2); color: white; }
        .btn-light:hover { background: rgba(255,255,255,0.3); }
        .btn-primary { background: #667eea; color: white; }
        .btn-success { background: #28a745; color: white; }
        .btn-warning { background: #ffc107; color: #333; }
        
        /* Main Content */
        .main-content {
            max-width: 1400px;
            margin: 0 auto;
            padding: 25px;
        }
        
        /* Grid Layout */
        .grid-2 {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 20px;
        }
        .grid-3 {
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            gap: 20px;
        }
        @media (max-width: 1200px) {
            .grid-3 { grid-template-columns: repeat(2, 1fr); }
        }
        @media (max-width: 768px) {
            .grid-2, .grid-3 { grid-template-columns: 1fr; }
        }
        
        /* Cards */
        .card {
            background: white;
            border-radius: 12px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
            overflow: hidden;
        }
        .card-header {
            padding: 15px 20px;
            border-bottom: 1px solid #eee;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .card-header h3 {
            font-size: 1.1rem;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        .card-body {
            padding: 20px;
        }
        
        /* Data Rows */
        .data-row {
            display: flex;
            justify-content: space-between;
            padding: 10px 0;
            border-bottom: 1px solid #f0f0f0;
        }
        .data-row:last-child { border-bottom: none; }
        .data-label {
            color: #666;
            font-size: 0.9rem;
        }
        .data-value {
            font-weight: 500;
            text-align: left;
        }
        .data-value.highlight {
            color: #667eea;
            font-size: 1.1rem;
        }
        
        /* Flight/Hotel Cards */
        .supplier-card {
            background: #f8f9fa;
            border-radius: 10px;
            padding: 15px;
            margin-bottom: 15px;
        }
        .supplier-card:last-child { margin-bottom: 0; }
        .supplier-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 12px;
            padding-bottom: 10px;
            border-bottom: 1px solid #e0e0e0;
        }
        .supplier-name {
            font-weight: 600;
            font-size: 1rem;
        }
        .supplier-status {
            padding: 4px 12px;
            border-radius: 15px;
            font-size: 0.8rem;
            font-weight: 600;
        }
        .supplier-status.paid { background: #d4edda; color: #155724; }
        .supplier-status.unpaid { background: #f8d7da; color: #721c24; }
        
        .supplier-details {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 8px;
        }
        .supplier-detail {
            display: flex;
            flex-direction: column;
        }
        .supplier-detail-label {
            font-size: 0.75rem;
            color: #888;
            margin-bottom: 2px;
        }
        .supplier-detail-value {
            font-weight: 500;
            font-size: 0.95rem;
        }
        
        /* Payments Table */
        .payments-table {
            width: 100%;
            border-collapse: collapse;
        }
        .payments-table th {
            text-align: right;
            padding: 10px;
            background: #f8f9fa;
            font-weight: 600;
            font-size: 0.85rem;
            color: #555;
        }
        .payments-table td {
            padding: 10px;
            border-bottom: 1px solid #eee;
        }
        .payments-table tr:last-child td { border-bottom: none; }
        
        /* Passengers List */
        .passengers-list {
            display: flex;
            flex-wrap: wrap;
            gap: 8px;
        }
        .passenger-tag {
            background: #e8f4fd;
            color: #1976d2;
            padding: 6px 12px;
            border-radius: 20px;
            font-size: 0.9rem;
        }
        
        /* Summary Box */
        .summary-box {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 12px;
            padding: 25px;
        }
        .summary-row {
            display: flex;
            justify-content: space-between;
            padding: 8px 0;
        }
        .summary-row.total {
            border-top: 2px solid rgba(255,255,255,0.3);
            margin-top: 10px;
            padding-top: 15px;
            font-size: 1.3rem;
            font-weight: bold;
        }
        .summary-label { opacity: 0.9; }
        
        /* תיקון כיווניות מספרים */
        .summary-row span:last-child,
        .data-value,
        [class*="amount"],
        [class*="cost"],
        [class*="price"],
        [class*="profit"] {
            direction: ltr;
            unicode-bidi: isolate;
        }
        
        /* Notes */
        .notes-box {
            background: #fffbeb;
            border: 1px solid #fcd34d;
            border-radius: 8px;
            padding: 15px;
            white-space: pre-wrap;
        }
        
        /* Section Spacing */
        .section { margin-bottom: 25px; }
        .section-title {
            font-size: 1.2rem;
            font-weight: 600;
            margin-bottom: 15px;
            color: #333;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        /* Voucher link */
        .voucher-link {
            display: inline-flex;
            align-items: center;
            gap: 5px;
            color: #667eea;
            text-decoration: none;
            font-size: 0.85rem;
        }
        .voucher-link:hover { text-decoration: underline; }

        /* Full width card */
        .full-width { grid-column: 1 / -1; }
        
        /* תוספות */
        .additions-section {
            margin-top: 15px;
            padding-top: 12px;
            border-top: 1px dashed #ddd;
        }
        .additions-title {
            font-size: 0.85rem;
            color: #666;
            margin-bottom: 8px;
        }
        .addition-row {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 6px 0;
            font-size: 0.9rem;
            flex-wrap: wrap;
            gap: 5px;
        }
        .addition-info {
            display: flex;
            flex-direction: column;
            gap: 2px;
        }
        .addition-desc { color: #555; }
        .addition-date {
            font-size: 0.75rem;
            color: #999;
        }
        .addition-amount {
            display: flex;
            align-items: center;
            gap: 8px;
            flex-wrap: wrap;
        }
        .btn-charge-customer {
            background: #28a745;
            border: none;
            border-radius: 4px;
            padding: 3px 8px;
            cursor: pointer;
            font-size: 0.85rem;
        }
        .btn-charge-customer:hover {
            background: #218838;
        }
        .paid-badge {
            font-size: 0.75rem;
            padding: 2px 6px;
            border-radius: 10px;
        }
        .paid-badge.paid { background: #d4edda; color: #155724; }
        .paid-badge.unpaid { background: #fff3cd; color: #856404; }
        .btn-delete-addition {
            background: none;
            border: none;
            color: #dc3545;
            cursor: pointer;
            font-size: 1.1rem;
            padding: 0 5px;
            opacity: 0.6;
        }
        .btn-delete-addition:hover { opacity: 1; }
        .total-row {
            margin-top: 10px;
            padding-top: 10px;
            border-top: 1px solid #ddd;
        }
        .btn-add-addition {
            display: block;
            width: 100%;
            margin-top: 15px;
            padding: 8px;
            background: #f8f9fa;
            border: 1px dashed #ccc;
            border-radius: 6px;
            color: #666;
            cursor: pointer;
            font-size: 0.85rem;
            transition: all 0.2s;
        }
        .btn-add-addition:hover {
            background: #e9ecef;
            border-color: #667eea;
            color: #667eea;
        }
        
        /* מודאל */
        .modal-overlay {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0,0,0,0.5);
            z-index: 1000;
            justify-content: center;
            align-items: center;
        }
        .modal-overlay.active { display: flex; }
        .modal-content {
            background: white;
            padding: 25px;
            border-radius: 12px;
            width: 90%;
            max-width: 400px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
        }
        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }
        .modal-header h3 { margin: 0; color: #333; }
        .modal-close {
            background: none;
            border: none;
            font-size: 1.5rem;
            cursor: pointer;
            color: #999;
        }
        .modal-close:hover { color: #333; }
        .form-group {
            margin-bottom: 15px;
        }
        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: 500;
            color: #555;
        }
        .form-group input, .form-group select {
            width: 100%;
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 6px;
            font-size: 1rem;
        }
        .form-row {
            display: flex;
            gap: 10px;
        }
        .form-row .form-group { flex: 1; }
        .checkbox-group {
            display: flex;
            align-items: center;
            gap: 8px;
        }
        .checkbox-group input { width: auto; }
        .modal-footer {
            display: flex;
            gap: 10px;
            margin-top: 20px;
        }
        .modal-footer button {
            flex: 1;
            padding: 12px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 1rem;
            font-weight: 500;
        }
        .btn-cancel { background: #f0f0f0; color: #333; }
        .btn-save { background: #667eea; color: white; }
        .btn-save:hover { background: #5a6fd6; }
        
        /* כפתורי החזרים */
        .btn-add-small {
            background: #28a745;
            color: white;
            border: none;
            padding: 6px 12px;
            border-radius: 5px;
            cursor: pointer;
            font-size: 13px;
        }
        .btn-add-small:hover { background: #218838; }
        
        .btn-delete-small {
            background: transparent;
            border: none;
            cursor: pointer;
            font-size: 16px;
            padding: 4px;
            opacity: 0.6;
        }
        .btn-delete-small:hover { opacity: 1; }
        .exchange-rate-info {
            font-size: 0.8rem;
            color: #888;
            margin-top: 5px;
        }
        .profit-preview {
            background: #f0f7ff;
            padding: 10px 15px;
            border-radius: 6px;
            margin: 10px 0;
            font-weight: 500;
        }
        .profit-preview #profitAmount {
            color: #28a745;
        }
        .profit-preview #profitAmount.negative {
            color: #dc3545;
        }
    </style>
</head>
<body>
    <!-- Header -->
    <div class="header">
        <div class="header-right">
            <div class="order-number">🎫 הזמנה #<?= $order['id'] ?></div>
            <div class="order-status" style="background: <?= $statusColors[$order['status']] ?? '#6c757d' ?>">
                <?= $statusNames[$order['status']] ?? $order['status'] ?>
            </div>
        </div>
        <div class="header-actions">
            <a href="orders_new.php" class="btn btn-light">← חזרה לרשימה</a>
            <button onclick="window.print()" class="btn btn-light">🖨️ הדפסה</button>
            <button onclick="window.location='send_confirmation.php?id=<?= $order['id'] ?>'" class="btn btn-success">📧 שלח אישור ללקוח</button>
            <a href="edit_order.php?id=<?= $order['id'] ?>" class="btn btn-warning">✏️ עריכה</a>
        </div>
    </div>
    
    <div class="main-content">
        <!-- שורה עליונה: פרטי הזמנה + לקוח + נוסעים -->
        <div class="section">
            <div class="grid-3">
                <!-- פרטי הזמנה -->
                <div class="card">
                    <div class="card-header">
                        <h3>📅 פרטי הזמנה</h3>
                    </div>
                    <div class="card-body">
                        <?php 
                        $orderType = $order['order_type'] ?? 'separate';
                        $orderTypeLabels = [
                            'separate' => ['label' => 'חבילה נפרדת', 'icon' => '✈️🏨', 'color' => '#667eea'],
                            'package' => ['label' => 'נופשון קומפלט', 'icon' => '📦', 'color' => '#28a745'],
                            'tour' => ['label' => 'טיול מאורגן', 'icon' => '🚌', 'color' => '#fd7e14'],
                            'organized' => ['label' => 'טיול מאורגן', 'icon' => '🚌', 'color' => '#fd7e14']
                        ];
                        $typeInfo = $orderTypeLabels[$orderType] ?? $orderTypeLabels['separate'];
                        ?>
                        <div class="data-row" style="background: <?= $typeInfo['color'] ?>15; padding: 8px; border-radius: 8px; margin-bottom: 10px;">
                            <span class="data-label">סוג הזמנה</span>
                            <span class="data-value" style="color: <?= $typeInfo['color'] ?>; font-weight: bold;">
                                <?= $typeInfo['icon'] ?> <?= $typeInfo['label'] ?>
                            </span>
                        </div>
                        <div class="data-row">
                            <span class="data-label">תאריך הזמנה</span>
                            <span class="data-value editable" data-field="order_date" data-type="date" data-value="<?= $order['order_date'] ?? $order['created_at'] ?>" onclick="startInlineEdit(this)"><?= date('d/m/Y', strtotime($order['order_date'] ?? $order['created_at'])) ?></span>
                        </div>
                        <div class="data-row">
                            <span class="data-label">שעת הזמנה</span>
                            <span class="data-value editable" style="cursor: default;"><?= date('H:i', strtotime($order['created_at'])) ?></span>
                        </div>
                        <div class="data-row">
                            <span class="data-label">יעד</span>
                            <span class="data-value highlight editable" data-field="destination" data-type="text" data-value="<?= htmlspecialchars($order['destination']) ?>" onclick="startInlineEdit(this)"><?= htmlspecialchars($order['destination']) ?></span>
                        </div>
                        <div class="data-row">
                            <span class="data-label">תאריך יציאה</span>
                            <span class="data-value editable" data-field="departure_date" data-type="date" data-value="<?= $order['departure_date'] ?>" onclick="startInlineEdit(this)"><?= date('d/m/Y', strtotime($order['departure_date'])) ?></span>
                        </div>
                        <div class="data-row">
                            <span class="data-label">תאריך חזרה</span>
                            <span class="data-value editable" data-field="return_date" data-type="date" data-value="<?= $order['return_date'] ?>" onclick="startInlineEdit(this)"><?= date('d/m/Y', strtotime($order['return_date'])) ?></span>
                        </div>
                        <div class="data-row">
                            <span class="data-label">נציג</span>
                            <?php $agentOptions = implode(',', array_map(function($a) { return $a['full_name']; }, $agents)); ?>
                            <span class="data-value editable" data-field="agent_id" data-type="select" data-value="<?= htmlspecialchars($order['agent_name']) ?>" data-options="<?= htmlspecialchars($agentOptions) ?>" onclick="startInlineEdit(this)"><?= htmlspecialchars($order['agent_name']) ?></span>
                        </div>
                        <div class="data-row">
                            <span class="data-label">🏆 בונה הדיל</span>
                            <span class="data-value editable" data-field="deal_builder" data-type="select" data-value="<?= htmlspecialchars($order['deal_builder'] ?? '') ?>" data-options="<?= htmlspecialchars($agentOptions) ?>" onclick="startInlineEdit(this)"><?= htmlspecialchars($order['deal_builder'] ?? '-') ?></span>
                        </div>
                    </div>
                </div>
                
                <!-- פרטי לקוח -->
                <div class="card">
                    <div class="card-header">
                        <h3>👤 פרטי לקוח</h3>
                    </div>
                    <div class="card-body">
                        <div class="data-row">
                            <span class="data-label">טלפון</span>
                            <span class="data-value editable" data-field="customer_phone" data-type="text" data-value="<?= htmlspecialchars($order['customer_phone']) ?>" onclick="startInlineEdit(this)">
                                <?= htmlspecialchars($order['customer_phone']) ?>
                            </span>
                        </div>
                        <?php if ($order['customer_email']): ?>
                        <div class="data-row">
                            <span class="data-label">אימייל</span>
                            <span class="data-value editable" data-field="customer_email" data-type="text" data-value="<?= htmlspecialchars($order['customer_email']) ?>" onclick="startInlineEdit(this)">
                                <?= htmlspecialchars($order['customer_email']) ?>
                            </span>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- נוסעים -->
                <div class="card">
                    <div class="card-header">
                        <h3>👥 נוסעים (<?= count($passengers) ?>)</h3>
                    </div>
                    <div class="card-body">
                        <div class="passengers-list" id="passengers-list">
                            <?php foreach ($passengers as $i => $passenger): ?>
                                <span class="passenger-tag editable-passenger" data-index="<?= $i ?>" onclick="startPassengerEdit(this)"><?= ($i + 1) ?>. <?= htmlspecialchars($passenger) ?></span>
                            <?php endforeach; ?>
                            <button class="btn-add-passenger" onclick="addNewPassenger()" title="הוסף נוסע">+</button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- שורה שנייה: טיסות + מלון -->
        <div class="section">
            <div class="grid-3">
                <!-- טיסת הלוך -->
                <div class="card">
                    <div class="card-header">
                        <h3>✈️ טיסת הלוך</h3>
                        <?php
                        $flightOutAdditions = $additions['flight_outbound'];
                        $flightOutAdditionsTotal = array_sum(array_column($flightOutAdditions, 'amount'));
                        $flightOutPaidAdditions = array_sum(array_map(function($a) { return $a['is_paid'] ? $a['amount'] : 0; }, $flightOutAdditions));
                        $flightOutUnpaidAdditions = $flightOutAdditionsTotal - $flightOutPaidAdditions;
                        $flightOutTotalCost = $flightCost + $flightOutAdditionsTotal;
                        $basePaid = ($order['flight_supplier_paid'] ?? 'unpaid') === 'paid';
                        $allPaid = $basePaid && ($flightOutUnpaidAdditions == 0);
                        ?>
                        <span class="supplier-status <?= $allPaid ? 'paid' : 'unpaid' ?>">
                            <?= $allPaid ? '✓ שולם' : '⏳ לא שולם' ?>
                        </span>
                    </div>
                    <div class="card-body">
                        <div class="data-row">
                            <span class="data-label">ספק</span>
                            <span class="data-value editable" data-field="flight_supplier" data-type="text" data-value="<?= htmlspecialchars($order['flight_supplier'] ?? '') ?>" onclick="startInlineEdit(this)"><?= htmlspecialchars($order['flight_supplier'] ?? '-') ?></span>
                        </div>
                        <div class="data-row">
                            <span class="data-label">חברת תעופה</span>
                            <span class="data-value editable" data-field="airline_outbound" data-type="text" data-value="<?= htmlspecialchars($order['airline_outbound'] ?? '') ?>" onclick="startInlineEdit(this)"><?= htmlspecialchars($order['airline_outbound'] ?? '-') ?></span>
                        </div>
                        <div class="data-row">
                            <span class="data-label">קוד הזמנה</span>
                            <span class="data-value highlight editable" data-field="booking_code_outbound" data-type="text" data-value="<?= htmlspecialchars($order['booking_code_outbound'] ?? '') ?>" onclick="startInlineEdit(this)"><?= htmlspecialchars($order['booking_code_outbound'] ?? '-') ?></span>
                        </div>
                        <div class="data-row">
                            <span class="data-label">עלות בסיס</span>
                            <span class="data-value">
                                <?php 
                                $sym = $currencySymbols[$order['flight_supplier_currency'] ?? 'USD'] ?? '$';
                                ?>
                                <span class="editable" data-field="flight_supplier_cost" data-type="number" data-value="<?= $flightCost ?>" onclick="startInlineEdit(this)"><?= $sym . number_format($flightCost, 2) ?></span>
                                <span class="paid-badge editable <?= $basePaid ? 'paid' : 'unpaid' ?>" data-field="flight_supplier_paid" data-type="paid" data-value="<?= $basePaid ? 'paid' : 'unpaid' ?>" onclick="startInlineEdit(this)"><?= $basePaid ? '✓ שולם' : '⏳ לא שולם' ?></span>
                            </span>
                        </div>
                        
                        <?php if (!empty($flightOutAdditions)): ?>
                        <div class="additions-section">
                            <div class="additions-title">תוספות:</div>
                            <?php foreach ($flightOutAdditions as $add): ?>
                            <div class="addition-row">
                                <div class="addition-info">
                                    <span class="addition-desc">+ <?= htmlspecialchars($add['description']) ?></span>
                                    <span class="addition-date"><?= date('d/m/Y', strtotime($add['created_at'])) ?></span>
                                </div>
                                <span class="addition-amount">
                                    לספק: <?= $currencySymbols[$add['currency']] ?? '$' ?><?= number_format($add['amount'], 2) ?>
                                    <?php if ($add['customer_charge'] > 0): ?>
                                    | מלקוח: ₪<?= number_format($add['customer_charge'], 2) ?>
                                    <?php if (!empty($add['customer_paid'])): ?>
                                        <span class="paid-badge paid">💳 שולם</span>
                                    <?php else: ?>
                                        <button class="btn-charge-customer" onclick="chargeCustomerForAddition(<?= $add['id'] ?>, <?= $add['customer_charge'] ?>, '<?= htmlspecialchars($add['description']) ?>')" title="חייב לקוח">💳 לחיוב</button>
                                    <?php endif; ?>
                                    | רווח: <span style="color:<?= $add['profit'] >= 0 ? '#28a745' : '#dc3545' ?>">₪<?= number_format($add['profit'], 2) ?></span>
                                    <?php endif; ?>
                                    <span class="paid-badge <?= $add['is_paid'] ? 'paid' : 'unpaid' ?>" onclick="toggleAdditionPaid(<?= $add['id'] ?>, <?= $add['is_paid'] ? 0 : 1 ?>)" style="cursor:pointer;" title="לספק: לחץ לשינוי"><?= $add['is_paid'] ? '✓ שולם לספק' : '⏳ לא שולם לספק' ?></span>
                                    <button class="btn-delete-addition" onclick="deleteAddition(<?= $add['id'] ?>)" title="מחק">×</button>
                                </span>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <?php endif; ?>
                        
                        <?php if ($flightOutAdditionsTotal > 0): ?>
                        <div class="data-row total-row">
                            <span class="data-label"><strong>סה"כ רכיב</strong></span>
                            <span class="data-value"><strong><?= $sym ?><?= number_format($flightOutTotalCost, 2) ?></strong></span>
                        </div>
                        <?php endif; ?>
                        
                        <button class="btn-add-addition" onclick="openAdditionModal('flight_outbound', 'טיסת הלוך', '<?= $order['flight_supplier_currency'] ?? 'USD' ?>')">
                            + הוסף תוספת
                        </button>
                    </div>
                </div>
                
                <!-- טיסת חזור -->
                <div class="card">
                    <div class="card-header">
                        <h3>✈️ טיסת חזור</h3>
                        <?php
                        $flightRetAdditions = $additions['flight_return'];
                        $flightRetAdditionsTotal = array_sum(array_column($flightRetAdditions, 'amount'));
                        $flightRetPaidAdditions = array_sum(array_map(function($a) { return $a['is_paid'] ? $a['amount'] : 0; }, $flightRetAdditions));
                        $flightRetUnpaidAdditions = $flightRetAdditionsTotal - $flightRetPaidAdditions;
                        $flightRetTotalCost = $returnFlightCost + $flightRetAdditionsTotal;
                        $basePaidRet = ($order['return_flight_supplier_paid'] ?? 'unpaid') === 'paid';
                        $allPaidRet = $basePaidRet && ($flightRetUnpaidAdditions == 0);
                        ?>
                        <span class="supplier-status <?= $allPaidRet ? 'paid' : 'unpaid' ?>">
                            <?= $allPaidRet ? '✓ שולם' : '⏳ לא שולם' ?>
                        </span>
                    </div>
                    <div class="card-body">
                        <div class="data-row">
                            <span class="data-label">ספק</span>
                            <span class="data-value editable" data-field="return_flight_supplier" data-type="text" data-value="<?= htmlspecialchars($order['return_flight_supplier'] ?? '') ?>" onclick="startInlineEdit(this)"><?= htmlspecialchars($order['return_flight_supplier'] ?? $order['flight_supplier'] ?? '-') ?></span>
                        </div>
                        <div class="data-row">
                            <span class="data-label">חברת תעופה</span>
                            <span class="data-value editable" data-field="airline_return" data-type="text" data-value="<?= htmlspecialchars($order['airline_return'] ?? '') ?>" onclick="startInlineEdit(this)"><?= htmlspecialchars($order['airline_return'] ?? $order['airline_outbound'] ?? '-') ?></span>
                        </div>
                        <div class="data-row">
                            <span class="data-label">קוד הזמנה</span>
                            <span class="data-value highlight editable" data-field="booking_code_return" data-type="text" data-value="<?= htmlspecialchars($order['booking_code_return'] ?? '') ?>" onclick="startInlineEdit(this)"><?= htmlspecialchars($order['booking_code_return'] ?? '-') ?></span>
                        </div>
                        <div class="data-row">
                            <span class="data-label">עלות בסיס</span>
                            <span class="data-value">
                                <?php 
                                $symRet = $currencySymbols[$order['return_flight_supplier_currency'] ?? 'USD'] ?? '$';
                                ?>
                                <span class="editable" data-field="return_flight_supplier_cost" data-type="number" data-value="<?= $returnFlightCost ?>" onclick="startInlineEdit(this)"><?= $symRet . number_format($returnFlightCost, 2) ?></span>
                                <span class="paid-badge editable <?= $basePaidRet ? 'paid' : 'unpaid' ?>" data-field="return_flight_supplier_paid" data-type="paid" data-value="<?= $basePaidRet ? 'paid' : 'unpaid' ?>" onclick="startInlineEdit(this)"><?= $basePaidRet ? '✓ שולם' : '⏳ לא שולם' ?></span>
                            </span>
                        </div>
                        
                        <?php if (!empty($flightRetAdditions)): ?>
                        <div class="additions-section">
                            <div class="additions-title">תוספות:</div>
                            <?php foreach ($flightRetAdditions as $add): ?>
                            <div class="addition-row">
                                <div class="addition-info">
                                    <span class="addition-desc">+ <?= htmlspecialchars($add['description']) ?></span>
                                    <span class="addition-date"><?= date('d/m/Y', strtotime($add['created_at'])) ?></span>
                                </div>
                                <span class="addition-amount">
                                    לספק: <?= $currencySymbols[$add['currency']] ?? '$' ?><?= number_format($add['amount'], 2) ?>
                                    <?php if ($add['customer_charge'] > 0): ?>
                                    | מלקוח: ₪<?= number_format($add['customer_charge'], 2) ?>
                                    <?php if (!empty($add['customer_paid'])): ?>
                                        <span class="paid-badge paid">💳 שולם</span>
                                    <?php else: ?>
                                        <button class="btn-charge-customer" onclick="chargeCustomerForAddition(<?= $add['id'] ?>, <?= $add['customer_charge'] ?>, '<?= htmlspecialchars($add['description']) ?>')" title="חייב לקוח">💳 לחיוב</button>
                                    <?php endif; ?>
                                    | רווח: <span style="color:<?= $add['profit'] >= 0 ? '#28a745' : '#dc3545' ?>">₪<?= number_format($add['profit'], 2) ?></span>
                                    <?php endif; ?>
                                    <span class="paid-badge <?= $add['is_paid'] ? 'paid' : 'unpaid' ?>" onclick="toggleAdditionPaid(<?= $add['id'] ?>, <?= $add['is_paid'] ? 0 : 1 ?>)" style="cursor:pointer;" title="לספק: לחץ לשינוי"><?= $add['is_paid'] ? '✓ שולם לספק' : '⏳ לא שולם לספק' ?></span>
                                    <button class="btn-delete-addition" onclick="deleteAddition(<?= $add['id'] ?>)" title="מחק">×</button>
                                </span>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <?php endif; ?>
                        
                        <?php if ($flightRetAdditionsTotal > 0): ?>
                        <div class="data-row total-row">
                            <span class="data-label"><strong>סה"כ רכיב</strong></span>
                            <span class="data-value"><strong><?= $symRet ?><?= number_format($flightRetTotalCost, 2) ?></strong></span>
                        </div>
                        <?php endif; ?>
                        
                        <button class="btn-add-addition" onclick="openAdditionModal('flight_return', 'טיסת חזור', '<?= $order['return_flight_supplier_currency'] ?? 'USD' ?>')">
                            + הוסף תוספת
                        </button>
                    </div>
                </div>
                
                <!-- מלונות -->
                <div class="card">
                    <div class="card-header">
                        <h3>🏨 מלונות</h3>
                        <?php
                        $hotelAdditions = $additions['hotel'];
                        $hotelAdditionsTotal = array_sum(array_column($hotelAdditions, 'amount'));
                        $hotelPaidAdditions = array_sum(array_map(function($a) { return $a['is_paid'] ? $a['amount'] : 0; }, $hotelAdditions));
                        $hotelUnpaidAdditions = $hotelAdditionsTotal - $hotelPaidAdditions;
                        
                        // בדיקה אם כל המלונות שולמו
                        $allHotelsPaid = true;
                        if (!empty($orderHotels)) {
                            foreach ($orderHotels as $hotel) {
                                if (($hotel['supplier_paid'] ?? 'unpaid') !== 'paid') {
                                    $allHotelsPaid = false;
                                    break;
                                }
                            }
                        } else {
                            $allHotelsPaid = ($order['hotel_supplier_paid'] ?? 'unpaid') === 'paid';
                        }
                        $allPaidHotel = $allHotelsPaid && ($hotelUnpaidAdditions == 0);
                        ?>
                        <span class="supplier-status <?= $allPaidHotel ? 'paid' : 'unpaid' ?>">
                            <?= $allPaidHotel ? '✓ שולם' : '⏳ לא שולם' ?>
                        </span>
                    </div>
                    <div class="card-body">
                        <?php if (!empty($orderHotels)): ?>
                            <?php foreach ($orderHotels as $hotelIndex => $hotel): ?>
                            <div class="hotel-item" style="<?= $hotelIndex > 0 ? 'margin-top: 15px; padding-top: 15px; border-top: 1px dashed #ddd;' : '' ?>">
                                <?php if (count($orderHotels) > 1): ?>
                                <div style="font-weight: bold; color: #667eea; margin-bottom: 10px;">מלון #<?= $hotelIndex + 1 ?></div>
                                <?php endif; ?>
                                <div class="data-row">
                                    <span class="data-label">ספק</span>
                                    <span class="data-value editable" data-field="supplier" data-type="select" data-table="order_hotels" data-row-id="<?= $hotel['id'] ?>" data-value="<?= htmlspecialchars($hotel['supplier'] ?? 'DIRECT') ?>" data-options="DIRECT,ORANGE,YELLOW,BLUE" onclick="startInlineEdit(this)"><?= htmlspecialchars($hotel['supplier'] ?? 'DIRECT') ?></span>
                                </div>
                                <div class="data-row">
                                    <span class="data-label">קוד הזמנה</span>
                                    <span class="data-value highlight editable" data-field="confirmation_code" data-type="text" data-table="order_hotels" data-row-id="<?= $hotel['id'] ?>" data-value="<?= htmlspecialchars($hotel['confirmation_code'] ?? '') ?>" onclick="startInlineEdit(this)"><?= htmlspecialchars($hotel['confirmation_code'] ?? '-') ?></span>
                                </div>
                                <div class="data-row">
                                    <span class="data-label">עלות</span>
                                    <span class="data-value">
                                        <?php $symH = $currencySymbols[$hotel['supplier_currency'] ?? 'USD'] ?? '$'; ?>
                                        <span class="editable" data-field="supplier_cost" data-type="number" data-table="order_hotels" data-row-id="<?= $hotel['id'] ?>" data-value="<?= $hotel['supplier_cost'] ?? 0 ?>" onclick="startInlineEdit(this)"><?= $symH . number_format($hotel['supplier_cost'] ?? 0, 2) ?></span>
                                        <?php $hotelPaid = ($hotel['supplier_paid'] ?? 'unpaid') === 'paid'; ?>
                                        <span class="paid-badge editable <?= $hotelPaid ? 'paid' : 'unpaid' ?>" data-field="supplier_paid" data-type="paid" data-table="order_hotels" data-row-id="<?= $hotel['id'] ?>" data-value="<?= $hotelPaid ? 'paid' : 'unpaid' ?>" onclick="startInlineEdit(this)"><?= $hotelPaid ? '✓ שולם' : '⏳ לא שולם' ?></span>
                                    </span>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <!-- fallback לנתונים מטבלת orders -->
                            <?php 
                            $basePaidHotel = ($order['hotel_supplier_paid'] ?? 'unpaid') === 'paid';
                            $symHotel = $currencySymbols[$order['hotel_supplier_currency'] ?? 'USD'] ?? '$';
                            ?>
                            <div class="data-row">
                                <span class="data-label">ספק</span>
                                <span class="data-value editable" data-field="hotel_supplier" data-type="text" data-value="<?= htmlspecialchars($order['hotel_supplier'] ?? '') ?>" onclick="startInlineEdit(this)"><?= htmlspecialchars($order['hotel_supplier'] ?? '-') ?></span>
                            </div>
                            <div class="data-row">
                                <span class="data-label">קוד הזמנה</span>
                                <span class="data-value highlight editable" data-field="hotel_supplier_booking" data-type="text" data-value="<?= htmlspecialchars($order['hotel_supplier_booking'] ?? $order['hotel_booking_code'] ?? '') ?>" onclick="startInlineEdit(this)"><?= htmlspecialchars($order['hotel_supplier_booking'] ?? $order['hotel_booking_code'] ?? $order['hotel_confirmation'] ?? '-') ?></span>
                            </div>
                            <div class="data-row">
                                <span class="data-label">עלות בסיס</span>
                                <span class="data-value">
                                    <span class="editable" data-field="hotel_supplier_cost" data-type="number" data-value="<?= $hotelCost ?>" onclick="startInlineEdit(this)"><?= $symHotel . number_format($hotelCost, 2) ?></span>
                                    <span class="paid-badge editable <?= $basePaidHotel ? 'paid' : 'unpaid' ?>" data-field="hotel_supplier_paid" data-type="paid" data-value="<?= $basePaidHotel ? 'paid' : 'unpaid' ?>" onclick="startInlineEdit(this)"><?= $basePaidHotel ? '✓ שולם' : '⏳ לא שולם' ?></span>
                                </span>
                            </div>
                        <?php endif; ?>
                        
                        <?php if (!empty($hotelAdditions)): ?>
                        <div class="additions-section">
                            <div class="additions-title">תוספות:</div>
                            <?php foreach ($hotelAdditions as $add): ?>
                            <div class="addition-row">
                                <div class="addition-info">
                                    <span class="addition-desc">+ <?= htmlspecialchars($add['description']) ?></span>
                                    <span class="addition-date"><?= date('d/m/Y', strtotime($add['created_at'])) ?></span>
                                </div>
                                <span class="addition-amount">
                                    לספק: <?= $currencySymbols[$add['currency']] ?? '$' ?><?= number_format($add['amount'], 2) ?>
                                    <?php if ($add['customer_charge'] > 0): ?>
                                    | מלקוח: ₪<?= number_format($add['customer_charge'], 2) ?>
                                    <?php if (!empty($add['customer_paid'])): ?>
                                        <span class="paid-badge paid">💳 שולם</span>
                                    <?php else: ?>
                                        <button class="btn-charge-customer" onclick="chargeCustomerForAddition(<?= $add['id'] ?>, <?= $add['customer_charge'] ?>, '<?= htmlspecialchars($add['description']) ?>')" title="חייב לקוח">💳 לחיוב</button>
                                    <?php endif; ?>
                                    | רווח: <span style="color:<?= $add['profit'] >= 0 ? '#28a745' : '#dc3545' ?>">₪<?= number_format($add['profit'], 2) ?></span>
                                    <?php endif; ?>
                                    <span class="paid-badge <?= $add['is_paid'] ? 'paid' : 'unpaid' ?>" onclick="toggleAdditionPaid(<?= $add['id'] ?>, <?= $add['is_paid'] ? 0 : 1 ?>)" style="cursor:pointer;" title="לספק: לחץ לשינוי"><?= $add['is_paid'] ? '✓ שולם לספק' : '⏳ לא שולם לספק' ?></span>
                                    <button class="btn-delete-addition" onclick="deleteAddition(<?= $add['id'] ?>)" title="מחק">×</button>
                                </span>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <?php endif; ?>
                        
                        <button class="btn-add-addition" onclick="openAdditionModal('hotel', 'מלון', '<?= $order['hotel_supplier_currency'] ?? 'USD' ?>')">
                            + הוסף תוספת
                        </button>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- שורה שלישית: תשלומים + סיכום כספי -->
        <div class="section">
            <div class="grid-2">
                <!-- תשלומים מלקוחות -->
                <div class="card">
                    <div class="card-header">
                        <h3>💳 תשלומים מלקוחות</h3>
                    </div>
                    <div class="card-body">
                        <?php if (!empty($payments)): ?>
                        <table class="payments-table">
                            <thead>
                                <tr>
                                    <th>שם המשלם</th>
                                    <th>סכום</th>
                                    <th>אמצעי</th>
                                    <th>תשלומים</th>
                                    <th>4 ספרות</th>
                                    <th>תאריך</th>
                                    <th>קבלה</th>
                                    <th>סטטוס</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($payments as $p): ?>
                                <tr>
                                    <td><?= htmlspecialchars($p['name'] ?? '-') ?></td>
                                    <td><strong>₪<?= number_format($p['amount'] ?? 0) ?></strong></td>
                                    <td><?php 
                                        $m = $p['method'] ?? '';
                                        if ($m === 'bank_transfer') echo 'העברה בנקאית';
                                        elseif ($m === 'credit_card') echo 'אשראי';
                                        elseif ($m === 'cash') echo 'מזומן';
                                        elseif ($m === 'bit') echo 'ביט';
                                        else echo htmlspecialchars($m ?: '-');
                                    ?></td>
                                    <td><?= $p['installments'] ?? 1 ?></td>
                                    <td><?= !empty($p['last4']) ? htmlspecialchars($p['last4']) : '-' ?></td>
                                    <td><?= !empty($p['date']) ? date('d/m/Y', strtotime($p['date'])) : '-' ?></td>
                                    <td>
                                        <?php if (!empty($p['receipt_number'])): ?>
                                            <?php if (!empty($p['receipt_pdf'])): ?>
                                                <a href="<?= htmlspecialchars($p['receipt_pdf']) ?>" target="_blank" style="color:#667eea; text-decoration:none;">
                                                    <?= htmlspecialchars($p['receipt_number']) ?> 📄
                                                </a>
                                            <?php else: ?>
                                                <?= htmlspecialchars($p['receipt_number']) ?>
                                            <?php endif; ?>
                                        <?php else: ?>
                                            -
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php 
                                        $method = $p['method'] ?? '';
                                        if ($method === 'bank_transfer') {
                                            // חיפוש סטטוס ההעברה לפי ID ישיר (קשר 1:1)
                                            $transferStatus = null;
                                            $bankTransferId = $p['bank_transfer_id'] ?? null;
                                            
                                            if ($bankTransferId) {
                                                // חיפוש לפי ID ישיר - הדרך הנכונה
                                                foreach ($bankTransfers as $bt) {
                                                    if ($bt['id'] == $bankTransferId) {
                                                        $transferStatus = $bt['status'];
                                                        break;
                                                    }
                                                }
                                            }
                                            
                                            // אם אין ID (הזמנות ישנות), fallback לחיפוש לפי סכום
                                            if ($transferStatus === null) {
                                                $paymentAmount = floatval($p['amount']);
                                                foreach ($bankTransfers as $bt) {
                                                    if (abs(floatval($bt['amount']) - $paymentAmount) < 1) {
                                                        $transferStatus = $bt['status'];
                                                        break;
                                                    }
                                                }
                                            }
                                            
                                            if ($transferStatus === 'approved') {
                                                echo '<span style="color:#28a745; font-weight:bold;">✅ אומת</span>';
                                            } elseif ($transferStatus === 'rejected') {
                                                echo '<span style="color:#dc3545; font-weight:bold;">❌ לא התקבל</span>';
                                            } else {
                                                // pending או null - טרם אומת
                                                echo '<span style="color:#ffc107; font-weight:bold;">⏳ טרם אומת</span>';
                                            }
                                        } else {
                                            echo '<span style="color:#28a745;">✓</span>';
                                        }
                                        ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                                
                                <?php foreach ($additionPayments as $ap): ?>
                                <tr style="background: #f0f7ff;">
                                    <td>
                                        <?= htmlspecialchars($ap['payer_name'] ?? '-') ?>
                                        <br><small style="color:#666;">📦 <?= htmlspecialchars($ap['addon_description'] ?? 'תוספת') ?></small>
                                    </td>
                                    <td><strong>₪<?= number_format($ap['amount'] ?? 0) ?></strong></td>
                                    <td><?php
                                        $method = $ap['payment_method'] ?? $ap['draft_payment_method'] ?? 'credit_card';
                                        if ($method === 'bank_transfer') echo 'העברה בנקאית';
                                        elseif ($method === 'bit') echo 'ביט';
                                        elseif ($method === 'cash') echo 'מזומן';
                                        else echo 'אשראי';
                                    ?></td>
                                    <td><?= $ap['installments'] ?? 1 ?></td>
                                    <td><?= !empty($ap['last_4_digits']) ? htmlspecialchars($ap['last_4_digits']) : '-' ?></td>
                                    <td><?= !empty($ap['payment_date']) ? date('d/m/Y', strtotime($ap['payment_date'])) : '-' ?></td>
                                    <td>
                                        <?php if (!empty($ap['receipt_number'])): ?>
                                            <?php if (!empty($ap['receipt_pdf'])): ?>
                                                <a href="<?= htmlspecialchars($ap['receipt_pdf']) ?>" target="_blank" style="color:#667eea; text-decoration:none;">
                                                    <?= htmlspecialchars($ap['receipt_number']) ?> 📄
                                                </a>
                                            <?php else: ?>
                                                <?= htmlspecialchars($ap['receipt_number']) ?>
                                            <?php endif; ?>
                                        <?php else: ?>
                                            -
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php 
                                        if ($method === 'bank_transfer') {
                                            $transferStatus = null;
                                            $btId = $ap['bank_transfer_id'] ?? null;
                                            $draftId = $ap['draft_id'] ?? null;
                                            
                                            if ($btId) {
                                                foreach ($bankTransfers as $bt) {
                                                    if ($bt['id'] == $btId) { $transferStatus = $bt['status']; break; }
                                                }
                                            }
                                            if ($transferStatus === null && $draftId) {
                                                foreach ($bankTransfers as $bt) {
                                                    if ($bt['draft_id'] == $draftId) { $transferStatus = $bt['status']; break; }
                                                }
                                            }
                                            
                                            if ($transferStatus === 'approved') echo '<span style="color:#28a745; font-weight:bold;">✅ אומת</span>';
                                            elseif ($transferStatus === 'rejected') echo '<span style="color:#dc3545; font-weight:bold;">❌ לא התקבל</span>';
                                            else echo '<span style="color:#ffc107; font-weight:bold;">⏳ טרם אומת</span>';
                                        } else {
                                            echo '<span style="color:#28a745;">✓</span>';
                                        }
                                        ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                        <?php else: ?>
                        
                        <?php if (!empty($additionPayments)): ?>
                        <table class="payments-table">
                            <thead>
                                <tr>
                                    <th>שם המשלם</th>
                                    <th>סכום</th>
                                    <th>אמצעי</th>
                                    <th>תשלומים</th>
                                    <th>4 ספרות</th>
                                    <th>תאריך</th>
                                    <th>קבלה</th>
                                    <th>סטטוס</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($additionPayments as $ap): ?>
                                <tr style="background: #f0f7ff;">
                                    <td>
                                        <?= htmlspecialchars($ap['payer_name'] ?? '-') ?>
                                        <br><small style="color:#666;">📦 <?= htmlspecialchars($ap['addon_description'] ?? 'תוספת') ?></small>
                                    </td>
                                    <td><strong>₪<?= number_format($ap['amount'] ?? 0) ?></strong></td>
                                    <td><?php
                                        $method = $ap['payment_method'] ?? $ap['draft_payment_method'] ?? 'credit_card';
                                        if ($method === 'bank_transfer') echo 'העברה בנקאית';
                                        elseif ($method === 'bit') echo 'ביט';
                                        elseif ($method === 'cash') echo 'מזומן';
                                        else echo 'אשראי';
                                    ?></td>
                                    <td><?= $ap['installments'] ?? 1 ?></td>
                                    <td><?= !empty($ap['last_4_digits']) ? htmlspecialchars($ap['last_4_digits']) : '-' ?></td>
                                    <td><?= !empty($ap['payment_date']) ? date('d/m/Y', strtotime($ap['payment_date'])) : '-' ?></td>
                                    <td>
                                        <?php if (!empty($ap['receipt_number'])): ?>
                                            <?php if (!empty($ap['receipt_pdf'])): ?>
                                                <a href="<?= htmlspecialchars($ap['receipt_pdf']) ?>" target="_blank" style="color:#667eea; text-decoration:none;">
                                                    <?= htmlspecialchars($ap['receipt_number']) ?> 📄
                                                </a>
                                            <?php else: ?>
                                                <?= htmlspecialchars($ap['receipt_number']) ?>
                                            <?php endif; ?>
                                        <?php else: ?>
                                            -
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php 
                                        if ($method === 'bank_transfer') {
                                            $transferStatus = null;
                                            $btId = $ap['bank_transfer_id'] ?? null;
                                            $draftId = $ap['draft_id'] ?? null;
                                            
                                            if ($btId) {
                                                foreach ($bankTransfers as $bt) {
                                                    if ($bt['id'] == $btId) { $transferStatus = $bt['status']; break; }
                                                }
                                            }
                                            if ($transferStatus === null && $draftId) {
                                                foreach ($bankTransfers as $bt) {
                                                    if ($bt['draft_id'] == $draftId) { $transferStatus = $bt['status']; break; }
                                                }
                                            }
                                            
                                            if ($transferStatus === 'approved') echo '<span style="color:#28a745; font-weight:bold;">✅ אומת</span>';
                                            elseif ($transferStatus === 'rejected') echo '<span style="color:#dc3545; font-weight:bold;">❌ לא התקבל</span>';
                                            else echo '<span style="color:#ffc107; font-weight:bold;">⏳ טרם אומת</span>';
                                        } else {
                                            echo '<span style="color:#28a745;">✓</span>';
                                        }
                                        ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                        <?php else: ?>
                        <p style="color:#666;">אין תשלומים רשומים</p>
                        <?php endif; ?>
                        
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- סיכום כספי -->
                <div class="card">
                    <div class="card-header">
                        <h3>📊 סיכום כספי</h3>
                    </div>
                    <div class="card-body">
                        <div class="summary-box">
                            <div class="summary-row">
                                <span class="summary-label">סה"כ מלקוחות (הזמנה מקורית)</span>
                                <span>₪<?= number_format($totalPayments) ?></span>
                            </div>
                            <?php if ($totalAdditionPayments > 0): ?>
                            <div class="summary-row">
                                <span class="summary-label">תשלומים על תוספות</span>
                                <span>₪<?= number_format($totalAdditionPayments) ?></span>
                            </div>
                            <div class="summary-row" style="font-weight:bold; background:#2e7d32; color:white; padding:8px; border-radius:4px;">
                                <span class="summary-label">סה"כ התקבל מלקוחות</span>
                                <span>₪<?= number_format($totalPayments + $totalAdditionPayments) ?></span>
                            </div>
                            <?php endif; ?>
                            <?php 
                            // הצג "ממתין לגביה" רק אם יש הפרש
                            $pendingCollection = $totalAdditionsCustomerCharge - $totalAdditionPayments;
                            if ($pendingCollection > 0): ?>
                            <div class="summary-row" style="background:#fff3cd; padding:8px; border-radius:4px;">
                                <span class="summary-label">⏳ ממתין לגביה על תוספות</span>
                                <span>₪<?= number_format($pendingCollection) ?></span>
                            </div>
                            <?php endif; ?>
                            <div class="summary-row" style="border-top: 1px dashed #ddd; padding-top: 8px; margin-top: 8px;">
                                <span class="summary-label">עלות טיסת הלוך</span>
                                <span><?= $currencySymbols[$order['flight_supplier_currency'] ?? 'USD'] ?? '$' ?><?= number_format($flightCost, 2) ?></span>
                            </div>
                            <div class="summary-row">
                                <span class="summary-label">עלות טיסת חזור</span>
                                <span><?= $currencySymbols[$order['return_flight_supplier_currency'] ?? 'USD'] ?? '$' ?><?= number_format($returnFlightCost, 2) ?></span>
                            </div>
                            <?php if (!empty($orderHotels)): ?>
                                <?php foreach ($orderHotels as $hIdx => $hotel): ?>
                                <div class="summary-row">
                                    <span class="summary-label">עלות מלון #<?= $hIdx + 1 ?></span>
                                    <span><?= $currencySymbols[$hotel['supplier_currency'] ?? 'USD'] ?? '$' ?><?= number_format($hotel['supplier_cost'] ?? 0, 2) ?></span>
                                </div>
                                <?php endforeach; ?>
                            <?php else: ?>
                            <div class="summary-row">
                                <span class="summary-label">עלות מלון</span>
                                <span><?= $currencySymbols[$order['hotel_supplier_currency'] ?? 'USD'] ?? '$' ?><?= number_format($hotelCost, 2) ?></span>
                            </div>
                            <?php endif; ?>
                            <?php if ($extrasCostIls > 0): ?>
                            <div class="summary-row">
                                <span class="summary-label">🎫 עלות שונות (בשקלים)</span>
                                <span>₪<?= number_format($extrasCostIls) ?></span>
                            </div>
                            <?php endif; ?>
                            <?php if ($totalAdditionsCostIls > 0): ?>
                            <div class="summary-row">
                                <span class="summary-label">עלות תוספות (בשקלים)</span>
                                <span>₪<?= number_format($totalAdditionsCostIls) ?></span>
                            </div>
                            <?php endif; ?>
                            <div class="summary-row total" style="border-top: 2px solid #ddd; padding-top: 10px; margin-top: 10px;">
                                <span class="summary-label">רווח הזמנה מקורית</span>
                                <span>₪<?= number_format($order['profit'] ?? 0) ?></span>
                            </div>
                            <?php if ($totalAdditionsProfit != 0): ?>
                            <div class="summary-row">
                                <span class="summary-label">רווח מתוספות</span>
                                <span style="color: <?= $totalAdditionsProfit >= 0 ? '#28a745' : '#dc3545' ?>">₪<?= number_format($totalAdditionsProfit) ?></span>
                            </div>
                            <?php endif; ?>
                            <?php if ($totalRefundProfit != 0): ?>
                            <div class="summary-row">
                                <span class="summary-label">📥 רווח מהחזרים</span>
                                <span style="color: <?= $totalRefundProfit >= 0 ? '#28a745' : '#dc3545' ?>">₪<?= number_format($totalRefundProfit) ?></span>
                            </div>
                            <?php endif; ?>
                            <?php if ($totalRefundReturned > 0): ?>
                            <div class="summary-row" style="background:#e3f2fd; padding:8px; border-radius:4px;">
                                <span class="summary-label">💸 סה"כ הוחזר ללקוחות</span>
                                <span style="color: #1565c0;">₪<?= number_format($totalRefundReturned) ?></span>
                            </div>
                            <?php endif; ?>
                            <?php 
                            $totalProfit = ($order['profit'] ?? 0) + $totalAdditionsProfit + $totalRefundProfit;
                            if ($totalAdditionsProfit != 0 || $totalRefundProfit != 0): 
                            ?>
                            <div class="summary-row total" style="background: #2e7d32; color:white; padding: 10px; border-radius: 6px; margin-top: 8px;">
                                <span class="summary-label"><strong>רווח כולל</strong></span>
                                <span style="color: <?= $totalProfit >= 0 ? 'white' : '#ffcdd2' ?>">
                                    <strong>₪<?= number_format($totalProfit) ?></strong>
                                </span>
                            </div>
                            <?php endif; ?>
                        </div>
                        
                        <!-- שערי מטבע ביום ההזמנה -->
                        <div style="margin-top: 15px; padding: 12px; background: #f8f9fa; border-radius: 8px; font-size: 0.85rem;">
                            <strong>💱 שערי מטבע ביום ההזמנה:</strong>
                            <div style="display: flex; gap: 20px; margin-top: 8px;">
                                <span>דולר: ₪<?= number_format($order['usd_rate'] ?? 3.65, 2) ?></span>
                                <span>יורו: ₪<?= number_format($order['eur_rate'] ?? 3.95, 2) ?></span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- שונות (אם יש) -->
        <?php if (!empty($extras)): ?>
        <div class="section">
            <div class="card">
                <div class="card-header">
                    <h3>🎁 שונות / תוספות</h3>
                </div>
                <div class="card-body">
                    <table class="payments-table">
                        <thead>
                            <tr>
                                <th>תיאור</th>
                                <th>קוד הזמנה</th>
                                <th>עלות</th>
                                <th>סטטוס</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($extras as $extra): ?>
                            <tr>
                                <td><?= htmlspecialchars($extra['description'] ?? '-') ?></td>
                                <td><?= htmlspecialchars($extra['booking_code'] ?? '-') ?></td>
                                <td>
                                    <?= $currencySymbols[$extra['currency'] ?? 'USD'] ?? '$' ?>
                                    <?= number_format($extra['cost'] ?? 0, 2) ?>
                                </td>
                                <td>
                                    <span class="supplier-status <?= ($extra['paid'] ?? 'unpaid') ?>">
                                        <?= ($extra['paid'] ?? 'unpaid') === 'paid' ? '✓ שולם' : '⏳ לא שולם' ?>
                                    </span>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        <?php endif; ?>
        
        <!-- החזרים -->
        <div class="section">
            <div class="card">
                <div class="card-header" style="display: flex; justify-content: space-between; align-items: center;">
                    <h3>📥 החזרים</h3>
                    <button type="button" class="btn-add-small" onclick="openRefundModal()">+ הוסף החזר</button>
                </div>
                <div class="card-body">
                    <?php if (empty($refunds)): ?>
                        <p style="color: #999; text-align: center; padding: 20px;">אין החזרים להזמנה זו</p>
                    <?php else: ?>
                        <table class="payments-table">
                            <thead>
                                <tr>
                                    <th>תאריך</th>
                                    <th>תיאור</th>
                                    <th>ספק / חברת תעופה</th>
                                    <th>קיבלנו</th>
                                    <th>החזרנו ללקוח</th>
                                    <th>רווח</th>
                                    <th>פעולות</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($refunds as $refund): 
                                    $profit = floatval($refund['received_from_supplier']) - floatval($refund['returned_to_customer']);
                                    $sourceInfo = [];
                                    if (!empty($refund['supplier_name'])) $sourceInfo[] = $refund['supplier_name'];
                                    if (!empty($refund['airline'])) $sourceInfo[] = $refund['airline'];
                                ?>
                                <tr>
                                    <td><?= $refund['refund_date'] ? date('d/m/Y', strtotime($refund['refund_date'])) : '-' ?></td>
                                    <td>
                                        <?= htmlspecialchars($refund['description']) ?>
                                        <?php if (!empty($refund['notes'])): ?>
                                            <br><small style="color: #666;"><?= htmlspecialchars($refund['notes']) ?></small>
                                        <?php endif; ?>
                                    </td>
                                    <td><?= !empty($sourceInfo) ? htmlspecialchars(implode(' / ', $sourceInfo)) : '-' ?></td>
                                    <td style="color: #28a745; font-weight: bold;">₪<?= number_format($refund['received_from_supplier'], 2) ?></td>
                                    <td style="color: #dc3545; font-weight: bold;">₪<?= number_format($refund['returned_to_customer'], 2) ?></td>
                                    <td style="color: <?= $profit >= 0 ? '#28a745' : '#dc3545' ?>; font-weight: bold;">
                                        ₪<?= number_format($profit, 2) ?>
                                    </td>
                                    <td>
                                        <button type="button" class="btn-delete-small" onclick="deleteRefund(<?= $refund['id'] ?>)" title="מחק">🗑️</button>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                            <tfoot style="background: #f8f9fa; font-weight: bold;">
                                <tr>
                                    <td colspan="3" style="text-align: left;">סה"כ:</td>
                                    <td style="color: #28a745;">₪<?= number_format($totalRefundReceived, 2) ?></td>
                                    <td style="color: #dc3545;">₪<?= number_format($totalRefundReturned, 2) ?></td>
                                    <td style="color: <?= $totalRefundProfit >= 0 ? '#28a745' : '#dc3545' ?>;">
                                        ₪<?= number_format($totalRefundProfit, 2) ?>
                                    </td>
                                    <td></td>
                                </tr>
                            </tfoot>
                        </table>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        
        <!-- כבודה והערות -->
        <div class="section">
            <div class="grid-2">
                <div class="card">
                    <div class="card-header">
                        <h3>🧳 כבודה</h3>
                    </div>
                    <div class="card-body">
                        <div class="notes-box editable" data-field="luggage_details" data-type="textarea" data-value="<?= htmlspecialchars($order['luggage_details'] ?? '') ?>" onclick="startInlineEdit(this)"><?= $order['luggage_details'] ? nl2br(htmlspecialchars($order['luggage_details'])) : '<span style="color:#999;">לחץ להוספת פרטי כבודה...</span>' ?></div>
                    </div>
                </div>
                
                <div class="card">
                    <div class="card-header">
                        <h3>📝 הערות</h3>
                    </div>
                    <div class="card-body">
                        <div class="notes-box editable" data-field="notes" data-type="textarea" data-value="<?= htmlspecialchars($order['notes'] ?? '') ?>" onclick="startInlineEdit(this)"><?= $order['notes'] ? nl2br(htmlspecialchars($order['notes'])) : '<span style="color:#999;">לחץ להוספת הערות...</span>' ?></div>
                    </div>
                </div>
            </div>
        </div>
        
    </div>
    
    <!-- מודאל הוספת תוספת -->
    <div class="modal-overlay" id="additionModal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>➕ הוספת תוספת ל<span id="modalComponentName"></span></h3>
                <button class="modal-close" onclick="closeModal()">&times;</button>
            </div>
            <form id="additionForm">
                <input type="hidden" name="order_id" value="<?= $order['id'] ?>">
                <input type="hidden" name="component" id="modalComponent">
                <input type="hidden" name="action" value="add">
                
                <div class="form-group">
                    <label>תיאור התוספת</label>
                    <input type="text" name="description" id="additionDesc" placeholder="לדוגמה: מזוודה 23 ק״ג" required>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label>עלות לספק</label>
                        <input type="number" step="0.01" name="amount" id="additionAmount" placeholder="50.00" required oninput="calculateAdditionProfit()">
                    </div>
                    <div class="form-group">
                        <label>מטבע</label>
                        <select name="currency" id="additionCurrency" onchange="updateExchangeRate(); calculateAdditionProfit();">
                            <option value="USD">$ דולר</option>
                            <option value="EUR">€ יורו</option>
                            <option value="ILS">₪ שקל</option>
                        </select>
                    </div>
                </div>
                
                <div class="exchange-rate-info" id="exchangeRateInfo">
                    שער היום: $1 = ₪<?= number_format($currentRates['USD'], 2) ?>
                </div>
                
                <div class="form-group">
                    <label>גביה מלקוח (בשקלים)</label>
                    <input type="number" step="0.01" name="customer_charge" id="additionCustomerCharge" placeholder="200.00" oninput="calculateAdditionProfit()">
                </div>
                
                <div class="profit-preview" id="profitPreview">
                    רווח על התוספת: <span id="profitAmount">₪0</span>
                </div>
                
                <div class="form-group">
                    <div class="checkbox-group">
                        <input type="checkbox" name="is_paid" id="additionPaid" value="1">
                        <label for="additionPaid">כבר שולם לספק</label>
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn-cancel" onclick="closeModal()">ביטול</button>
                    <button type="submit" class="btn-save">💾 שמור</button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- מודאל הוספת החזר -->
    <div class="modal-overlay" id="refundModal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>📥 הוספת החזר</h3>
                <button class="modal-close" onclick="closeRefundModal()">&times;</button>
            </div>
            <form id="refundForm">
                <input type="hidden" name="order_id" value="<?= $order['id'] ?>">
                
                <div class="form-group">
                    <label>תיאור ההחזר *</label>
                    <input type="text" name="description" id="refundDesc" placeholder="לדוגמה: ביטול לילה במלון" required>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label>ספק</label>
                        <input type="text" name="supplier_name" id="refundSupplier" placeholder="לדוגמה: DIRECT, KAVEI" list="suppliersList">
                        <datalist id="suppliersList">
                            <option value="DIRECT">
                            <option value="FLYING">
                            <option value="KAVEI">
                            <option value="KISHE">
                            <option value="ESHET">
                            <option value="OR">
                            <option value="BLUE">
                            <option value="ORANGE">
                        </datalist>
                    </div>
                    <div class="form-group">
                        <label>חברת תעופה</label>
                        <input type="text" name="airline" id="refundAirline" placeholder="לדוגמה: Wizz Air, Ryanair" list="airlinesList">
                        <datalist id="airlinesList">
                            <option value="Wizz Air">
                            <option value="Ryanair">
                            <option value="EasyJet">
                            <option value="El Al">
                            <option value="Israir">
                            <option value="Turkish Airlines">
                            <option value="Aegean">
                            <option value="LOT">
                            <option value="Lufthansa">
                        </datalist>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label>קיבלנו מהספק (₪)</label>
                        <input type="number" step="0.01" name="received_from_supplier" id="refundReceived" placeholder="350.00" oninput="calculateRefundProfit()">
                    </div>
                    <div class="form-group">
                        <label>החזרנו ללקוח (₪)</label>
                        <input type="number" step="0.01" name="returned_to_customer" id="refundReturned" placeholder="300.00" oninput="calculateRefundProfit()">
                    </div>
                </div>
                
                <div class="profit-preview" id="refundProfitPreview">
                    רווח מההחזר: <span id="refundProfitAmount">₪0</span>
                </div>
                
                <div class="form-group">
                    <label>תאריך ההחזר</label>
                    <input type="date" name="refund_date" id="refundDate" value="<?= date('Y-m-d') ?>">
                </div>
                
                <div class="form-group">
                    <label>הערות (אופציונלי)</label>
                    <textarea name="notes" id="refundNotes" rows="2" placeholder="הערות פנימיות..."></textarea>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn-cancel" onclick="closeRefundModal()">ביטול</button>
                    <button type="submit" class="btn-save">💾 שמור</button>
                </div>
            </form>
        </div>
    </div>
    
    <script>
        const orderId = <?= $order['id'] ?>;
        const exchangeRates = <?= json_encode($currentRates) ?>;
        
        function openAdditionModal(component, componentName, defaultCurrency) {
            document.getElementById('modalComponent').value = component;
            document.getElementById('modalComponentName').textContent = componentName;
            document.getElementById('additionCurrency').value = defaultCurrency || 'USD';
            document.getElementById('additionDesc').value = '';
            document.getElementById('additionAmount').value = '';
            document.getElementById('additionCustomerCharge').value = '';
            document.getElementById('additionPaid').checked = false;
            document.getElementById('profitAmount').textContent = '₪0';
            document.getElementById('profitAmount').classList.remove('negative');
            updateExchangeRate();
            document.getElementById('additionModal').classList.add('active');
            document.getElementById('additionDesc').focus();
        }
        
        function closeModal() {
            document.getElementById('additionModal').classList.remove('active');
        }
        
        function updateExchangeRate() {
            const currency = document.getElementById('additionCurrency').value;
            const info = document.getElementById('exchangeRateInfo');
            if (currency === 'ILS') {
                info.textContent = '';
            } else if (currency === 'USD') {
                info.textContent = 'שער היום: $1 = ₪' + exchangeRates.USD.toFixed(2);
            } else if (currency === 'EUR') {
                info.textContent = 'שער היום: €1 = ₪' + exchangeRates.EUR.toFixed(2);
            }
        }
        
        function calculateAdditionProfit() {
            const amount = parseFloat(document.getElementById('additionAmount').value) || 0;
            const currency = document.getElementById('additionCurrency').value;
            const customerCharge = parseFloat(document.getElementById('additionCustomerCharge').value) || 0;
            
            let amountIls = amount;
            if (currency === 'USD') {
                amountIls = amount * exchangeRates.USD;
            } else if (currency === 'EUR') {
                amountIls = amount * exchangeRates.EUR;
            }
            
            const profit = customerCharge - amountIls;
            const profitEl = document.getElementById('profitAmount');
            profitEl.textContent = '₪' + profit.toFixed(2);
            
            if (profit < 0) {
                profitEl.classList.add('negative');
            } else {
                profitEl.classList.remove('negative');
            }
        }
        
        // שמירת תוספת
        document.getElementById('additionForm').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            
            try {
                const response = await fetch('api/order_additions.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    closeModal();
                    location.reload();
                } else {
                    alert('❌ שגיאה: ' + (result.error || 'לא ידוע'));
                }
            } catch (e) {
                alert('❌ שגיאה: ' + e.message);
            }
        });
        
        // מחיקת תוספת
        async function deleteAddition(additionId) {
            if (!confirm('למחוק את התוספת?')) return;
            
            try {
                const formData = new FormData();
                formData.append('action', 'delete');
                formData.append('addition_id', additionId);
                
                const response = await fetch('api/order_additions.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    location.reload();
                } else {
                    alert('❌ שגיאה: ' + (result.error || 'לא ידוע'));
                }
            } catch (e) {
                alert('❌ שגיאה: ' + e.message);
            }
        }
        
        // סגירה בלחיצה מחוץ למודאל
        document.getElementById('additionModal').addEventListener('click', function(e) {
            if (e.target === this) closeModal();
        });
        
        // סגירה ב-ESC
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') closeModal();
        });
        
        // עדכון סטטוס תשלום לספק
        async function toggleAdditionPaid(additionId, newStatus) {
            try {
                const formData = new FormData();
                formData.append('action', 'update_paid');
                formData.append('addition_id', additionId);
                formData.append('is_paid', newStatus);
                
                const response = await fetch('api/order_additions.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    location.reload();
                } else {
                    alert('❌ שגיאה: ' + (result.error || 'לא ידוע'));
                }
            } catch (e) {
                alert('❌ שגיאה: ' + e.message);
            }
        }
        
        // חיוב לקוח עבור תוספת - פותח דף טיוטה חדש
        function chargeCustomerForAddition(additionId, amount, description) {
            const customerName = '<?= htmlspecialchars($passengers[0]['first_name'] ?? '') ?> <?= htmlspecialchars($passengers[0]['last_name'] ?? '') ?>'.trim();
            const customerEmail = '<?= htmlspecialchars($order['customer_email'] ?? '') ?>';
            const customerPhone = '<?= htmlspecialchars($order['customer_phone'] ?? '') ?>';
            
            // פתיחת דף יצירת טיוטה עם הפרמטרים
            const params = new URLSearchParams({
                order_id: orderId,
                addition_id: additionId,
                amount: amount,
                description: description,
                customer_name: customerName,
                customer_email: customerEmail,
                customer_phone: customerPhone
            });
            
            window.open('create_draft_for_addition.php?' + params.toString(), '_blank');
        }
        
        // ===== INLINE EDITING =====
        
        // מערך נוסעים נוכחי
        let currentPassengers = <?= json_encode($passengers, JSON_UNESCAPED_UNICODE) ?>;
        
        // עריכת נוסע
        function startPassengerEdit(element) {
            if (element.classList.contains('editing')) return;
            
            const index = parseInt(element.dataset.index);
            const currentValue = currentPassengers[index] || '';
            
            element.classList.add('editing');
            element.dataset.originalValue = currentValue;
            
            element.innerHTML = `
                <input type="text" class="inline-input passenger-input" value="${currentValue}" style="width: 150px;">
                <span class="inline-actions" style="display: inline-flex; margin-right: 5px;">
                    <button type="button" class="inline-btn save" onclick="savePassengerEdit(this.closest('.editable-passenger'))" title="שמור">✓</button>
                    <button type="button" class="inline-btn cancel" onclick="cancelPassengerEdit(this.closest('.editable-passenger'))" title="בטל">✕</button>
                    <button type="button" class="inline-btn delete" onclick="deletePassenger(this.closest('.editable-passenger'))" title="מחק" style="background:#ff9800;">🗑</button>
                </span>
            `;
            
            const input = element.querySelector('.inline-input');
            input.focus();
            input.select();
            
            input.addEventListener('keydown', function(e) {
                if (e.key === 'Enter') {
                    e.preventDefault();
                    savePassengerEdit(element);
                } else if (e.key === 'Escape') {
                    cancelPassengerEdit(element);
                }
            });
        }
        
        // שמירת נוסע
        async function savePassengerEdit(element) {
            const index = parseInt(element.dataset.index);
            const input = element.querySelector('.inline-input');
            const newValue = input.value.trim();
            
            if (!newValue) {
                alert('שם נוסע לא יכול להיות ריק');
                return;
            }
            
            currentPassengers[index] = newValue;
            
            element.innerHTML = '<span class="inline-loading">⏳ שומר...</span>';
            
            try {
                const formData = new FormData();
                formData.append('order_id', orderId);
                formData.append('field', 'passengers');
                formData.append('value', JSON.stringify(currentPassengers));
                
                const response = await fetch('api/update_order_field.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    element.classList.remove('editing');
                    element.textContent = (index + 1) + '. ' + newValue;
                    element.classList.add('save-success');
                    setTimeout(() => element.classList.remove('save-success'), 1500);
                } else {
                    throw new Error(result.error);
                }
            } catch (error) {
                alert('❌ שגיאה: ' + error.message);
                cancelPassengerEdit(element);
            }
        }
        
        // ביטול עריכת נוסע
        function cancelPassengerEdit(element) {
            const index = parseInt(element.dataset.index);
            const originalValue = element.dataset.originalValue || currentPassengers[index] || '';
            element.classList.remove('editing');
            element.textContent = (index + 1) + '. ' + originalValue;
        }
        
        // מחיקת נוסע
        async function deletePassenger(element) {
            if (!confirm('למחוק את הנוסע?')) return;
            
            const index = parseInt(element.dataset.index);
            currentPassengers.splice(index, 1);
            
            try {
                const formData = new FormData();
                formData.append('order_id', orderId);
                formData.append('field', 'passengers');
                formData.append('value', JSON.stringify(currentPassengers));
                
                const response = await fetch('api/update_order_field.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    refreshPassengersList();
                } else {
                    throw new Error(result.error);
                }
            } catch (error) {
                alert('❌ שגיאה: ' + error.message);
            }
        }
        
        // הוספת נוסע חדש
        async function addNewPassenger() {
            const name = prompt('שם הנוסע (באנגלית):');
            if (!name || !name.trim()) return;
            
            currentPassengers.push(name.trim());
            
            try {
                const formData = new FormData();
                formData.append('order_id', orderId);
                formData.append('field', 'passengers');
                formData.append('value', JSON.stringify(currentPassengers));
                
                const response = await fetch('api/update_order_field.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    refreshPassengersList();
                } else {
                    throw new Error(result.error);
                }
            } catch (error) {
                alert('❌ שגיאה: ' + error.message);
                currentPassengers.pop();
            }
        }
        
        // רענון רשימת נוסעים
        function refreshPassengersList() {
            const list = document.getElementById('passengers-list');
            list.innerHTML = '';
            
            currentPassengers.forEach((passenger, index) => {
                const span = document.createElement('span');
                span.className = 'passenger-tag editable-passenger';
                span.dataset.index = index;
                span.onclick = function() { startPassengerEdit(this); };
                span.textContent = (index + 1) + '. ' + passenger;
                list.appendChild(span);
            });
            
            const btn = document.createElement('button');
            btn.className = 'btn-add-passenger';
            btn.onclick = addNewPassenger;
            btn.title = 'הוסף נוסע';
            btn.textContent = '+';
            list.appendChild(btn);
            
            // עדכון כותרת
            document.querySelector('.card-header h3').textContent = '👥 נוסעים (' + currentPassengers.length + ')';
        }
        
        // פתיחת עריכה
        function startInlineEdit(element) {
            if (element.classList.contains('editing')) return;
            
            const field = element.dataset.field;
            const fieldType = element.dataset.type || 'text';
            const currentValue = element.dataset.value || element.textContent.trim();
            
            element.classList.add('editing');
            element.dataset.originalValue = currentValue;
            
            let inputHtml = '';
            
            if (fieldType === 'date') {
                // המרת תאריך מ-dd/mm/yyyy ל-yyyy-mm-dd
                let dateValue = currentValue;
                if (currentValue.match(/^\d{2}\/\d{2}\/\d{4}$/)) {
                    const parts = currentValue.split('/');
                    dateValue = parts[2] + '-' + parts[1] + '-' + parts[0];
                }
                inputHtml = `<input type="date" class="inline-input" value="${dateValue}">`;
            } else if (fieldType === 'select') {
                let options = [];
                const optionsData = element.dataset.options || '[]';
                // תמיכה בפורמט פשוט (מופרד בפסיקים) ובפורמט JSON
                if (optionsData.startsWith('[')) {
                    options = JSON.parse(optionsData);
                } else {
                    // פורמט פשוט: "DIRECT,ORANGE,YELLOW,BLUE"
                    options = optionsData.split(',').map(opt => ({ value: opt.trim(), label: opt.trim() }));
                }
                inputHtml = `<select class="inline-input">`;
                options.forEach(opt => {
                    const optValue = opt.value || opt;
                    const optLabel = opt.label || opt;
                    const selected = optValue === currentValue ? 'selected' : '';
                    inputHtml += `<option value="${optValue}" ${selected}>${optLabel}</option>`;
                });
                inputHtml += `</select>`;
            } else if (fieldType === 'textarea') {
                inputHtml = `<textarea class="inline-input inline-textarea">${currentValue}</textarea>`;
            } else if (fieldType === 'number') {
                inputHtml = `<input type="number" step="0.01" class="inline-input" value="${currentValue.replace(/[^\d.-]/g, '')}">`;
            } else if (fieldType === 'paid') {
                const isPaid = currentValue.includes('שולם') || currentValue.includes('✓');
                inputHtml = `<select class="inline-input">
                    <option value="paid" ${isPaid ? 'selected' : ''}>✓ שולם</option>
                    <option value="unpaid" ${!isPaid ? 'selected' : ''}>⏳ לא שולם</option>
                </select>`;
            } else {
                inputHtml = `<input type="text" class="inline-input" value="${currentValue}">`;
            }
            
            inputHtml += `
                <span class="inline-actions">
                    <button type="button" class="inline-btn save" onclick="saveInlineEdit(this.closest('.editable'))" title="שמור">✓</button>
                    <button type="button" class="inline-btn cancel" onclick="cancelInlineEdit(this.closest('.editable'))" title="בטל">✕</button>
                </span>
            `;
            
            element.innerHTML = inputHtml;
            element.querySelector('.inline-input').focus();
            element.querySelector('.inline-input').select();
            
            // Enter לשמירה, Escape לביטול
            element.querySelector('.inline-input').addEventListener('keydown', function(e) {
                if (e.key === 'Enter' && fieldType !== 'textarea') {
                    e.preventDefault();
                    saveInlineEdit(element);
                } else if (e.key === 'Escape') {
                    cancelInlineEdit(element);
                }
            });
        }
        
        // שמירת עריכה
        async function saveInlineEdit(element) {
            const field = element.dataset.field;
            const fieldType = element.dataset.type || 'text';
            const table = element.dataset.table || 'orders';
            const rowId = element.dataset.rowId || null;
            const input = element.querySelector('.inline-input');
            const newValue = input.value;
            const originalValue = element.dataset.originalValue;
            
            if (newValue === originalValue) {
                cancelInlineEdit(element);
                return;
            }
            
            // הצג טעינה
            element.innerHTML = '<span class="inline-loading">⏳ שומר...</span>';
            
            try {
                const formData = new FormData();
                formData.append('order_id', orderId);
                formData.append('field', field);
                formData.append('value', newValue);
                formData.append('table', table);
                if (rowId) {
                    formData.append('row_id', rowId);
                }
                
                const response = await fetch('api/update_order_field.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    // עדכון התצוגה
                    let displayValue = result.display_value || newValue;
                    
                    // עיצוב מיוחד לסטטוס תשלום
                    if (fieldType === 'paid') {
                        const isPaid = newValue === 'paid';
                        displayValue = isPaid ? '✓ שולם' : '⏳ לא שולם';
                        element.className = element.className.replace(/paid|unpaid/g, '').trim();
                        element.classList.add('editable', isPaid ? 'paid' : 'unpaid');
                    }
                    
                    // עיצוב מיוחד לעלות - הוסף סימן מטבע
                    if (fieldType === 'number' && field.includes('cost')) {
                        const currency = element.closest('.data-value')?.querySelector('span')?.textContent?.charAt(0) || '$';
                        displayValue = currency + displayValue;
                    }
                    
                    element.textContent = displayValue;
                    element.dataset.value = newValue;
                    element.classList.remove('editing');
                    
                    // אפקט הצלחה
                    element.classList.add('save-success');
                    setTimeout(() => element.classList.remove('save-success'), 1500);
                    
                    // רענון העמוד אם זה שדה מטבע (לעדכון הסיכום)
                    if (field.includes('cost')) {
                        setTimeout(() => location.reload(), 1000);
                    }
                } else {
                    throw new Error(result.error);
                }
            } catch (error) {
                alert('❌ שגיאה בשמירה: ' + error.message);
                cancelInlineEdit(element);
            }
        }
        
        // ביטול עריכה - סוגר את השדה
        function cancelInlineEdit(element) {
            const originalValue = element.dataset.originalValue || element.dataset.value || '';
            const fieldType = element.dataset.type || 'text';
            
            element.classList.remove('editing');
            
            if (fieldType === 'paid') {
                const isPaid = originalValue === 'paid' || originalValue.includes('שולם');
                element.textContent = isPaid ? '✓ שולם' : '⏳ לא שולם';
            } else if (fieldType === 'textarea') {
                element.innerHTML = originalValue ? originalValue.replace(/\n/g, '<br>') : '<span style="color:#999;">לחץ להוספה...</span>';
            } else {
                element.textContent = originalValue || '-';
            }
        }
        
        // ===== פונקציות החזרים =====
        
        function openRefundModal() {
            document.getElementById('refundDesc').value = '';
            document.getElementById('refundSupplier').value = '';
            document.getElementById('refundAirline').value = '';
            document.getElementById('refundReceived').value = '';
            document.getElementById('refundReturned').value = '';
            document.getElementById('refundDate').value = new Date().toISOString().split('T')[0];
            document.getElementById('refundNotes').value = '';
            document.getElementById('refundProfitAmount').textContent = '₪0';
            document.getElementById('refundProfitAmount').classList.remove('negative');
            document.getElementById('refundModal').classList.add('active');
        }
        
        function closeRefundModal() {
            document.getElementById('refundModal').classList.remove('active');
        }
        
        function calculateRefundProfit() {
            const received = parseFloat(document.getElementById('refundReceived').value) || 0;
            const returned = parseFloat(document.getElementById('refundReturned').value) || 0;
            const profit = received - returned;
            
            const profitElement = document.getElementById('refundProfitAmount');
            profitElement.textContent = '₪' + profit.toFixed(2);
            
            if (profit < 0) {
                profitElement.classList.add('negative');
            } else {
                profitElement.classList.remove('negative');
            }
        }
        
        document.getElementById('refundForm').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const formData = {
                order_id: orderId,
                description: document.getElementById('refundDesc').value,
                supplier_name: document.getElementById('refundSupplier').value,
                airline: document.getElementById('refundAirline').value,
                received_from_supplier: parseFloat(document.getElementById('refundReceived').value) || 0,
                returned_to_customer: parseFloat(document.getElementById('refundReturned').value) || 0,
                refund_date: document.getElementById('refundDate').value,
                notes: document.getElementById('refundNotes').value
            };
            
            try {
                const response = await fetch('api/order_refunds.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify(formData)
                });
                
                const result = await response.json();
                
                if (result.success) {
                    alert('ההחזר נוסף בהצלחה!');
                    location.reload();
                } else {
                    alert('שגיאה: ' + result.error);
                }
            } catch (error) {
                alert('שגיאה בשמירת ההחזר');
                console.error(error);
            }
        });
        
        async function deleteRefund(refundId) {
            if (!confirm('האם למחוק החזר זה?')) return;
            
            try {
                const response = await fetch('api/order_refunds.php?refund_id=' + refundId, {
                    method: 'DELETE'
                });
                
                const result = await response.json();
                
                if (result.success) {
                    alert('ההחזר נמחק בהצלחה');
                    location.reload();
                } else {
                    alert('שגיאה: ' + result.error);
                }
            } catch (error) {
                alert('שגיאה במחיקת ההחזר');
                console.error(error);
            }
        }
        
        // סגירת מודאל החזרים בלחיצה על הרקע
        document.getElementById('refundModal').addEventListener('click', function(e) {
            if (e.target === this) closeRefundModal();
        });
    </script>
    
    <style>
        /* Inline Editing Styles */
        .editable {
            cursor: pointer;
            padding: 2px 6px;
            border-radius: 4px;
            transition: all 0.2s;
            border: 1px solid transparent;
        }
        .editable:hover {
            background: #e3f2fd;
            border-color: #90caf9;
        }
        .editable.editing {
            background: white;
            padding: 0;
        }
        .editable.editing:hover {
            background: white;
        }
        .inline-input {
            padding: 6px 10px;
            border: 2px solid #667eea;
            border-radius: 6px;
            font-size: inherit;
            font-family: inherit;
            width: 100%;
            min-width: 120px;
            box-sizing: border-box;
        }
        .inline-input:focus {
            outline: none;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.2);
        }
        .inline-textarea {
            min-height: 60px;
            resize: vertical;
        }
        .inline-actions {
            display: flex;
            gap: 5px;
            margin-top: 5px;
        }
        .inline-btn {
            width: 32px;
            height: 32px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 1rem;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.2s;
        }
        .inline-btn.save {
            background: #4caf50;
            color: white;
        }
        .inline-btn.save:hover {
            background: #43a047;
        }
        .inline-btn.cancel {
            background: #f44336;
            color: white;
        }
        .inline-btn.cancel:hover {
            background: #e53935;
        }
        .inline-loading {
            color: #667eea;
            font-size: 0.9rem;
        }
        .save-success {
            animation: successFlash 1.5s;
        }
        @keyframes successFlash {
            0%, 100% { background: transparent; }
            50% { background: #c8e6c9; }
        }
        
        /* עריכת נוסעים */
        .passenger-tag.editable-passenger {
            cursor: pointer;
            transition: all 0.2s;
        }
        .passenger-tag.editable-passenger:hover {
            background: #bbdefb;
            transform: scale(1.02);
        }
        .passenger-tag.editing {
            background: white;
            padding: 4px 8px;
        }
        .btn-add-passenger {
            background: #e8f5e9;
            color: #4caf50;
            border: 2px dashed #4caf50;
            padding: 6px 14px;
            border-radius: 20px;
            cursor: pointer;
            font-size: 1rem;
            font-weight: bold;
            transition: all 0.2s;
        }
        .btn-add-passenger:hover {
            background: #4caf50;
            color: white;
        }
        .inline-btn.delete {
            background: #ff9800;
        }
        .inline-btn.delete:hover {
            background: #f57c00;
        }
    </style>
</body>
</html>