<?php
// Voucher - PDF (English version)
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once 'includes/config.php';

$token = $_GET['token'] ?? null;

if (!$token) {
    die('Token not found');
}

try {
    $pdo = getDB();
    $stmt = $pdo->prepare("SELECT o.*, a.full_name as agent_name FROM orders o JOIN agents a ON o.agent_id = a.id WHERE o.draft_token = ?");
    $stmt->execute([$token]);
    $order = $stmt->fetch();
} catch (Exception $e) {
    die('Database error: ' . $e->getMessage());
}

if (!$order) {
    die('Order not found');
}

$passengers = json_decode($order['passengers'] ?? '[]', true) ?: [];

function cleanText($text) {
    if (!$text) return '-';
    $text = str_replace(['&#039;', '&apos;', "'", "'", "'", "׳"], "'", $text);
    return $text;
}

function formatEnglishDate($date) {
    if (!$date || $date === '0000-00-00') return '-';
    try {
        $d = new DateTime($date);
        return $d->format('M d, Y');
    } catch (Exception $e) {
        return '-';
    }
}

function formatTime($time) {
    if (!$time || $time === '00:00:00' || $time === '00:00') return '';
    return substr($time, 0, 5);
}

$mealBasisEnglish = [
    'ללא ארוחות' => 'Room Only',
    'לינה וארוחת בוקר' => 'Bed & Breakfast',
    'חצי פנסיון' => 'Half Board',
    'פנסיון מלא' => 'Full Board',
    'הכל כלול' => 'All Inclusive',
    'RO' => 'Room Only',
    'BB' => 'Bed & Breakfast',
    'HB' => 'Half Board',
    'FB' => 'Full Board',
    'AI' => 'All Inclusive'
];

// המרת פרטי מזוודות לאנגלית
function formatLuggageEnglish($luggageData, $passengers) {
    if (!$luggageData) return '-';
    
    // נקה escape characters ו-HTML entities
    $luggageData = stripslashes($luggageData);
    $luggageData = html_entity_decode($luggageData, ENT_QUOTES, 'UTF-8');
    $luggageData = trim($luggageData);
    
    // החלף כל סוגי הגרשיים לגרשיים רגילים
    $luggageData = str_replace(['"', '"', '״', '&quot;'], '"', $luggageData);
    
    // הסר BOM ותווים לא נראים
    $luggageData = preg_replace('/[\x00-\x1F\x7F\xEF\xBB\xBF]/u', '', $luggageData);
    
    // נסה לפרסר כ-JSON
    $luggage = json_decode($luggageData, true);
    
    if (!$luggage || !is_array($luggage)) {
        // אם עדיין לא עובד, זה טקסט רגיל - תרגם אותו
        $text = $luggageData;
        $translations = [
            'לכל נוסע מגיעה מזוודה עד' => 'Each passenger: suitcase up to',
            'וטרולי עד' => 'and trolley up to',
            'קילו' => 'kg',
            'ק"ג' => 'kg'
        ];
        foreach ($translations as $heb => $eng) {
            $text = str_replace($heb, $eng, $text);
        }
        return $text;
    }
    
    // Helper function to build luggage text
    $buildLuggageText = function($data) {
        $items = [];
        if (!empty($data['backpack'])) $items[] = 'Backpack';
        if (!empty($data['trolley'])) $items[] = 'Trolley ' . $data['trolley'] . 'kg';
        if (!empty($data['suitcase'])) $items[] = 'Suitcase ' . $data['suitcase'] . 'kg';
        return implode(' + ', $items);
    };
    
    // זה JSON - בנה טקסט באנגלית לכל נוסע
    $lines = [];
    foreach ($luggage as $p) {
        $name = $p['name'] ?? 'Passenger';
        
        // בדיקה אם יש כבודה נפרדת להלוך וחזור
        if (isset($p['same_both_ways']) && $p['same_both_ways'] === false) {
            // כבודה שונה להלוך וחזור
            $outText = isset($p['outbound']) ? $buildLuggageText($p['outbound']) : 'None';
            $retText = isset($p['return']) ? $buildLuggageText($p['return']) : 'None';
            
            if ($outText === $retText) {
                // למרות שמסומן נפרד, הכבודה זהה
                $lines[] = $name . ': ' . ($outText ?: 'No luggage');
            } else {
                // כבודה שונה באמת
                $lines[] = $name . ':';
                $lines[] = '&nbsp;&nbsp;✈️ Outbound: ' . ($outText ?: 'No luggage');
                $lines[] = '&nbsp;&nbsp;✈️ Return: ' . ($retText ?: 'No luggage');
            }
        } else {
            // כבודה זהה לשני הכיוונים (פורמט ישן או same_both_ways = true)
            $items = [];
            if (!empty($p['backpack'])) $items[] = 'Backpack';
            if (!empty($p['trolley'])) $items[] = 'Trolley ' . $p['trolley'] . 'kg';
            if (!empty($p['suitcase'])) $items[] = 'Suitcase ' . $p['suitcase'] . 'kg';
            
            if (!empty($items)) {
                $lines[] = $name . ': ' . implode(' + ', $items);
            }
        }
    }
    
    return implode('<br>', $lines) ?: '-';
}

$destination = cleanText($order['destination'] ?? 'Unknown');
$timeOut = formatTime($order['flight_time_outbound'] ?? '');
$timeReturn = formatTime($order['flight_time_return'] ?? '');
$luggage = formatLuggageEnglish($order['luggage_details'] ?? '', $passengers);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Voucher - <?= htmlspecialchars($destination) ?></title>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/html2pdf.js/0.10.1/html2pdf.bundle.min.js"></script>
    <link href="https://fonts.googleapis.com/css2?family=Open+Sans:wght@400;600;700&display=swap" rel="stylesheet">
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        body {
            font-family: 'Open Sans', Arial, sans-serif;
            background: #f5f5f5;
        }
        
        .download-bar {
            background: #1e3c72;
            padding: 15px;
            text-align: center;
        }
        
        .download-btn {
            background: #e94560;
            color: white;
            border: none;
            padding: 12px 30px;
            border-radius: 25px;
            font-size: 16px;
            cursor: pointer;
        }
        
        .download-btn:hover { background: #d63050; }
        
        .page-wrapper {
            display: flex;
            justify-content: center;
            padding: 20px;
        }
        
        #voucher {
            width: 210mm;
            background: white;
            padding: 12mm 15mm;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
        }
        
        .header {
            text-align: center;
            padding-bottom: 10px;
            border-bottom: 2px solid #1e3c72;
            margin-bottom: 12px;
        }
        
        .logo { width: 50px; height: 50px; margin-bottom: 5px; }
        
        .company { 
            font-size: 22px; 
            font-weight: bold; 
            color: #1e3c72;
        }
        
        .subtitle { font-size: 12px; color: #666; }
        
        .destination-box {
            background: linear-gradient(135deg, #e94560, #ff6b6b);
            color: white;
            text-align: center;
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 12px;
        }
        
        .destination-box h1 { 
            font-size: 24px; 
            margin-bottom: 5px;
        }
        .destination-box p { font-size: 13px; }
        
        .section {
            border: 1px solid #ddd;
            border-radius: 6px;
            margin-bottom: 10px;
            overflow: hidden;
        }
        
        .section-title {
            background: #f0f0f0;
            padding: 8px 12px;
            font-weight: bold;
            font-size: 13px;
            color: #1e3c72;
            border-bottom: 1px solid #ddd;
        }
        
        .section-content { padding: 10px 12px; }
        
        .row {
            display: flex;
            padding: 5px 0;
            border-bottom: 1px solid #f0f0f0;
            font-size: 12px;
        }
        
        .row:last-child { border-bottom: none; }
        .label { width: 130px; color: #666; }
        .value { flex: 1; font-weight: 600; color: #333; }
        
        .passenger-list { list-style: none; font-size: 12px; }
        .passenger-list li { padding: 3px 0; }
        
        .footer {
            text-align: center;
            margin-top: 12px;
            padding-top: 12px;
            border-top: 1px solid #ddd;
        }
        
        .booking-num {
            background: #1e3c72;
            color: white;
            display: inline-block;
            padding: 8px 20px;
            border-radius: 20px;
            font-size: 13px;
            margin-bottom: 8px;
        }
        
        .contact { font-size: 11px; color: #666; }
    </style>
</head>
<body>
    <div class="download-bar">
        <button class="download-btn" onclick="downloadPDF()">📄 Download PDF</button>
    </div>
    
    <div class="page-wrapper">
        <div id="voucher">
            <div class="header">
                <img src="assets/logo.png" class="logo" alt="Logo">
                <div class="company">Holiday in Sale</div>
                <div class="subtitle">Booking Confirmation</div>
            </div>
            
            <div class="destination-box">
                <h1>✈️ <?= $destination ?></h1>
                <p><?= formatEnglishDate($order['departure_date'] ?? null) ?> - <?= formatEnglishDate($order['return_date'] ?? null) ?></p>
            </div>
            
            <div class="section">
                <div class="section-title">👥 Passengers</div>
                <div class="section-content">
                    <ul class="passenger-list">
                        <?php foreach ($passengers as $i => $p): ?>
                            <li><?= ($i + 1) ?>. <?= cleanText($p) ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            </div>
            
            <div class="section">
                <div class="section-title">✈️ Flight Details</div>
                <div class="section-content">
                    <div class="row">
                        <span class="label">Airline:</span>
                        <span class="value"><?= cleanText($order['airline_outbound'] ?? '-') ?></span>
                    </div>
                    <div class="row">
                        <span class="label">Booking Code:</span>
                        <span class="value"><?= cleanText($order['booking_code_outbound'] ?? '-') ?></span>
                    </div>
                    <?php if (!empty($order['airline_return']) && $order['airline_return'] !== $order['airline_outbound']): ?>
                    <div class="row">
                        <span class="label">Return Airline:</span>
                        <span class="value"><?= cleanText($order['airline_return'] ?? '-') ?></span>
                    </div>
                    <div class="row">
                        <span class="label">Return Code:</span>
                        <span class="value"><?= cleanText($order['booking_code_return'] ?? '-') ?></span>
                    </div>
                    <?php endif; ?>
                    <div class="row">
                        <span class="label">Departure:</span>
                        <span class="value"><?= formatEnglishDate($order['departure_date'] ?? null) ?><?php if ($timeOut): ?> at <?= $timeOut ?><?php endif; ?></span>
                    </div>
                    <div class="row">
                        <span class="label">Return:</span>
                        <span class="value"><?= formatEnglishDate($order['return_date'] ?? null) ?><?php if ($timeReturn): ?> at <?= $timeReturn ?><?php endif; ?></span>
                    </div>
                    <?php if (!empty($order['luggage_details'])): ?>
                    <div class="row">
                        <span class="label">Luggage:</span>
                        <span class="value"><?= $luggage ?></span>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <?php if (!empty($order['hotel_name'])): ?>
            <div class="section">
                <div class="section-title">🏨 Hotel Details</div>
                <div class="section-content">
                    <div class="row">
                        <span class="label">Hotel:</span>
                        <span class="value"><?= cleanText($order['hotel_name'] ?? '-') ?></span>
                    </div>
                    <div class="row">
                        <span class="label">Confirmation #:</span>
                        <span class="value"><?= cleanText($order['hotel_confirmation'] ?? '-') ?></span>
                    </div>
                    <div class="row">
                        <span class="label">Check-in:</span>
                        <span class="value"><?= formatEnglishDate($order['hotel_checkin'] ?? null) ?></span>
                    </div>
                    <div class="row">
                        <span class="label">Check-out:</span>
                        <span class="value"><?= formatEnglishDate($order['hotel_checkout'] ?? null) ?></span>
                    </div>
                    <?php if (!empty($order['room_type'])): ?>
                    <div class="row">
                        <span class="label">Room Type:</span>
                        <span class="value"><?= cleanText($order['room_type'] ?? '-') ?></span>
                    </div>
                    <?php endif; ?>
                    <div class="row">
                        <span class="label">Meal Plan:</span>
                        <span class="value"><?php 
                            $meal = $order['meal_basis'] ?? '';
                            echo $mealBasisEnglish[$meal] ?? ($meal ?: 'Bed & Breakfast');
                        ?></span>
                    </div>
                </div>
            </div>
            <?php endif; ?>
            
            <div class="footer">
                <div class="booking-num">Booking #<?= $order['id'] ?></div>
                <div class="contact">
                    📞 Contact: 050-6101031<br>
                    Holiday in Sale - European Travel Experts
                </div>
            </div>
        </div>
    </div>
    
    <script>
        function downloadPDF() {
            const element = document.getElementById('voucher');
            const opt = {
                margin: [5, 5, 5, 5],
                filename: 'voucher-<?= $order['id'] ?>.pdf',
                image: { type: 'jpeg', quality: 0.98 },
                html2canvas: { 
                    scale: 2,
                    useCORS: true,
                    logging: false
                },
                jsPDF: { 
                    unit: 'mm', 
                    format: 'a4', 
                    orientation: 'portrait'
                }
            };
            html2pdf().set(opt).from(element).save();
        }
    </script>
</body>
</html>
